%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  This file is part of Logtalk <https://logtalk.org/>
%  SPDX-FileCopyrightText: 1998-2026 Paulo Moura <pmoura@logtalk.org>
%  SPDX-License-Identifier: Apache-2.0
%
%  Licensed under the Apache License, Version 2.0 (the "License");
%  you may not use this file except in compliance with the License.
%  You may obtain a copy of the License at
%
%      http://www.apache.org/licenses/LICENSE-2.0
%
%  Unless required by applicable law or agreed to in writing, software
%  distributed under the License is distributed on an "AS IS" BASIS,
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%  See the License for the specific language governing permissions and
%  limitations under the License.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


:- object(tests,
	extends(lgtunit)).

	:- info([
		version is 1:1:0,
		author is 'Paulo Moura and Daniel L. Dudley',
		date is 2026-02-25,
		comment is 'Unit tests for the iso8601 library.'
	]).

	:- uses(iso8601, [
		date/4, date/5, date/6, date/7,
		date_string/3,
		duration_string/2,
		interval_string/2,
		valid_date/3,
		leap_year/1, calendar_month/3, easter_day/3
	]).

	% date/4 tests

	- test(iso8601_date_4_01, deterministic(t(JD,Year,Month,Day) == t(2453471,2005,4,10))) :-
		% Current date (i.e., today)
		date(JD, Year, Month, Day).

	test(iso8601_date_4_02, deterministic(JD == 2451604)) :-
		% Convert a date to its Julian day number
		date(JD, 2000, 2, 29).

	test(iso8601_date_4_03, deterministic(d(Year,Month,Day) == d(2000,2,29))) :-
		% Convert a Julian day number to its date
		date(2451604, Year, Month, Day).

	test(iso8601_date_4_04, deterministic(JD == 2451604)) :-
		% What is the date of day # 60 in year 2000?
		date(JD, 2000, 1, 60).

	test(iso8601_date_4_05, deterministic(JD == 2451544)) :-
		% What is the Julian of the 1st day prior to 2000-1-1?
		date(JD, 2000, 1, 0).

	test(iso8601_date_4_06, deterministic(JD == 2451485)) :-
		% What is the Julian of the 60th day prior to 2000-1-1?
		date(JD, 2000, 1, -59).

	test(iso8601_date_4_07, deterministic(JD == 2415080)) :-
		% Illegal date is auto-adjusted (see also next query)
		date(JD, 1900, 2, 29).

	test(iso8601_date_4_08, deterministic(d(Year,Month,Day) == d(1900,3,1))) :-
		% This is the correct date!
		date(2415080, Year, Month, Day).

	% date/5 tests

	test(iso8601_date_5_01, deterministic(JD-DoW == 2451604-2)) :-
		% Get the Julian and the day-of-week # of a date
		date(JD, 2000, 2, 29, DoW).

	test(iso8601_date_5_02, false) :-
		% Check the validity of a given date (day-of-week is 2, not 4)
		date(_, 2002, 3, 5, 4).

	test(iso8601_date_5_03, deterministic(JD == 2453065)) :-
		% Get the Julian day of a given date if it is a Sunday
		date(JD, 2004, 2, 29, 7).

	test(iso8601_date_5_04, deterministic(d(Year,Month,Day,DoW) == d(2000,1,1,6))) :-
		% Get the date and day-of-week # of a Julian
		date(2451545, Year, Month, Day, DoW).

	% date/6 tests

	test(iso8601_date_6_01, deterministic(DoW-Week == 6-week(52,1999))) :-
		% Get the day-of-week and week number of a date
		date(_, 2000, 1, 1, DoW, Week).

	- test(iso8601_date_6_02, deterministic(Week == week(7, 2004))) :-
		% Get the week number and year of this week
		date(_, _, _, _, _, Week).

	test(iso8601_date_6_03, deterministic(JD-Week == 2453065-week(9,2004))) :-
		% Get the Julian number and the week of a date if it is a Sunday
		date(JD, 2004, 2, 29, 7, Week).

	test(iso8601_date_6_04, deterministic(DoW-Week == 1-week(10,2004))) :-
		% Get the day-of-week and week of a Julian day number
		date(2453066, _, _, _, DoW, Week).

	test(iso8601_date_6_05, deterministic) :-
		% Check that given date data matches
		date(_, 2004, 3, 1, 1, week(10,2004)).

	test(iso8601_date_6_06, deterministic(d(Year,Month,Day,DoW) == d(2004,6,21,1))) :-
		% What is the date of a day of week (default is 1) in given week # and year?
		date(_, Year, Month, Day, DoW, week(26,2004)).

	test(iso8601_date_6_07, deterministic(d(Year,Month,Day) == d(2005,1,9))) :-
		% Ditto for Sunday
		date(_, Year, Month, Day, 7, week(1,2005)).

	test(iso8601_date_6_08, deterministic(d(Year,Month,Day) == d(2005,1,11))) :-
		% Ditto for Tuesday in following week
		date(_, Year, Month, Day, 9, week(1,2005)).

	test(iso8601_date_6_09, deterministic(d(Year,Month,Day) == d(2004,12,30))) :-
		% Ditto for Thursday in the prior week
		date(_, Year, Month, Day, 4, week(0,2005)).

	test(iso8601_date_6_10, deterministic(d(Year,Month,Day) == d(2004,12,21))) :-
		% Ditto for Tuesday two weeks prior
		date(_, Year, Month, Day, 2, week(-1,2005)).

	test(iso8601_date_6_11, deterministic(d(Year,Month,Day) == d(2005,1,1))) :-
		% Ditto for Saturday
		date(_, Year, Month, Day, 6, week(53,2004)).

	test(iso8601_date_6_12, deterministic(d(Year,Month,Day) == d(2005,2,14))) :-
		% Ditto for Monday (note automatic compensation of nonexistent week number)
		date(_, Year, Month, Day, 1, week(60,2004)).

	% date/7 tests

	test(iso8601_date_7_01, deterministic(d(Year,Month,Day,DoY) == d(2000,4,14,105))) :-
		% Get the date and day-of-year of a Julian number
		date(2451649, Year, Month, Day, _, _, DoY).

	test(iso8601_date_7_02, deterministic(d(JD,Week,DoY) == d(2453065,week(9,2004),60))) :-
		% Get the Julian number, week number and day-of-year of a date, confirming that it is a Sunday
		date(JD, 2004, 2, 29, 7, Week, DoY).

	test(iso8601_date_7_03, deterministic) :-
		% Confirm that a date is, in fact, a specific day-of-year
		date(_, 2004, 3, 1, _, _, 61).

	test(iso8601_date_7_04, deterministic(d(JD,DoW,DoY) == d(2453297,1,292))) :-
		% Get the Julian number, week day and day-of-year of a date
		date(JD, 2004, 10, 18, DoW, _, DoY).

	- test(iso8601_date_7_05, deterministic(DoY == 54)) :-
		% Get today''s day-of-year
		date(_, _, _, _, _, _, DoY).

	test(iso8601_date_7_06, deterministic(d(Month,Day,DoW,Week) == d(2,29,7,week(9,2004)))) :-
		% Get all missing date data (excl. Julian number) for the 60th calendar day of 2004
		date(_, 2004, Month, Day, DoW, Week, 60).

	test(iso8601_date_7_07, deterministic(d(Day,DoW,Week) == d(1,1,week(10,2004)))) :-
		% Match given date data and, if true, return the missing data (excl. Julian number)
		date(_, 2004, 3, Day, DoW, Week, 61).

	test(iso8601_date_7_08, false) :-
		% Ditto (the 61st day-of-year cannot be both day 1 and 2 of the month)
		date(_, 2004, _Month, 2, _DoW, _Week, 61).

	% date_string/3 tests

	test(iso8601_date_string_3_01, deterministic(String == '20040229')) :-
		% Date, complete, basic (section 5.2.1.1)
		date_string('YYYYMMDD', [2004,2,29], String).

	test(iso8601_date_string_3_02, deterministic(Day == [2004,2,29])) :-
		% Date, complete, basic (section 5.2.1.1)
		date_string('YYYYMMDD', Day, '20040229').

	test(iso8601_date_string_3_03, deterministic(String == '2003-12-16')) :-
		% Date, complete, extended (section 5.2.1.1)
		date_string('YYYY-MM-DD', [2003,12,16], String).

	test(iso8601_date_string_3_04, deterministic(Day == [2003,12,16])) :-
		% Date, complete, extended (section 5.2.1.1)
		date_string('YYYY-MM-DD', Day, '2003-12-16').

	- test(iso8601_date_string_3_05, deterministic(String == '2004-02-17')) :-
		% Date, complete, extended (section 5.2.1.1)
		date_string('YYYY-MM-DD', _, String).

	test(iso8601_date_string_3_06, deterministic(Day == [2004,2,17])) :-
		% Date, complete, extended (section 5.2.1.1)
		date_string('YYYY-MM-DD', Day, '2004-02-17').

	test(iso8601_date_string_3_07, deterministic(String == '2004-09')) :-
		% Date, reduced, month (section 5.2.1.2 a)
		date_string('YYYY-MM', [2004,9,18], String).

	test(iso8601_date_string_3_08, deterministic(Day == [2004,9])) :-
		% Date, reduced, month (section 5.2.1.2 a)
		date_string('YYYY-MM',Day,'2004-09').

	test(iso8601_date_string_3_09, deterministic(String == '1900')) :-
		% Date, reduced, year (section 5.2.1.2 b)
		date_string('YYYY', [1900,7,24], String).

	test(iso8601_date_string_3_10, deterministic(Day == [1900])) :-
		% Date, reduced, year (section 5.2.1.2 b)
		date_string('YYYY', Day, '1900').

	test(iso8601_date_string_3_11, deterministic(String == '20')) :-
		% Date, reduced, century (section 5.2.1.2 c)
		date_string('YY', 2456557, String).

	test(iso8601_date_string_3_12, deterministic(Day == [20])) :-
		% Date, reduced, century (section 5.2.1.2 c)
		date_string('YY', Day, '20').

	test(iso8601_date_string_3_13, deterministic(String == '2005084')) :-
		% Date, ordinal, complete (section 5.2.2.1)
		date_string('YYYYDDD', [2005,3,25], String).

	test(iso8601_date_string_3_14, deterministic(Day == [2005,84])) :-
		% Date, ordinal, complete (section 5.2.2.1)
		date_string('YYYYDDD', Day, '2005084').

	test(iso8601_date_string_3_15, deterministic(String == '1854-338')) :-
		% Date, ordinal, extended (section 5.2.2.1)
		date_string('YYYY-DDD', [1854,12,4], String).

	test(iso8601_date_string_3_16, deterministic(Day == [1854,338])) :-
		% Date, ordinal, extended (section 5.2.2.1)
		date_string('YYYY-DDD', Day, '1854-338').

	test(iso8601_date_string_3_17, deterministic(String == '1999W527')) :-
		% Week, complete, basic (section 5.2.3.1)
		date_string('YYYYWwwD', [2000,1,2], String).

	test(iso8601_date_string_3_18, deterministic(Day == [1999,52,7])) :-
		% Week, complete, basic (section 5.2.3.1)
		date_string('YYYYWwwD', Day, '1999W527').

	test(iso8601_date_string_3_19, deterministic(Str == '2004-W01-1')) :-
		% Week, complete, extended (section 5.2.3.1)
		date_string('YYYY-Www-D', [2003,12,29], Str).

	test(iso8601_date_string_3_20, deterministic(Day == [2004,1,1])) :-
		% Week, complete, extended (section 5.2.3.1)
		date_string('YYYY-Www-D', Day, '2004-W01-1').

	test(iso8601_date_string_3_21, deterministic(String == '2004-W24-4')) :-
		% Week, complete, extended (section 5.2.3.1)
		date_string('YYYY-Www-D', 2453167, String).

	test(iso8601_date_string_3_22, deterministic(Day == [2004,24,4])) :-
		% Week, complete, extended (section 5.2.3.1)
		date_string('YYYY-Www-D', Day, '2004-W24-4').

	test(iso8601_date_string_3_23, deterministic(String == '2004W09')) :-
		% Week, reduced, basic (section 5.2.3.2)
		date_string('YYYYWww', [2004,2,29], String).

	test(iso8601_date_string_3_24, deterministic(Day == [2004,9])) :-
		% Week, reduced, basic (section 5.2.3.2)
		date_string('YYYYWww', Day, '2004W09').

	test(iso8601_date_string_3_25, deterministic(String == '2004-W09')) :-
		% Week, reduced, extended (section 5.2.3.2)
		date_string('YYYY-Www', [2004,2,29], String).

	test(iso8601_date_string_3_26, deterministic(Day == [2004,9])) :-
		% Week, reduced, extended (section 5.2.3.2)
		date_string('YYYY-Www', Day, '2004-W09').

	% duration_string/2 tests

	test(iso8601_duration_string_2_01, deterministic(String == 'P1Y2M3DT4H5M6S')) :-
		duration_string(duration(1, 2, 3, 4, 5, 6), String).

	test(iso8601_duration_string_2_02, deterministic(Duration == duration(0, 0, 0, 0, 45, 0))) :-
		duration_string(Duration, 'PT45M').

	test(iso8601_duration_string_2_03, deterministic(String == 'PT0S')) :-
		duration_string(duration(0, 0, 0, 0, 0, 0), String).

	% interval_string/2 tests

	test(iso8601_interval_string_2_01, deterministic(String == '2026-02-25/2026-03-01')) :-
		interval_string(interval([2026,2,25], [2026,3,1]), String).

	test(iso8601_interval_string_2_02, deterministic(String == '2026-02-25/P3D')) :-
		interval_string(interval([2026,2,25], duration(0, 0, 3, 0, 0, 0)), String).

	test(iso8601_interval_string_2_03, deterministic(String == 'P3D/2026-02-25')) :-
		interval_string(interval(duration(0, 0, 3, 0, 0, 0), [2026,2,25]), String).

	test(iso8601_interval_string_2_04, deterministic(Interval == interval([2026,2,25], duration(0, 0, 3, 0, 0, 0)))) :-
		interval_string(Interval, '2026-02-25/P3D').

	% valid_date/3 tests

	test(iso8601_valid_date_3_01, deterministic) :-
		% Yes, the recent millennium was a leap year
		valid_date(2000, 2, 29).

	test(iso8601_valid_date_3_02, deterministic) :-
		% 2004 was also a leap year
		valid_date(2004, 2, 29).

	test(iso8601_valid_date_3_03, false) :-
		% Only 30 days in April
		valid_date(2004, 4, 31).

	test(iso8601_valid_date_3_04, deterministic) :-
		% 1 BC was a leap year
		valid_date(-1, 2, 29).

	% leap_year/1

	test(iso8601_leap_year_1_01, false) :-
		% No, the prior centenary was not a leap year
		leap_year(1900).

	test(iso8601_leap_year_1_02, deterministic) :-
		% The recent millennium
		leap_year(2000).

	- test(iso8601_leap_year_1_03, deterministic(Year == 2004)) :-
		% This year
		leap_year(Year).

	- test(iso8601_leap_year_1_04, deterministic) :-
		% This year (equivalent to prior query)
		leap_year(_).

	test(iso8601_leap_year_1_05, false) :-
		% Next centennial
		leap_year(2100).

	test(iso8601_leap_year_1_06, deterministic) :-
		% Year 0, equivalent to 1 BC
		leap_year(0).

	test(iso8601_leap_year_1_07, deterministic) :-
		% 1 BC
		leap_year(-1).

	test(iso8601_leap_year_1_08, false) :-
		% 4 BC
		leap_year(-4).

	test(iso8601_leap_year_1_09, deterministic) :-
		% 5 BC
		leap_year(-5).

	% calendar_month/3 tests

	test(iso8601_calendar_month_3_01) :-
		% Compute the calendar of March, 2005
		calendar_month(2005, 3, Calendar),
		^^assertion(Calendar == m(2005, 3, [
			w( 9, [ 0,  1,  2,  3,  4,  5,  6]),
			w(10, [ 7,  8,  9, 10, 11, 12, 13]),
			w(11, [14, 15, 16, 17, 18, 19, 20]),
			w(12, [21, 22, 23, 24, 25, 26, 27]),
			w(13, [28, 29, 30, 31,  0,  0,  0]),
			w( 0, [ 0,  0,  0,  0,  0,  0,  0])
		])).

	% easter_day/3 tests

	test(iso8601_easter_day_3_01, deterministic(Month-Day == 4-16)) :-
		% Compute Easter Sunday for a particular year
		easter_day(2006, Month, Day).

	- test(iso8601_easter_day_3_02, deterministic(d(Year,Month,Day) == d(2005,3,27))) :-
		% Compute Easter Sunday for the current year
		easter_day(Year, Month, Day).

:- end_object.
