/*
 * Decompiled with CFR 0.152.
 */
package brut.androlib;

import brut.androlib.Config;
import brut.androlib.exceptions.AndrolibException;
import brut.androlib.exceptions.InFileNotFoundException;
import brut.androlib.exceptions.OutDirExistsException;
import brut.androlib.meta.ApkInfo;
import brut.androlib.res.ResDecoder;
import brut.androlib.smali.SmaliDecoder;
import brut.common.Log;
import brut.directory.Directory;
import brut.directory.DirectoryException;
import brut.directory.ExtFile;
import brut.util.BackgroundWorker;
import brut.util.OS;
import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Pattern;
import org.apache.commons.io.FilenameUtils;

public class ApkDecoder {
    private static final String TAG = ApkDecoder.class.getName();
    private static final Pattern NO_COMPRESS_EXT_PATTERN = Pattern.compile("dex|arsc|so|jpg|jpeg|png|gif|wav|mp2|mp3|ogg|aac|mpg|mpeg|mid|midi|smf|jet|rtttl|imy|xmf|mp4|m4a|m4v|3gp|3gpp|3g2|3gpp2|amr|awb|wma|wmv|webm|webp|mkv");
    private final ExtFile mApkFile;
    private final Config mConfig;
    private final AtomicReference<AndrolibException> mFirstError;
    private ApkInfo mApkInfo;
    private SmaliDecoder mSmaliDecoder;
    private ResDecoder mResDecoder;
    private BackgroundWorker mWorker;

    public ApkDecoder(File apkFile, Config config) {
        this.mApkFile = new ExtFile(apkFile);
        this.mConfig = config;
        this.mFirstError = new AtomicReference();
    }

    public void decode(File outDir) throws AndrolibException {
        if (!this.mConfig.isForced() && outDir.exists()) {
            throw new OutDirExistsException(outDir.getPath());
        }
        if (!this.mApkFile.isFile() || !this.mApkFile.canRead()) {
            throw new InFileNotFoundException(this.mApkFile.getPath());
        }
        if (this.mConfig.getJobs() > 1) {
            this.mWorker = new BackgroundWorker(this.mConfig.getJobs() - 1);
        }
        try {
            this.mApkInfo = new ApkInfo();
            this.mApkInfo.setVersion(this.mConfig.getVersion());
            this.mApkInfo.setApkFile(this.mApkFile);
            this.mSmaliDecoder = new SmaliDecoder(this.mApkFile, this.mConfig.isBaksmaliDebugMode());
            this.mResDecoder = new ResDecoder(this.mApkInfo, this.mConfig);
            OS.rmdir(outDir);
            OS.mkdir(outDir);
            Log.i(TAG, "Using Apktool " + this.mConfig.getVersion() + " on " + this.mApkFile.getName() + (this.mWorker != null ? " with " + this.mConfig.getJobs() + " threads" : ""));
            this.decodeSources(outDir);
            this.decodeResources(outDir);
            this.decodeManifest(outDir);
            if (this.mWorker != null) {
                this.mWorker.waitForFinish();
                if (this.mFirstError.get() != null) {
                    throw this.mFirstError.get();
                }
            }
            this.copyOriginalFiles(outDir);
            this.copyRawFiles(outDir);
            this.copyUnknownFiles(outDir);
            this.writeApkInfo(outDir);
        }
        finally {
            if (this.mWorker != null) {
                this.mWorker.shutdownNow();
            }
            try {
                this.mApkFile.close();
            }
            catch (DirectoryException directoryException) {}
        }
    }

    @VisibleForTesting
    ApkInfo getApkInfo() {
        return this.mApkInfo;
    }

    private void decodeSources(File outDir) throws AndrolibException {
        if (!this.mApkInfo.hasSources()) {
            return;
        }
        try {
            Directory in = this.mApkFile.getDirectory();
            boolean allSrc = this.mConfig.isDecodeSourcesFull();
            boolean noSrc = this.mConfig.isDecodeSourcesNone();
            for (String fileName : in.getFiles(allSrc)) {
                if (!allSrc ? !ApkInfo.CLASSES_FILES_PATTERN.matcher(fileName).matches() : !fileName.endsWith(".dex")) continue;
                if (noSrc) {
                    this.copySourcesRaw(outDir, fileName);
                    continue;
                }
                this.decodeSourcesSmali(outDir, fileName);
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void copySourcesRaw(File outDir, String fileName) throws AndrolibException {
        Log.i(TAG, "Copying raw " + fileName + "...");
        try {
            Directory in = this.mApkFile.getDirectory();
            in.copyToDir(outDir, fileName);
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void decodeSourcesSmali(File outDir, String fileName) throws AndrolibException {
        if (this.mWorker != null) {
            this.mWorker.submit(() -> {
                if (this.mFirstError.get() == null) {
                    try {
                        this.decodeSourcesSmaliJob(outDir, fileName);
                    }
                    catch (AndrolibException ex) {
                        this.mFirstError.compareAndSet(null, ex);
                    }
                }
            });
        } else {
            this.decodeSourcesSmaliJob(outDir, fileName);
        }
    }

    private void decodeSourcesSmaliJob(File outDir, String fileName) throws AndrolibException {
        Log.i(TAG, "Baksmaling " + fileName + "...");
        this.mSmaliDecoder.decode(fileName, outDir);
    }

    private void decodeResources(File outDir) throws AndrolibException {
        if (!this.mApkInfo.hasResources()) {
            return;
        }
        if (this.mConfig.isDecodeResourcesFull()) {
            this.mResDecoder.decodeResources(outDir);
        } else {
            this.copyResourcesRaw(outDir);
        }
    }

    private void copyResourcesRaw(File outDir) throws AndrolibException {
        Log.i(TAG, "Copying raw resources.arsc...");
        try {
            Directory in = this.mApkFile.getDirectory();
            in.copyToDir(outDir, "resources.arsc");
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void decodeManifest(File outDir) throws AndrolibException {
        if (!this.mApkInfo.hasManifest()) {
            return;
        }
        if (!this.mConfig.isDecodeResourcesNone()) {
            this.mResDecoder.decodeManifest(outDir);
        } else {
            this.copyManifestRaw(outDir);
        }
    }

    private void copyManifestRaw(File outDir) throws AndrolibException {
        Log.i(TAG, "Copying raw AndroidManifest.xml...");
        try {
            Directory in = this.mApkFile.getDirectory();
            in.copyToDir(outDir, "AndroidManifest.xml");
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void copyRawFiles(File outDir) throws AndrolibException {
        try {
            Directory in = this.mApkFile.getDirectory();
            Set<String> dexFiles = this.mSmaliDecoder.getDexFiles();
            Map<String, String> resFileMap = this.mResDecoder.getResFileMap();
            boolean noAssets = this.mConfig.isDecodeAssetsNone();
            for (String dirName : ApkInfo.RAW_DIRS) {
                if (!in.containsDir(dirName) || noAssets && dirName.equals("assets")) continue;
                Log.i(TAG, "Copying " + dirName + "...");
                for (String fileName : in.getDir(dirName).getFiles(true)) {
                    fileName = dirName + in.separator + fileName;
                    if (ApkInfo.ORIGINAL_FILES_PATTERN.matcher(fileName).matches() || dexFiles.contains(fileName) || resFileMap.containsKey(fileName)) continue;
                    in.copyToDir(outDir, fileName);
                }
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void copyOriginalFiles(File outDir) throws AndrolibException {
        File originalDir = new File(outDir, "original");
        Log.i(TAG, "Copying original files...");
        try {
            Directory in = this.mApkFile.getDirectory();
            for (String fileName : in.getFiles(true)) {
                if (!ApkInfo.ORIGINAL_FILES_PATTERN.matcher(fileName).matches()) continue;
                in.copyToDir(originalDir, fileName);
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void copyUnknownFiles(File outDir) throws AndrolibException {
        File unknownDir = new File(outDir, "unknown");
        Log.i(TAG, "Copying unknown files...");
        try {
            Directory in = this.mApkFile.getDirectory();
            Set<String> dexFiles = this.mSmaliDecoder.getDexFiles();
            Map<String, String> resFileMap = this.mResDecoder.getResFileMap();
            for (String fileName : in.getFiles(true)) {
                if (ApkInfo.STANDARD_FILES_PATTERN.matcher(fileName).matches() || dexFiles.contains(fileName) || resFileMap.containsKey(fileName)) continue;
                in.copyToDir(unknownDir, fileName);
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
    }

    private void writeApkInfo(File outDir) throws AndrolibException {
        int apiLevel;
        if ((!this.mApkInfo.hasManifest() || this.mConfig.isDecodeResourcesNone()) && (apiLevel = this.mSmaliDecoder.getInferredApiLevel()) > 0) {
            this.mApkInfo.getSdkInfo().setMinSdkVersion(Integer.toString(apiLevel));
        }
        try {
            String ext;
            Directory in = this.mApkFile.getDirectory();
            Map<String, String> resFileMap = this.mResDecoder.getResFileMap();
            HashSet<String> uncompressedExts = new HashSet<String>();
            HashSet<String> uncompressedFiles = new HashSet<String>();
            for (String fileName : in.getFiles(true)) {
                if (in.getCompressionLevel(fileName) != 0) continue;
                if (in.getSize(fileName) > 0L && !(ext = FilenameUtils.getExtension(fileName)).isEmpty() && NO_COMPRESS_EXT_PATTERN.matcher(ext).matches()) {
                    uncompressedExts.add(ext);
                    continue;
                }
                uncompressedFiles.add(resFileMap.getOrDefault(fileName, fileName));
            }
            if (!uncompressedExts.isEmpty() && !uncompressedFiles.isEmpty()) {
                Iterator it = uncompressedFiles.iterator();
                while (it.hasNext()) {
                    String fileName;
                    fileName = (String)it.next();
                    ext = FilenameUtils.getExtension(fileName);
                    if (!uncompressedExts.contains(ext)) continue;
                    it.remove();
                }
            }
            List<String> doNotCompress = this.mApkInfo.getDoNotCompress();
            if (!uncompressedExts.isEmpty()) {
                ArrayList uncompressedExtsList = new ArrayList(uncompressedExts);
                uncompressedExtsList.sort(null);
                doNotCompress.addAll(uncompressedExtsList);
            }
            if (!uncompressedFiles.isEmpty()) {
                ArrayList uncompressedFilesList = new ArrayList(uncompressedFiles);
                uncompressedFilesList.sort(null);
                doNotCompress.addAll(uncompressedFilesList);
            }
        }
        catch (DirectoryException ex) {
            throw new AndrolibException(ex);
        }
        this.mApkInfo.save(outDir);
    }
}

