#pragma once

/**
 * \file
 * Contains declarations of all functions and types which represent a public TDLib interface.
 */
#include "td/tl/TlObject.h"

#include <string>

#include <cstdint>
#include <utility>
#include <vector>

namespace td {
class TlStorerToString;

namespace td_api {

/**
 * This type is used to store 32-bit signed integers, which can be represented as Number in JSON.
 */
using int32 = std::int32_t;
/**
 * This type is used to store 53-bit signed integers, which can be represented as Number in JSON.
 */
using int53 = std::int64_t;
/**
 * This type is used to store 64-bit signed integers, which can't be represented as Number in JSON and are represented as String instead.
 */
using int64 = std::int64_t;

/**
 * This type is used to store UTF-8 strings.
 */
using string = std::string;

/**
 * This type is used to store arbitrary sequences of bytes. In JSON interface the bytes are base64-encoded.
 */
using bytes = std::string;

/**
 * This type is used to store a list of objects of any type and is represented as Array in JSON.
 */
template <class Type>
using array = std::vector<Type>;

/**
 * This class is a base class for all TDLib API classes and functions.
 */
using BaseObject = ::td::TlObject;

/**
 * A smart wrapper to store a pointer to a TDLib API object. Can be treated as an analogue of std::unique_ptr.
 */
template <class Type>
using object_ptr = ::td::tl_object_ptr<Type>;

/**
 * A function to create a dynamically allocated TDLib API object. Can be treated as an analogue of std::make_unique.
 * Usage example:
 * \code
 * auto get_me_request = td::td_api::make_object<td::td_api::getMe>();
 * auto message_text = td::td_api::make_object<td::td_api::formattedText>("Hello, world!!!",
 *                     td::td_api::array<td::td_api::object_ptr<td::td_api::textEntity>>());
 * auto send_message_request = td::td_api::make_object<td::td_api::sendMessage>(chat_id, 0, nullptr, nullptr, nullptr,
 *      td::td_api::make_object<td::td_api::inputMessageText>(std::move(message_text), nullptr, true));
 * \endcode
 *
 * \tparam Type Type of object to construct.
 * \param[in] args Arguments to pass to the object constructor.
 * \return Wrapped pointer to the created object.
 */
template <class Type, class... Args>
object_ptr<Type> make_object(Args &&... args) {
  return object_ptr<Type>(new Type(std::forward<Args>(args)...));
}

/**
 * A function to cast a wrapped in td::td_api::object_ptr TDLib API object to its subclass or superclass.
 * Casting an object to an incorrect type will lead to undefined behaviour.
 * Usage example:
 * \code
 * td::td_api::object_ptr<td::td_api::callState> call_state = ...;
 * switch (call_state->get_id()) {
 *   case td::td_api::callStatePending::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStatePending>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateExchangingKeys::ID: {
 *     // no additional fields, no casting is needed
 *     break;
 *   }
 *   case td::td_api::callStateReady::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateReady>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateHangingUp::ID: {
 *     // no additional fields, no casting is needed
 *     break;
 *   }
 *   case td::td_api::callStateDiscarded::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateDiscarded>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateError::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateError>(call_state);
 *     // use state
 *     break;
 *   }
 *   default:
 *     assert(false);
 * }
 * \endcode
 *
 * \tparam ToType Type of TDLib API object to move to.
 * \tparam FromType Type of TDLib API object to move from, this is auto-deduced.
 * \param[in] from Wrapped in td::td_api::object_ptr pointer to a TDLib API object.
 */
template <class ToType, class FromType>
object_ptr<ToType> move_object_as(FromType &&from) {
  return object_ptr<ToType>(static_cast<ToType *>(from.release()));
}

/**
 * Returns a string representation of a TDLib API object.
 * \param[in] value The object.
 * \return Object string representation.
 */
std::string to_string(const BaseObject &value);

/**
 * Returns a string representation of a TDLib API object.
 * \tparam T Object type, auto-deduced.
 * \param[in] value The object.
 * \return Object string representation.
 */
template <class T>
std::string to_string(const object_ptr<T> &value) {
  if (value == nullptr) {
    return "null";
  }

  return to_string(*value);
}

/**
 * Returns a string representation of a list of TDLib API objects.
 * \tparam T Object type, auto-deduced.
 * \param[in] values The objects.
 * \return Objects string representation.
 */
template <class T>
std::string to_string(const std::vector<object_ptr<T>> &values) {
  std::string result = "{\n";
  for (const auto &value : values) {
    if (value == nullptr) {
      result += "null\n";
    } else {
      result += to_string(*value);
    }
  }
  result += "}\n";
  return result;
}

/**
 * This class is a base class for all TDLib API classes.
 */
class Object: public TlObject {
 public:
};

/**
 * This class is a base class for all TDLib API functions.
 */
class Function: public TlObject {
 public:
};

/**
 * Contains information about supported accent color for user/chat name, background of empty chat photo, replies to messages and link previews.
 */
class accentColor final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Accent color identifier.
  int32 id_;
  /// Identifier of a built-in color to use in places, where only one color is needed; 0-6.
  int32 built_in_accent_color_id_;
  /// The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in light themes.
  array<int32> light_theme_colors_;
  /// The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes.
  array<int32> dark_theme_colors_;
  /// The minimum chat boost level required to use the color in a channel chat.
  int32 min_channel_chat_boost_level_;

  /**
   * Contains information about supported accent color for user/chat name, background of empty chat photo, replies to messages and link previews.
   */
  accentColor();

  /**
   * Contains information about supported accent color for user/chat name, background of empty chat photo, replies to messages and link previews.
   *
   * \param[in] id_ Accent color identifier.
   * \param[in] built_in_accent_color_id_ Identifier of a built-in color to use in places, where only one color is needed; 0-6.
   * \param[in] light_theme_colors_ The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in light themes.
   * \param[in] dark_theme_colors_ The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes.
   * \param[in] min_channel_chat_boost_level_ The minimum chat boost level required to use the color in a channel chat.
   */
  accentColor(int32 id_, int32 built_in_accent_color_id_, array<int32> &&light_theme_colors_, array<int32> &&dark_theme_colors_, int32 min_channel_chat_boost_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -496870680;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about the period of inactivity after which the current user's account will automatically be deleted.
 */
class accountTtl final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Number of days of inactivity before the account will be flagged for deletion; 30-730 days.
  int32 days_;

  /**
   * Contains information about the period of inactivity after which the current user's account will automatically be deleted.
   */
  accountTtl();

  /**
   * Contains information about the period of inactivity after which the current user's account will automatically be deleted.
   *
   * \param[in] days_ Number of days of inactivity before the account will be flagged for deletion; 30-730 days.
   */
  explicit accountTtl(int32 days_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1324495492;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class MessageSender;

class ReactionType;

/**
 * Represents a reaction applied to a message.
 */
class addedReaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of the reaction.
  object_ptr<ReactionType> type_;
  /// Identifier of the chat member, applied the reaction.
  object_ptr<MessageSender> sender_id_;
  /// True, if the reaction was added by the current user.
  bool is_outgoing_;
  /// Point in time (Unix timestamp) when the reaction was added.
  int32 date_;

  /**
   * Represents a reaction applied to a message.
   */
  addedReaction();

  /**
   * Represents a reaction applied to a message.
   *
   * \param[in] type_ Type of the reaction.
   * \param[in] sender_id_ Identifier of the chat member, applied the reaction.
   * \param[in] is_outgoing_ True, if the reaction was added by the current user.
   * \param[in] date_ Point in time (Unix timestamp) when the reaction was added.
   */
  addedReaction(object_ptr<ReactionType> &&type_, object_ptr<MessageSender> &&sender_id_, bool is_outgoing_, int32 date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1258586525;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class addedReaction;

/**
 * Represents a list of reactions added to a message.
 */
class addedReactions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The total number of found reactions.
  int32 total_count_;
  /// The list of added reactions.
  array<object_ptr<addedReaction>> reactions_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Represents a list of reactions added to a message.
   */
  addedReactions();

  /**
   * Represents a list of reactions added to a message.
   *
   * \param[in] total_count_ The total number of found reactions.
   * \param[in] reactions_ The list of added reactions.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  addedReactions(int32 total_count_, array<object_ptr<addedReaction>> &&reactions_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 226352304;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an address.
 */
class address final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A two-letter ISO 3166-1 alpha-2 country code.
  string country_code_;
  /// State, if applicable.
  string state_;
  /// City.
  string city_;
  /// First line of the address.
  string street_line1_;
  /// Second line of the address.
  string street_line2_;
  /// Address postal code.
  string postal_code_;

  /**
   * Describes an address.
   */
  address();

  /**
   * Describes an address.
   *
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code.
   * \param[in] state_ State, if applicable.
   * \param[in] city_ City.
   * \param[in] street_line1_ First line of the address.
   * \param[in] street_line2_ Second line of the address.
   * \param[in] postal_code_ Address postal code.
   */
  address(string const &country_code_, string const &state_, string const &city_, string const &street_line1_, string const &street_line2_, string const &postal_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2043654342;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class starAmount;

/**
 * Contains information about an affiliate that received commission from a Telegram Star transaction.
 */
class affiliateInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner.
  int32 commission_per_mille_;
  /// Identifier of the chat which received the commission.
  int53 affiliate_chat_id_;
  /// The amount of Telegram Stars that were received by the affiliate; can be negative for refunds.
  object_ptr<starAmount> star_amount_;

  /**
   * Contains information about an affiliate that received commission from a Telegram Star transaction.
   */
  affiliateInfo();

  /**
   * Contains information about an affiliate that received commission from a Telegram Star transaction.
   *
   * \param[in] commission_per_mille_ The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner.
   * \param[in] affiliate_chat_id_ Identifier of the chat which received the commission.
   * \param[in] star_amount_ The amount of Telegram Stars that were received by the affiliate; can be negative for refunds.
   */
  affiliateInfo(int32 commission_per_mille_, int53 affiliate_chat_id_, object_ptr<starAmount> &&star_amount_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1312695046;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class affiliateProgramParameters;

class starAmount;

/**
 * Contains information about an active affiliate program.
 */
class affiliateProgramInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Parameters of the affiliate program.
  object_ptr<affiliateProgramParameters> parameters_;
  /// Point in time (Unix timestamp) when the affiliate program will be closed; 0 if the affiliate program isn't scheduled to be closed. If positive, then the program can't be connected using connectChatAffiliateProgram, but active connections will work until the date.
  int32 end_date_;
  /// The amount of daily revenue per user in Telegram Stars of the bot that created the affiliate program.
  object_ptr<starAmount> daily_revenue_per_user_amount_;

  /**
   * Contains information about an active affiliate program.
   */
  affiliateProgramInfo();

  /**
   * Contains information about an active affiliate program.
   *
   * \param[in] parameters_ Parameters of the affiliate program.
   * \param[in] end_date_ Point in time (Unix timestamp) when the affiliate program will be closed; 0 if the affiliate program isn't scheduled to be closed. If positive, then the program can't be connected using connectChatAffiliateProgram, but active connections will work until the date.
   * \param[in] daily_revenue_per_user_amount_ The amount of daily revenue per user in Telegram Stars of the bot that created the affiliate program.
   */
  affiliateProgramInfo(object_ptr<affiliateProgramParameters> &&parameters_, int32 end_date_, object_ptr<starAmount> &&daily_revenue_per_user_amount_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1761810251;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes parameters of an affiliate program.
 */
class affiliateProgramParameters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner; getOption(&quot;affiliate_program_commission_per_mille_min&quot;)-getOption(&quot;affiliate_program_commission_per_mille_max&quot;).
  int32 commission_per_mille_;
  /// Number of months the program will be active; 0-36. If 0, then the program is eternal.
  int32 month_count_;

  /**
   * Describes parameters of an affiliate program.
   */
  affiliateProgramParameters();

  /**
   * Describes parameters of an affiliate program.
   *
   * \param[in] commission_per_mille_ The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner; getOption(&quot;affiliate_program_commission_per_mille_min&quot;)-getOption(&quot;affiliate_program_commission_per_mille_max&quot;).
   * \param[in] month_count_ Number of months the program will be active; 0-36. If 0, then the program is eternal.
   */
  affiliateProgramParameters(int32 commission_per_mille_, int32 month_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1642662996;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the order of the found affiliate programs.
 */
class AffiliateProgramSortOrder: public Object {
 public:
};

/**
 * The affiliate programs must be sorted by the profitability.
 */
class affiliateProgramSortOrderProfitability final : public AffiliateProgramSortOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The affiliate programs must be sorted by the profitability.
   */
  affiliateProgramSortOrderProfitability();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1963282585;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The affiliate programs must be sorted by creation date.
 */
class affiliateProgramSortOrderCreationDate final : public AffiliateProgramSortOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The affiliate programs must be sorted by creation date.
   */
  affiliateProgramSortOrderCreationDate();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1558628083;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The affiliate programs must be sorted by the expected revenue.
 */
class affiliateProgramSortOrderRevenue final : public AffiliateProgramSortOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The affiliate programs must be sorted by the expected revenue.
   */
  affiliateProgramSortOrderRevenue();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1923269304;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

/**
 * Describes an alternative re-encoded quality of a video file.
 */
class alternativeVideo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Video width.
  int32 width_;
  /// Video height.
  int32 height_;
  /// Codec used for video file encoding, for example, &quot;h264&quot;, &quot;h265&quot;, or &quot;av1&quot;.
  string codec_;
  /// HLS file describing the video.
  object_ptr<file> hls_file_;
  /// File containing the video.
  object_ptr<file> video_;

  /**
   * Describes an alternative re-encoded quality of a video file.
   */
  alternativeVideo();

  /**
   * Describes an alternative re-encoded quality of a video file.
   *
   * \param[in] width_ Video width.
   * \param[in] height_ Video height.
   * \param[in] codec_ Codec used for video file encoding, for example, &quot;h264&quot;, &quot;h265&quot;, or &quot;av1&quot;.
   * \param[in] hls_file_ HLS file describing the video.
   * \param[in] video_ File containing the video.
   */
  alternativeVideo(int32 width_, int32 height_, string const &codec_, object_ptr<file> &&hls_file_, object_ptr<file> &&video_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1076216909;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

/**
 * Animated variant of a chat photo in MPEG4 format.
 */
class animatedChatPhoto final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Animation width and height.
  int32 length_;
  /// Information about the animation file.
  object_ptr<file> file_;
  /// Timestamp of the frame, used as a static chat photo.
  double main_frame_timestamp_;

  /**
   * Animated variant of a chat photo in MPEG4 format.
   */
  animatedChatPhoto();

  /**
   * Animated variant of a chat photo in MPEG4 format.
   *
   * \param[in] length_ Animation width and height.
   * \param[in] file_ Information about the animation file.
   * \param[in] main_frame_timestamp_ Timestamp of the frame, used as a static chat photo.
   */
  animatedChatPhoto(int32 length_, object_ptr<file> &&file_, double main_frame_timestamp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 191994926;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

class sticker;

/**
 * Describes an animated or custom representation of an emoji.
 */
class animatedEmoji final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Sticker for the emoji; may be null if yet unknown for a custom emoji. If the sticker is a custom emoji, then it can have arbitrary format.
  object_ptr<sticker> sticker_;
  /// Expected width of the sticker, which can be used if the sticker is null.
  int32 sticker_width_;
  /// Expected height of the sticker, which can be used if the sticker is null.
  int32 sticker_height_;
  /// Emoji modifier fitzpatrick type; 0-6; 0 if none.
  int32 fitzpatrick_type_;
  /// File containing the sound to be played when the sticker is clicked; may be null. The sound is encoded with the Opus codec, and stored inside an OGG container.
  object_ptr<file> sound_;

  /**
   * Describes an animated or custom representation of an emoji.
   */
  animatedEmoji();

  /**
   * Describes an animated or custom representation of an emoji.
   *
   * \param[in] sticker_ Sticker for the emoji; may be null if yet unknown for a custom emoji. If the sticker is a custom emoji, then it can have arbitrary format.
   * \param[in] sticker_width_ Expected width of the sticker, which can be used if the sticker is null.
   * \param[in] sticker_height_ Expected height of the sticker, which can be used if the sticker is null.
   * \param[in] fitzpatrick_type_ Emoji modifier fitzpatrick type; 0-6; 0 if none.
   * \param[in] sound_ File containing the sound to be played when the sticker is clicked; may be null. The sound is encoded with the Opus codec, and stored inside an OGG container.
   */
  animatedEmoji(object_ptr<sticker> &&sticker_, int32 sticker_width_, int32 sticker_height_, int32 fitzpatrick_type_, object_ptr<file> &&sound_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1378918079;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

class minithumbnail;

class thumbnail;

/**
 * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
 */
class animation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Duration of the animation, in seconds; as defined by the sender.
  int32 duration_;
  /// Width of the animation.
  int32 width_;
  /// Height of the animation.
  int32 height_;
  /// Original name of the file; as defined by the sender.
  string file_name_;
  /// MIME type of the file, usually &quot;image/gif&quot; or &quot;video/mp4&quot;.
  string mime_type_;
  /// True, if stickers were added to the animation. The list of corresponding sticker set can be received using getAttachedStickerSets.
  bool has_stickers_;
  /// Animation minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// Animation thumbnail in JPEG or MPEG4 format; may be null.
  object_ptr<thumbnail> thumbnail_;
  /// File containing the animation.
  object_ptr<file> animation_;

  /**
   * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
   */
  animation();

  /**
   * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
   *
   * \param[in] duration_ Duration of the animation, in seconds; as defined by the sender.
   * \param[in] width_ Width of the animation.
   * \param[in] height_ Height of the animation.
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ MIME type of the file, usually &quot;image/gif&quot; or &quot;video/mp4&quot;.
   * \param[in] has_stickers_ True, if stickers were added to the animation. The list of corresponding sticker set can be received using getAttachedStickerSets.
   * \param[in] minithumbnail_ Animation minithumbnail; may be null.
   * \param[in] thumbnail_ Animation thumbnail in JPEG or MPEG4 format; may be null.
   * \param[in] animation_ File containing the animation.
   */
  animation(int32 duration_, int32 width_, int32 height_, string const &file_name_, string const &mime_type_, bool has_stickers_, object_ptr<minithumbnail> &&minithumbnail_, object_ptr<thumbnail> &&thumbnail_, object_ptr<file> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -872359106;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class animation;

/**
 * Represents a list of animations.
 */
class animations final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of animations.
  array<object_ptr<animation>> animations_;

  /**
   * Represents a list of animations.
   */
  animations();

  /**
   * Represents a list of animations.
   *
   * \param[in] animations_ List of animations.
   */
  explicit animations(array<object_ptr<animation>> &&animations_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 344216945;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains settings for automatic moving of chats to and from the Archive chat lists.
 */
class archiveChatListSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if new chats from non-contacts will be automatically archived and muted. Can be set to true only if the option &quot;can_archive_and_mute_new_chats_from_unknown_users&quot; is true.
  bool archive_and_mute_new_chats_from_unknown_users_;
  /// True, if unmuted chats will be kept in the Archive chat list when they get a new message.
  bool keep_unmuted_chats_archived_;
  /// True, if unmuted chats, that are always included or pinned in a folder, will be kept in the Archive chat list when they get a new message. Ignored if keep_unmuted_chats_archived == true.
  bool keep_chats_from_folders_archived_;

  /**
   * Contains settings for automatic moving of chats to and from the Archive chat lists.
   */
  archiveChatListSettings();

  /**
   * Contains settings for automatic moving of chats to and from the Archive chat lists.
   *
   * \param[in] archive_and_mute_new_chats_from_unknown_users_ True, if new chats from non-contacts will be automatically archived and muted. Can be set to true only if the option &quot;can_archive_and_mute_new_chats_from_unknown_users&quot; is true.
   * \param[in] keep_unmuted_chats_archived_ True, if unmuted chats will be kept in the Archive chat list when they get a new message.
   * \param[in] keep_chats_from_folders_archived_ True, if unmuted chats, that are always included or pinned in a folder, will be kept in the Archive chat list when they get a new message. Ignored if keep_unmuted_chats_archived == true.
   */
  archiveChatListSettings(bool archive_and_mute_new_chats_from_unknown_users_, bool keep_unmuted_chats_archived_, bool keep_chats_from_folders_archived_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1058499236;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class attachmentMenuBotColor;

class file;

/**
 * Represents a bot, which can be added to attachment or side menu.
 */
class attachmentMenuBot final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the bot.
  int53 bot_user_id_;
  /// True, if the bot supports opening from attachment menu in the chat with the bot.
  bool supports_self_chat_;
  /// True, if the bot supports opening from attachment menu in private chats with ordinary users.
  bool supports_user_chats_;
  /// True, if the bot supports opening from attachment menu in private chats with other bots.
  bool supports_bot_chats_;
  /// True, if the bot supports opening from attachment menu in basic group and supergroup chats.
  bool supports_group_chats_;
  /// True, if the bot supports opening from attachment menu in channel chats.
  bool supports_channel_chats_;
  /// True, if the user must be asked for the permission to send messages to the bot.
  bool request_write_access_;
  /// True, if the bot was explicitly added by the user. If the bot isn't added, then on the first bot launch toggleBotIsAddedToAttachmentMenu must be called and the bot must be added or removed.
  bool is_added_;
  /// True, if the bot must be shown in the attachment menu.
  bool show_in_attachment_menu_;
  /// True, if the bot must be shown in the side menu.
  bool show_in_side_menu_;
  /// True, if a disclaimer, why the bot is shown in the side menu, is needed.
  bool show_disclaimer_in_side_menu_;
  /// Name for the bot in attachment menu.
  string name_;
  /// Color to highlight selected name of the bot if appropriate; may be null.
  object_ptr<attachmentMenuBotColor> name_color_;
  /// Default icon for the bot in SVG format; may be null.
  object_ptr<file> default_icon_;
  /// Icon for the bot in SVG format for the official iOS app; may be null.
  object_ptr<file> ios_static_icon_;
  /// Icon for the bot in TGS format for the official iOS app; may be null.
  object_ptr<file> ios_animated_icon_;
  /// Icon for the bot in PNG format for the official iOS app side menu; may be null.
  object_ptr<file> ios_side_menu_icon_;
  /// Icon for the bot in TGS format for the official Android app; may be null.
  object_ptr<file> android_icon_;
  /// Icon for the bot in SVG format for the official Android app side menu; may be null.
  object_ptr<file> android_side_menu_icon_;
  /// Icon for the bot in TGS format for the official native macOS app; may be null.
  object_ptr<file> macos_icon_;
  /// Icon for the bot in PNG format for the official macOS app side menu; may be null.
  object_ptr<file> macos_side_menu_icon_;
  /// Color to highlight selected icon of the bot if appropriate; may be null.
  object_ptr<attachmentMenuBotColor> icon_color_;
  /// Default placeholder for opened Web Apps in SVG format; may be null.
  object_ptr<file> web_app_placeholder_;

  /**
   * Represents a bot, which can be added to attachment or side menu.
   */
  attachmentMenuBot();

  /**
   * Represents a bot, which can be added to attachment or side menu.
   *
   * \param[in] bot_user_id_ User identifier of the bot.
   * \param[in] supports_self_chat_ True, if the bot supports opening from attachment menu in the chat with the bot.
   * \param[in] supports_user_chats_ True, if the bot supports opening from attachment menu in private chats with ordinary users.
   * \param[in] supports_bot_chats_ True, if the bot supports opening from attachment menu in private chats with other bots.
   * \param[in] supports_group_chats_ True, if the bot supports opening from attachment menu in basic group and supergroup chats.
   * \param[in] supports_channel_chats_ True, if the bot supports opening from attachment menu in channel chats.
   * \param[in] request_write_access_ True, if the user must be asked for the permission to send messages to the bot.
   * \param[in] is_added_ True, if the bot was explicitly added by the user. If the bot isn't added, then on the first bot launch toggleBotIsAddedToAttachmentMenu must be called and the bot must be added or removed.
   * \param[in] show_in_attachment_menu_ True, if the bot must be shown in the attachment menu.
   * \param[in] show_in_side_menu_ True, if the bot must be shown in the side menu.
   * \param[in] show_disclaimer_in_side_menu_ True, if a disclaimer, why the bot is shown in the side menu, is needed.
   * \param[in] name_ Name for the bot in attachment menu.
   * \param[in] name_color_ Color to highlight selected name of the bot if appropriate; may be null.
   * \param[in] default_icon_ Default icon for the bot in SVG format; may be null.
   * \param[in] ios_static_icon_ Icon for the bot in SVG format for the official iOS app; may be null.
   * \param[in] ios_animated_icon_ Icon for the bot in TGS format for the official iOS app; may be null.
   * \param[in] ios_side_menu_icon_ Icon for the bot in PNG format for the official iOS app side menu; may be null.
   * \param[in] android_icon_ Icon for the bot in TGS format for the official Android app; may be null.
   * \param[in] android_side_menu_icon_ Icon for the bot in SVG format for the official Android app side menu; may be null.
   * \param[in] macos_icon_ Icon for the bot in TGS format for the official native macOS app; may be null.
   * \param[in] macos_side_menu_icon_ Icon for the bot in PNG format for the official macOS app side menu; may be null.
   * \param[in] icon_color_ Color to highlight selected icon of the bot if appropriate; may be null.
   * \param[in] web_app_placeholder_ Default placeholder for opened Web Apps in SVG format; may be null.
   */
  attachmentMenuBot(int53 bot_user_id_, bool supports_self_chat_, bool supports_user_chats_, bool supports_bot_chats_, bool supports_group_chats_, bool supports_channel_chats_, bool request_write_access_, bool is_added_, bool show_in_attachment_menu_, bool show_in_side_menu_, bool show_disclaimer_in_side_menu_, string const &name_, object_ptr<attachmentMenuBotColor> &&name_color_, object_ptr<file> &&default_icon_, object_ptr<file> &&ios_static_icon_, object_ptr<file> &&ios_animated_icon_, object_ptr<file> &&ios_side_menu_icon_, object_ptr<file> &&android_icon_, object_ptr<file> &&android_side_menu_icon_, object_ptr<file> &&macos_icon_, object_ptr<file> &&macos_side_menu_icon_, object_ptr<attachmentMenuBotColor> &&icon_color_, object_ptr<file> &&web_app_placeholder_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1183966273;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a color to highlight a bot added to attachment menu.
 */
class attachmentMenuBotColor final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Color in the RGB format for light themes.
  int32 light_color_;
  /// Color in the RGB format for dark themes.
  int32 dark_color_;

  /**
   * Describes a color to highlight a bot added to attachment menu.
   */
  attachmentMenuBotColor();

  /**
   * Describes a color to highlight a bot added to attachment menu.
   *
   * \param[in] light_color_ Color in the RGB format for light themes.
   * \param[in] dark_color_ Color in the RGB format for dark themes.
   */
  attachmentMenuBotColor(int32 light_color_, int32 dark_color_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1680039612;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

class minithumbnail;

class thumbnail;

/**
 * Describes an audio file. Audio is usually in MP3 or M4A format.
 */
class audio final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Duration of the audio, in seconds; as defined by the sender.
  int32 duration_;
  /// Title of the audio; as defined by the sender.
  string title_;
  /// Performer of the audio; as defined by the sender.
  string performer_;
  /// Original name of the file; as defined by the sender.
  string file_name_;
  /// The MIME type of the file; as defined by the sender.
  string mime_type_;
  /// The minithumbnail of the album cover; may be null.
  object_ptr<minithumbnail> album_cover_minithumbnail_;
  /// The thumbnail of the album cover in JPEG format; as defined by the sender. The full size thumbnail is expected to be extracted from the downloaded audio file; may be null.
  object_ptr<thumbnail> album_cover_thumbnail_;
  /// Album cover variants to use if the downloaded audio file contains no album cover. Provided thumbnail dimensions are approximate.
  array<object_ptr<thumbnail>> external_album_covers_;
  /// File containing the audio.
  object_ptr<file> audio_;

  /**
   * Describes an audio file. Audio is usually in MP3 or M4A format.
   */
  audio();

  /**
   * Describes an audio file. Audio is usually in MP3 or M4A format.
   *
   * \param[in] duration_ Duration of the audio, in seconds; as defined by the sender.
   * \param[in] title_ Title of the audio; as defined by the sender.
   * \param[in] performer_ Performer of the audio; as defined by the sender.
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ The MIME type of the file; as defined by the sender.
   * \param[in] album_cover_minithumbnail_ The minithumbnail of the album cover; may be null.
   * \param[in] album_cover_thumbnail_ The thumbnail of the album cover in JPEG format; as defined by the sender. The full size thumbnail is expected to be extracted from the downloaded audio file; may be null.
   * \param[in] external_album_covers_ Album cover variants to use if the downloaded audio file contains no album cover. Provided thumbnail dimensions are approximate.
   * \param[in] audio_ File containing the audio.
   */
  audio(int32 duration_, string const &title_, string const &performer_, string const &file_name_, string const &mime_type_, object_ptr<minithumbnail> &&album_cover_minithumbnail_, object_ptr<thumbnail> &&album_cover_thumbnail_, array<object_ptr<thumbnail>> &&external_album_covers_, object_ptr<file> &&audio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -166398841;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class AuthenticationCodeType;

/**
 * Information about the authentication code that was sent.
 */
class authenticationCodeInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A phone number that is being authenticated.
  string phone_number_;
  /// The way the code was sent to the user.
  object_ptr<AuthenticationCodeType> type_;
  /// The way the next code will be sent to the user; may be null.
  object_ptr<AuthenticationCodeType> next_type_;
  /// Timeout before the code can be re-sent, in seconds.
  int32 timeout_;

  /**
   * Information about the authentication code that was sent.
   */
  authenticationCodeInfo();

  /**
   * Information about the authentication code that was sent.
   *
   * \param[in] phone_number_ A phone number that is being authenticated.
   * \param[in] type_ The way the code was sent to the user.
   * \param[in] next_type_ The way the next code will be sent to the user; may be null.
   * \param[in] timeout_ Timeout before the code can be re-sent, in seconds.
   */
  authenticationCodeInfo(string const &phone_number_, object_ptr<AuthenticationCodeType> &&type_, object_ptr<AuthenticationCodeType> &&next_type_, int32 timeout_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -860345416;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class FirebaseDeviceVerificationParameters;

/**
 * This class is an abstract base class.
 * Provides information about the method by which an authentication code is delivered to the user.
 */
class AuthenticationCodeType: public Object {
 public:
};

/**
 * A digit-only authentication code is delivered via a private Telegram message, which can be viewed from another active session.
 */
class authenticationCodeTypeTelegramMessage final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Length of the code.
  int32 length_;

  /**
   * A digit-only authentication code is delivered via a private Telegram message, which can be viewed from another active session.
   */
  authenticationCodeTypeTelegramMessage();

  /**
   * A digit-only authentication code is delivered via a private Telegram message, which can be viewed from another active session.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeTelegramMessage(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2079628074;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A digit-only authentication code is delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
 */
class authenticationCodeTypeSms final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Length of the code.
  int32 length_;

  /**
   * A digit-only authentication code is delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   */
  authenticationCodeTypeSms();

  /**
   * A digit-only authentication code is delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeSms(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 962650760;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is a word delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
 */
class authenticationCodeTypeSmsWord final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The first letters of the word if known.
  string first_letter_;

  /**
   * An authentication code is a word delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   */
  authenticationCodeTypeSmsWord();

  /**
   * An authentication code is a word delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   *
   * \param[in] first_letter_ The first letters of the word if known.
   */
  explicit authenticationCodeTypeSmsWord(string const &first_letter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1509540765;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is a phrase from multiple words delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
 */
class authenticationCodeTypeSmsPhrase final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The first word of the phrase if known.
  string first_word_;

  /**
   * An authentication code is a phrase from multiple words delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   */
  authenticationCodeTypeSmsPhrase();

  /**
   * An authentication code is a phrase from multiple words delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   *
   * \param[in] first_word_ The first word of the phrase if known.
   */
  explicit authenticationCodeTypeSmsPhrase(string const &first_word_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 784108753;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A digit-only authentication code is delivered via a phone call to the specified phone number.
 */
class authenticationCodeTypeCall final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Length of the code.
  int32 length_;

  /**
   * A digit-only authentication code is delivered via a phone call to the specified phone number.
   */
  authenticationCodeTypeCall();

  /**
   * A digit-only authentication code is delivered via a phone call to the specified phone number.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeCall(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1636265063;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
 */
class authenticationCodeTypeFlashCall final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Pattern of the phone number from which the call will be made.
  string pattern_;

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
   */
  authenticationCodeTypeFlashCall();

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
   *
   * \param[in] pattern_ Pattern of the phone number from which the call will be made.
   */
  explicit authenticationCodeTypeFlashCall(string const &pattern_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1395882402;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
 */
class authenticationCodeTypeMissedCall final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Prefix of the phone number from which the call will be made.
  string phone_number_prefix_;
  /// Number of digits in the code, excluding the prefix.
  int32 length_;

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
   */
  authenticationCodeTypeMissedCall();

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
   *
   * \param[in] phone_number_prefix_ Prefix of the phone number from which the call will be made.
   * \param[in] length_ Number of digits in the code, excluding the prefix.
   */
  authenticationCodeTypeMissedCall(string const &phone_number_prefix_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 700123783;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A digit-only authentication code is delivered to https://fragment.com. The user must be logged in there via a wallet owning the phone number's NFT.
 */
class authenticationCodeTypeFragment final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// URL to open to receive the code.
  string url_;
  /// Length of the code.
  int32 length_;

  /**
   * A digit-only authentication code is delivered to https://fragment.com. The user must be logged in there via a wallet owning the phone number's NFT.
   */
  authenticationCodeTypeFragment();

  /**
   * A digit-only authentication code is delivered to https://fragment.com. The user must be logged in there via a wallet owning the phone number's NFT.
   *
   * \param[in] url_ URL to open to receive the code.
   * \param[in] length_ Length of the code.
   */
  authenticationCodeTypeFragment(string const &url_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2129693491;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A digit-only authentication code is delivered via Firebase Authentication to the official Android application.
 */
class authenticationCodeTypeFirebaseAndroid final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Parameters to be used for device verification.
  object_ptr<FirebaseDeviceVerificationParameters> device_verification_parameters_;
  /// Length of the code.
  int32 length_;

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official Android application.
   */
  authenticationCodeTypeFirebaseAndroid();

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official Android application.
   *
   * \param[in] device_verification_parameters_ Parameters to be used for device verification.
   * \param[in] length_ Length of the code.
   */
  authenticationCodeTypeFirebaseAndroid(object_ptr<FirebaseDeviceVerificationParameters> &&device_verification_parameters_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1872475422;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A digit-only authentication code is delivered via Firebase Authentication to the official iOS application.
 */
class authenticationCodeTypeFirebaseIos final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Receipt of successful application token validation to compare with receipt from push notification.
  string receipt_;
  /// Time after the next authentication method is expected to be used if verification push notification isn't received, in seconds.
  int32 push_timeout_;
  /// Length of the code.
  int32 length_;

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official iOS application.
   */
  authenticationCodeTypeFirebaseIos();

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official iOS application.
   *
   * \param[in] receipt_ Receipt of successful application token validation to compare with receipt from push notification.
   * \param[in] push_timeout_ Time after the next authentication method is expected to be used if verification push notification isn't received, in seconds.
   * \param[in] length_ Length of the code.
   */
  authenticationCodeTypeFirebaseIos(string const &receipt_, int32 push_timeout_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -11162989;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class EmailAddressResetState;

class authenticationCodeInfo;

class emailAddressAuthenticationCodeInfo;

class termsOfService;

/**
 * This class is an abstract base class.
 * Represents the current authorization state of the TDLib client.
 */
class AuthorizationState: public Object {
 public:
};

/**
 * Initialization parameters are needed. Call setTdlibParameters to provide them.
 */
class authorizationStateWaitTdlibParameters final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Initialization parameters are needed. Call setTdlibParameters to provide them.
   */
  authorizationStateWaitTdlibParameters();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 904720988;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib needs the user's phone number to authorize. Call setAuthenticationPhoneNumber to provide the phone number, or use requestQrCodeAuthentication or checkAuthenticationBotToken for other authentication options.
 */
class authorizationStateWaitPhoneNumber final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * TDLib needs the user's phone number to authorize. Call setAuthenticationPhoneNumber to provide the phone number, or use requestQrCodeAuthentication or checkAuthenticationBotToken for other authentication options.
   */
  authorizationStateWaitPhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 306402531;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib needs the user's email address to authorize. Call setAuthenticationEmailAddress to provide the email address, or directly call checkAuthenticationEmailCode with Apple ID/Google ID token if allowed.
 */
class authorizationStateWaitEmailAddress final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if authorization through Apple ID is allowed.
  bool allow_apple_id_;
  /// True, if authorization through Google ID is allowed.
  bool allow_google_id_;

  /**
   * TDLib needs the user's email address to authorize. Call setAuthenticationEmailAddress to provide the email address, or directly call checkAuthenticationEmailCode with Apple ID/Google ID token if allowed.
   */
  authorizationStateWaitEmailAddress();

  /**
   * TDLib needs the user's email address to authorize. Call setAuthenticationEmailAddress to provide the email address, or directly call checkAuthenticationEmailCode with Apple ID/Google ID token if allowed.
   *
   * \param[in] allow_apple_id_ True, if authorization through Apple ID is allowed.
   * \param[in] allow_google_id_ True, if authorization through Google ID is allowed.
   */
  authorizationStateWaitEmailAddress(bool allow_apple_id_, bool allow_google_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1040478663;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib needs the user's authentication code sent to an email address to authorize. Call checkAuthenticationEmailCode to provide the code.
 */
class authorizationStateWaitEmailCode final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if authorization through Apple ID is allowed.
  bool allow_apple_id_;
  /// True, if authorization through Google ID is allowed.
  bool allow_google_id_;
  /// Information about the sent authentication code.
  object_ptr<emailAddressAuthenticationCodeInfo> code_info_;
  /// Reset state of the email address; may be null if the email address can't be reset.
  object_ptr<EmailAddressResetState> email_address_reset_state_;

  /**
   * TDLib needs the user's authentication code sent to an email address to authorize. Call checkAuthenticationEmailCode to provide the code.
   */
  authorizationStateWaitEmailCode();

  /**
   * TDLib needs the user's authentication code sent to an email address to authorize. Call checkAuthenticationEmailCode to provide the code.
   *
   * \param[in] allow_apple_id_ True, if authorization through Apple ID is allowed.
   * \param[in] allow_google_id_ True, if authorization through Google ID is allowed.
   * \param[in] code_info_ Information about the sent authentication code.
   * \param[in] email_address_reset_state_ Reset state of the email address; may be null if the email address can't be reset.
   */
  authorizationStateWaitEmailCode(bool allow_apple_id_, bool allow_google_id_, object_ptr<emailAddressAuthenticationCodeInfo> &&code_info_, object_ptr<EmailAddressResetState> &&email_address_reset_state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1868627365;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib needs the user's authentication code to authorize. Call checkAuthenticationCode to check the code.
 */
class authorizationStateWaitCode final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about the authorization code that was sent.
  object_ptr<authenticationCodeInfo> code_info_;

  /**
   * TDLib needs the user's authentication code to authorize. Call checkAuthenticationCode to check the code.
   */
  authorizationStateWaitCode();

  /**
   * TDLib needs the user's authentication code to authorize. Call checkAuthenticationCode to check the code.
   *
   * \param[in] code_info_ Information about the authorization code that was sent.
   */
  explicit authorizationStateWaitCode(object_ptr<authenticationCodeInfo> &&code_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 52643073;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
 */
class authorizationStateWaitOtherDeviceConfirmation final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A tg:// URL for the QR code. The link will be updated frequently.
  string link_;

  /**
   * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
   */
  authorizationStateWaitOtherDeviceConfirmation();

  /**
   * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
   *
   * \param[in] link_ A tg:// URL for the QR code. The link will be updated frequently.
   */
  explicit authorizationStateWaitOtherDeviceConfirmation(string const &link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 860166378;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration. Call registerUser to accept the terms of service and provide the data.
 */
class authorizationStateWaitRegistration final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Telegram terms of service.
  object_ptr<termsOfService> terms_of_service_;

  /**
   * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration. Call registerUser to accept the terms of service and provide the data.
   */
  authorizationStateWaitRegistration();

  /**
   * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration. Call registerUser to accept the terms of service and provide the data.
   *
   * \param[in] terms_of_service_ Telegram terms of service.
   */
  explicit authorizationStateWaitRegistration(object_ptr<termsOfService> &&terms_of_service_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 550350511;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has been authorized, but needs to enter a 2-step verification password to start using the application. Call checkAuthenticationPassword to provide the password, or requestAuthenticationPasswordRecovery to recover the password, or deleteAccount to delete the account after a week.
 */
class authorizationStateWaitPassword final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Hint for the password; may be empty.
  string password_hint_;
  /// True, if a recovery email address has been set up.
  bool has_recovery_email_address_;
  /// True, if some Telegram Passport elements were saved.
  bool has_passport_data_;
  /// Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent.
  string recovery_email_address_pattern_;

  /**
   * The user has been authorized, but needs to enter a 2-step verification password to start using the application. Call checkAuthenticationPassword to provide the password, or requestAuthenticationPasswordRecovery to recover the password, or deleteAccount to delete the account after a week.
   */
  authorizationStateWaitPassword();

  /**
   * The user has been authorized, but needs to enter a 2-step verification password to start using the application. Call checkAuthenticationPassword to provide the password, or requestAuthenticationPasswordRecovery to recover the password, or deleteAccount to delete the account after a week.
   *
   * \param[in] password_hint_ Hint for the password; may be empty.
   * \param[in] has_recovery_email_address_ True, if a recovery email address has been set up.
   * \param[in] has_passport_data_ True, if some Telegram Passport elements were saved.
   * \param[in] recovery_email_address_pattern_ Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent.
   */
  authorizationStateWaitPassword(string const &password_hint_, bool has_recovery_email_address_, bool has_passport_data_, string const &recovery_email_address_pattern_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 112238030;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has been successfully authorized. TDLib is now ready to answer general requests.
 */
class authorizationStateReady final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user has been successfully authorized. TDLib is now ready to answer general requests.
   */
  authorizationStateReady();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1834871737;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is currently logging out.
 */
class authorizationStateLoggingOut final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is currently logging out.
   */
  authorizationStateLoggingOut();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 154449270;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib is closing, all subsequent queries will be answered with the error 500. Note that closing TDLib can take a while. All resources will be freed only after authorizationStateClosed has been received.
 */
class authorizationStateClosing final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * TDLib is closing, all subsequent queries will be answered with the error 500. Note that closing TDLib can take a while. All resources will be freed only after authorizationStateClosed has been received.
   */
  authorizationStateClosing();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 445855311;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * TDLib client is in its final state. All databases are closed and all resources are released. No other updates will be received after this. All queries will be responded to with error code 500. To continue working, one must create a new instance of the TDLib client.
 */
class authorizationStateClosed final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * TDLib client is in its final state. All databases are closed and all resources are released. No other updates will be received after this. All queries will be responded to with error code 500. To continue working, one must create a new instance of the TDLib client.
   */
  authorizationStateClosed();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1526047584;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains auto-download settings.
 */
class autoDownloadSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if the auto-download is enabled.
  bool is_auto_download_enabled_;
  /// The maximum size of a photo file to be auto-downloaded, in bytes.
  int32 max_photo_file_size_;
  /// The maximum size of a video file to be auto-downloaded, in bytes.
  int53 max_video_file_size_;
  /// The maximum size of other file types to be auto-downloaded, in bytes.
  int53 max_other_file_size_;
  /// The maximum suggested bitrate for uploaded videos, in kbit/s.
  int32 video_upload_bitrate_;
  /// True, if the beginning of video files needs to be preloaded for instant playback.
  bool preload_large_videos_;
  /// True, if the next audio track needs to be preloaded while the user is listening to an audio file.
  bool preload_next_audio_;
  /// True, if stories needs to be preloaded.
  bool preload_stories_;
  /// True, if &quot;use less data for calls&quot; option needs to be enabled.
  bool use_less_data_for_calls_;

  /**
   * Contains auto-download settings.
   */
  autoDownloadSettings();

  /**
   * Contains auto-download settings.
   *
   * \param[in] is_auto_download_enabled_ True, if the auto-download is enabled.
   * \param[in] max_photo_file_size_ The maximum size of a photo file to be auto-downloaded, in bytes.
   * \param[in] max_video_file_size_ The maximum size of a video file to be auto-downloaded, in bytes.
   * \param[in] max_other_file_size_ The maximum size of other file types to be auto-downloaded, in bytes.
   * \param[in] video_upload_bitrate_ The maximum suggested bitrate for uploaded videos, in kbit/s.
   * \param[in] preload_large_videos_ True, if the beginning of video files needs to be preloaded for instant playback.
   * \param[in] preload_next_audio_ True, if the next audio track needs to be preloaded while the user is listening to an audio file.
   * \param[in] preload_stories_ True, if stories needs to be preloaded.
   * \param[in] use_less_data_for_calls_ True, if &quot;use less data for calls&quot; option needs to be enabled.
   */
  autoDownloadSettings(bool is_auto_download_enabled_, int32 max_photo_file_size_, int53 max_video_file_size_, int53 max_other_file_size_, int32 video_upload_bitrate_, bool preload_large_videos_, bool preload_next_audio_, bool preload_stories_, bool use_less_data_for_calls_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 991433696;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class autoDownloadSettings;

/**
 * Contains auto-download settings presets for the current user.
 */
class autoDownloadSettingsPresets final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Preset with lowest settings; expected to be used by default when roaming.
  object_ptr<autoDownloadSettings> low_;
  /// Preset with medium settings; expected to be used by default when using mobile data.
  object_ptr<autoDownloadSettings> medium_;
  /// Preset with highest settings; expected to be used by default when connected on Wi-Fi.
  object_ptr<autoDownloadSettings> high_;

  /**
   * Contains auto-download settings presets for the current user.
   */
  autoDownloadSettingsPresets();

  /**
   * Contains auto-download settings presets for the current user.
   *
   * \param[in] low_ Preset with lowest settings; expected to be used by default when roaming.
   * \param[in] medium_ Preset with medium settings; expected to be used by default when using mobile data.
   * \param[in] high_ Preset with highest settings; expected to be used by default when connected on Wi-Fi.
   */
  autoDownloadSettingsPresets(object_ptr<autoDownloadSettings> &&low_, object_ptr<autoDownloadSettings> &&medium_, object_ptr<autoDownloadSettings> &&high_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -782099166;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class autosaveSettingsException;

class scopeAutosaveSettings;

/**
 * Describes autosave settings.
 */
class autosaveSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Default autosave settings for private chats.
  object_ptr<scopeAutosaveSettings> private_chat_settings_;
  /// Default autosave settings for basic group and supergroup chats.
  object_ptr<scopeAutosaveSettings> group_settings_;
  /// Default autosave settings for channel chats.
  object_ptr<scopeAutosaveSettings> channel_settings_;
  /// Autosave settings for specific chats.
  array<object_ptr<autosaveSettingsException>> exceptions_;

  /**
   * Describes autosave settings.
   */
  autosaveSettings();

  /**
   * Describes autosave settings.
   *
   * \param[in] private_chat_settings_ Default autosave settings for private chats.
   * \param[in] group_settings_ Default autosave settings for basic group and supergroup chats.
   * \param[in] channel_settings_ Default autosave settings for channel chats.
   * \param[in] exceptions_ Autosave settings for specific chats.
   */
  autosaveSettings(object_ptr<scopeAutosaveSettings> &&private_chat_settings_, object_ptr<scopeAutosaveSettings> &&group_settings_, object_ptr<scopeAutosaveSettings> &&channel_settings_, array<object_ptr<autosaveSettingsException>> &&exceptions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1629412502;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class scopeAutosaveSettings;

/**
 * Contains autosave settings for a chat, which overrides default settings for the corresponding scope.
 */
class autosaveSettingsException final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier.
  int53 chat_id_;
  /// Autosave settings for the chat.
  object_ptr<scopeAutosaveSettings> settings_;

  /**
   * Contains autosave settings for a chat, which overrides default settings for the corresponding scope.
   */
  autosaveSettingsException();

  /**
   * Contains autosave settings for a chat, which overrides default settings for the corresponding scope.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] settings_ Autosave settings for the chat.
   */
  autosaveSettingsException(int53 chat_id_, object_ptr<scopeAutosaveSettings> &&settings_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1483470280;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes scope of autosave settings.
 */
class AutosaveSettingsScope: public Object {
 public:
};

/**
 * Autosave settings applied to all private chats without chat-specific settings.
 */
class autosaveSettingsScopePrivateChats final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Autosave settings applied to all private chats without chat-specific settings.
   */
  autosaveSettingsScopePrivateChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1395227007;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Autosave settings applied to all basic group and supergroup chats without chat-specific settings.
 */
class autosaveSettingsScopeGroupChats final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Autosave settings applied to all basic group and supergroup chats without chat-specific settings.
   */
  autosaveSettingsScopeGroupChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 853544526;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Autosave settings applied to all channel chats without chat-specific settings.
 */
class autosaveSettingsScopeChannelChats final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Autosave settings applied to all channel chats without chat-specific settings.
   */
  autosaveSettingsScopeChannelChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -499572783;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Autosave settings applied to a chat.
 */
class autosaveSettingsScopeChat final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier.
  int53 chat_id_;

  /**
   * Autosave settings applied to a chat.
   */
  autosaveSettingsScopeChat();

  /**
   * Autosave settings applied to a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit autosaveSettingsScopeChat(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1632255255;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ReactionType;

/**
 * Represents an available reaction.
 */
class availableReaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of the reaction.
  object_ptr<ReactionType> type_;
  /// True, if Telegram Premium is needed to send the reaction.
  bool needs_premium_;

  /**
   * Represents an available reaction.
   */
  availableReaction();

  /**
   * Represents an available reaction.
   *
   * \param[in] type_ Type of the reaction.
   * \param[in] needs_premium_ True, if Telegram Premium is needed to send the reaction.
   */
  availableReaction(object_ptr<ReactionType> &&type_, bool needs_premium_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -117292153;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ReactionUnavailabilityReason;

class availableReaction;

/**
 * Represents a list of reactions that can be added to a message.
 */
class availableReactions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of reactions to be shown at the top.
  array<object_ptr<availableReaction>> top_reactions_;
  /// List of recently used reactions.
  array<object_ptr<availableReaction>> recent_reactions_;
  /// List of popular reactions.
  array<object_ptr<availableReaction>> popular_reactions_;
  /// True, if any custom emoji reaction can be added by Telegram Premium subscribers.
  bool allow_custom_emoji_;
  /// True, if the reactions will be tags and the message can be found by them.
  bool are_tags_;
  /// The reason why the current user can't add reactions to the message, despite some other users can; may be null if none.
  object_ptr<ReactionUnavailabilityReason> unavailability_reason_;

  /**
   * Represents a list of reactions that can be added to a message.
   */
  availableReactions();

  /**
   * Represents a list of reactions that can be added to a message.
   *
   * \param[in] top_reactions_ List of reactions to be shown at the top.
   * \param[in] recent_reactions_ List of recently used reactions.
   * \param[in] popular_reactions_ List of popular reactions.
   * \param[in] allow_custom_emoji_ True, if any custom emoji reaction can be added by Telegram Premium subscribers.
   * \param[in] are_tags_ True, if the reactions will be tags and the message can be found by them.
   * \param[in] unavailability_reason_ The reason why the current user can't add reactions to the message, despite some other users can; may be null if none.
   */
  availableReactions(array<object_ptr<availableReaction>> &&top_reactions_, array<object_ptr<availableReaction>> &&recent_reactions_, array<object_ptr<availableReaction>> &&popular_reactions_, bool allow_custom_emoji_, bool are_tags_, object_ptr<ReactionUnavailabilityReason> &&unavailability_reason_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 912529522;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class BackgroundType;

class document;

/**
 * Describes a chat background.
 */
class background final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique background identifier.
  int64 id_;
  /// True, if this is one of default backgrounds.
  bool is_default_;
  /// True, if the background is dark and is recommended to be used with dark theme.
  bool is_dark_;
  /// Unique background name.
  string name_;
  /// Document with the background; may be null. Null only for filled and chat theme backgrounds.
  object_ptr<document> document_;
  /// Type of the background.
  object_ptr<BackgroundType> type_;

  /**
   * Describes a chat background.
   */
  background();

  /**
   * Describes a chat background.
   *
   * \param[in] id_ Unique background identifier.
   * \param[in] is_default_ True, if this is one of default backgrounds.
   * \param[in] is_dark_ True, if the background is dark and is recommended to be used with dark theme.
   * \param[in] name_ Unique background name.
   * \param[in] document_ Document with the background; may be null. Null only for filled and chat theme backgrounds.
   * \param[in] type_ Type of the background.
   */
  background(int64 id_, bool is_default_, bool is_dark_, string const &name_, object_ptr<document> &&document_, object_ptr<BackgroundType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -429971172;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a fill of a background.
 */
class BackgroundFill: public Object {
 public:
};

/**
 * Describes a solid fill of a background.
 */
class backgroundFillSolid final : public BackgroundFill {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A color of the background in the RGB format.
  int32 color_;

  /**
   * Describes a solid fill of a background.
   */
  backgroundFillSolid();

  /**
   * Describes a solid fill of a background.
   *
   * \param[in] color_ A color of the background in the RGB format.
   */
  explicit backgroundFillSolid(int32 color_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1010678813;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a gradient fill of a background.
 */
class backgroundFillGradient final : public BackgroundFill {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A top color of the background in the RGB format.
  int32 top_color_;
  /// A bottom color of the background in the RGB format.
  int32 bottom_color_;
  /// Clockwise rotation angle of the gradient, in degrees; 0-359. Must always be divisible by 45.
  int32 rotation_angle_;

  /**
   * Describes a gradient fill of a background.
   */
  backgroundFillGradient();

  /**
   * Describes a gradient fill of a background.
   *
   * \param[in] top_color_ A top color of the background in the RGB format.
   * \param[in] bottom_color_ A bottom color of the background in the RGB format.
   * \param[in] rotation_angle_ Clockwise rotation angle of the gradient, in degrees; 0-359. Must always be divisible by 45.
   */
  backgroundFillGradient(int32 top_color_, int32 bottom_color_, int32 rotation_angle_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1839206017;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a freeform gradient fill of a background.
 */
class backgroundFillFreeformGradient final : public BackgroundFill {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A list of 3 or 4 colors of the freeform gradient in the RGB format.
  array<int32> colors_;

  /**
   * Describes a freeform gradient fill of a background.
   */
  backgroundFillFreeformGradient();

  /**
   * Describes a freeform gradient fill of a background.
   *
   * \param[in] colors_ A list of 3 or 4 colors of the freeform gradient in the RGB format.
   */
  explicit backgroundFillFreeformGradient(array<int32> &&colors_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1145469255;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class BackgroundFill;

/**
 * This class is an abstract base class.
 * Describes the type of background.
 */
class BackgroundType: public Object {
 public:
};

/**
 * A wallpaper in JPEG format.
 */
class backgroundTypeWallpaper final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if the wallpaper must be downscaled to fit in 450x450 square and then box-blurred with radius 12.
  bool is_blurred_;
  /// True, if the background needs to be slightly moved when device is tilted.
  bool is_moving_;

  /**
   * A wallpaper in JPEG format.
   */
  backgroundTypeWallpaper();

  /**
   * A wallpaper in JPEG format.
   *
   * \param[in] is_blurred_ True, if the wallpaper must be downscaled to fit in 450x450 square and then box-blurred with radius 12.
   * \param[in] is_moving_ True, if the background needs to be slightly moved when device is tilted.
   */
  backgroundTypeWallpaper(bool is_blurred_, bool is_moving_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1972128891;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
 */
class backgroundTypePattern final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Fill of the background.
  object_ptr<BackgroundFill> fill_;
  /// Intensity of the pattern when it is shown above the filled background; 0-100.
  int32 intensity_;
  /// True, if the background fill must be applied only to the pattern itself. All other pixels are black in this case. For dark themes only.
  bool is_inverted_;
  /// True, if the background needs to be slightly moved when device is tilted.
  bool is_moving_;

  /**
   * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
   */
  backgroundTypePattern();

  /**
   * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
   *
   * \param[in] fill_ Fill of the background.
   * \param[in] intensity_ Intensity of the pattern when it is shown above the filled background; 0-100.
   * \param[in] is_inverted_ True, if the background fill must be applied only to the pattern itself. All other pixels are black in this case. For dark themes only.
   * \param[in] is_moving_ True, if the background needs to be slightly moved when device is tilted.
   */
  backgroundTypePattern(object_ptr<BackgroundFill> &&fill_, int32 intensity_, bool is_inverted_, bool is_moving_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1290213117;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A filled background.
 */
class backgroundTypeFill final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The background fill.
  object_ptr<BackgroundFill> fill_;

  /**
   * A filled background.
   */
  backgroundTypeFill();

  /**
   * A filled background.
   *
   * \param[in] fill_ The background fill.
   */
  explicit backgroundTypeFill(object_ptr<BackgroundFill> &&fill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 993008684;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A background from a chat theme; can be used only as a chat background in channels.
 */
class backgroundTypeChatTheme final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the chat theme.
  string theme_name_;

  /**
   * A background from a chat theme; can be used only as a chat background in channels.
   */
  backgroundTypeChatTheme();

  /**
   * A background from a chat theme; can be used only as a chat background in channels.
   *
   * \param[in] theme_name_ Name of the chat theme.
   */
  explicit backgroundTypeChatTheme(string const &theme_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1299879762;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class background;

/**
 * Contains a list of backgrounds.
 */
class backgrounds final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A list of backgrounds.
  array<object_ptr<background>> backgrounds_;

  /**
   * Contains a list of backgrounds.
   */
  backgrounds();

  /**
   * Contains a list of backgrounds.
   *
   * \param[in] backgrounds_ A list of backgrounds.
   */
  explicit backgrounds(array<object_ptr<background>> &&backgrounds_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 724728704;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an action associated with a bank card number.
 */
class bankCardActionOpenUrl final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Action text.
  string text_;
  /// The URL to be opened.
  string url_;

  /**
   * Describes an action associated with a bank card number.
   */
  bankCardActionOpenUrl();

  /**
   * Describes an action associated with a bank card number.
   *
   * \param[in] text_ Action text.
   * \param[in] url_ The URL to be opened.
   */
  bankCardActionOpenUrl(string const &text_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -196454267;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class bankCardActionOpenUrl;

/**
 * Information about a bank card.
 */
class bankCardInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Title of the bank card description.
  string title_;
  /// Actions that can be done with the bank card number.
  array<object_ptr<bankCardActionOpenUrl>> actions_;

  /**
   * Information about a bank card.
   */
  bankCardInfo();

  /**
   * Information about a bank card.
   *
   * \param[in] title_ Title of the bank card description.
   * \param[in] actions_ Actions that can be done with the bank card number.
   */
  bankCardInfo(string const &title_, array<object_ptr<bankCardActionOpenUrl>> &&actions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2116647730;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatMemberStatus;

/**
 * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
 */
class basicGroup final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Group identifier.
  int53 id_;
  /// Number of members in the group.
  int32 member_count_;
  /// Status of the current user in the group.
  object_ptr<ChatMemberStatus> status_;
  /// True, if the group is active.
  bool is_active_;
  /// Identifier of the supergroup to which this group was upgraded; 0 if none.
  int53 upgraded_to_supergroup_id_;

  /**
   * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
   */
  basicGroup();

  /**
   * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
   *
   * \param[in] id_ Group identifier.
   * \param[in] member_count_ Number of members in the group.
   * \param[in] status_ Status of the current user in the group.
   * \param[in] is_active_ True, if the group is active.
   * \param[in] upgraded_to_supergroup_id_ Identifier of the supergroup to which this group was upgraded; 0 if none.
   */
  basicGroup(int53 id_, int32 member_count_, object_ptr<ChatMemberStatus> &&status_, bool is_active_, int53 upgraded_to_supergroup_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -194767217;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class botCommands;

class chatInviteLink;

class chatMember;

class chatPhoto;

/**
 * Contains full information about a basic group.
 */
class basicGroupFullInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat photo; may be null if empty or unknown. If non-null, then it is the same photo as in chat.photo.
  object_ptr<chatPhoto> photo_;
  /// Group description. Updated only after the basic group is opened.
  string description_;
  /// User identifier of the creator of the group; 0 if unknown.
  int53 creator_user_id_;
  /// Group members.
  array<object_ptr<chatMember>> members_;
  /// True, if non-administrators and non-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non-administrators after upgrading the basic group to a supergroup.
  bool can_hide_members_;
  /// True, if aggressive anti-spam checks can be enabled or disabled in the supergroup after upgrading the basic group to a supergroup.
  bool can_toggle_aggressive_anti_spam_;
  /// Primary invite link for this group; may be null. For chat administrators with can_invite_users right only. Updated only after the basic group is opened.
  object_ptr<chatInviteLink> invite_link_;
  /// List of commands of bots in the group.
  array<object_ptr<botCommands>> bot_commands_;

  /**
   * Contains full information about a basic group.
   */
  basicGroupFullInfo();

  /**
   * Contains full information about a basic group.
   *
   * \param[in] photo_ Chat photo; may be null if empty or unknown. If non-null, then it is the same photo as in chat.photo.
   * \param[in] description_ Group description. Updated only after the basic group is opened.
   * \param[in] creator_user_id_ User identifier of the creator of the group; 0 if unknown.
   * \param[in] members_ Group members.
   * \param[in] can_hide_members_ True, if non-administrators and non-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non-administrators after upgrading the basic group to a supergroup.
   * \param[in] can_toggle_aggressive_anti_spam_ True, if aggressive anti-spam checks can be enabled or disabled in the supergroup after upgrading the basic group to a supergroup.
   * \param[in] invite_link_ Primary invite link for this group; may be null. For chat administrators with can_invite_users right only. Updated only after the basic group is opened.
   * \param[in] bot_commands_ List of commands of bots in the group.
   */
  basicGroupFullInfo(object_ptr<chatPhoto> &&photo_, string const &description_, int53 creator_user_id_, array<object_ptr<chatMember>> &&members_, bool can_hide_members_, bool can_toggle_aggressive_anti_spam_, object_ptr<chatInviteLink> &&invite_link_, array<object_ptr<botCommands>> &&bot_commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1879035520;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a birthdate of a user.
 */
class birthdate final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Day of the month; 1-31.
  int32 day_;
  /// Month of the year; 1-12.
  int32 month_;
  /// Birth year; 0 if unknown.
  int32 year_;

  /**
   * Represents a birthdate of a user.
   */
  birthdate();

  /**
   * Represents a birthdate of a user.
   *
   * \param[in] day_ Day of the month; 1-31.
   * \param[in] month_ Month of the year; 1-12.
   * \param[in] year_ Birth year; 0 if unknown.
   */
  birthdate(int32 day_, int32 month_, int32 year_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1644064030;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes type of block list.
 */
class BlockList: public Object {
 public:
};

/**
 * The main block list that disallows writing messages to the current user, receiving their status and photo, viewing of stories, and some other actions.
 */
class blockListMain final : public BlockList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The main block list that disallows writing messages to the current user, receiving their status and photo, viewing of stories, and some other actions.
   */
  blockListMain();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1352930172;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The block list that disallows viewing of stories of the current user.
 */
class blockListStories final : public BlockList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The block list that disallows viewing of stories of the current user.
   */
  blockListStories();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 103323228;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a command supported by a bot.
 */
class botCommand final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text of the bot command.
  string command_;
  /// Description of the bot command.
  string description_;

  /**
   * Represents a command supported by a bot.
   */
  botCommand();

  /**
   * Represents a command supported by a bot.
   *
   * \param[in] command_ Text of the bot command.
   * \param[in] description_ Description of the bot command.
   */
  botCommand(string const &command_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032140601;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the scope to which bot commands are relevant.
 */
class BotCommandScope: public Object {
 public:
};

/**
 * A scope covering all users.
 */
class botCommandScopeDefault final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A scope covering all users.
   */
  botCommandScopeDefault();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 795652779;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all private chats.
 */
class botCommandScopeAllPrivateChats final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A scope covering all private chats.
   */
  botCommandScopeAllPrivateChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -344889543;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all group and supergroup chats.
 */
class botCommandScopeAllGroupChats final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A scope covering all group and supergroup chats.
   */
  botCommandScopeAllGroupChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -981088162;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all group and supergroup chat administrators.
 */
class botCommandScopeAllChatAdministrators final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A scope covering all group and supergroup chat administrators.
   */
  botCommandScopeAllChatAdministrators();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1998329169;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all members of a chat.
 */
class botCommandScopeChat final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier.
  int53 chat_id_;

  /**
   * A scope covering all members of a chat.
   */
  botCommandScopeChat();

  /**
   * A scope covering all members of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit botCommandScopeChat(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -430234971;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering all administrators of a chat.
 */
class botCommandScopeChatAdministrators final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier.
  int53 chat_id_;

  /**
   * A scope covering all administrators of a chat.
   */
  botCommandScopeChatAdministrators();

  /**
   * A scope covering all administrators of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit botCommandScopeChatAdministrators(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119682126;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A scope covering a member of a chat.
 */
class botCommandScopeChatMember final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier.
  int53 chat_id_;
  /// User identifier.
  int53 user_id_;

  /**
   * A scope covering a member of a chat.
   */
  botCommandScopeChatMember();

  /**
   * A scope covering a member of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] user_id_ User identifier.
   */
  botCommandScopeChatMember(int53 chat_id_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -211380494;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class botCommand;

/**
 * Contains a list of bot commands.
 */
class botCommands final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Bot's user identifier.
  int53 bot_user_id_;
  /// List of bot commands.
  array<object_ptr<botCommand>> commands_;

  /**
   * Contains a list of bot commands.
   */
  botCommands();

  /**
   * Contains a list of bot commands.
   *
   * \param[in] bot_user_id_ Bot's user identifier.
   * \param[in] commands_ List of bot commands.
   */
  botCommands(int53 bot_user_id_, array<object_ptr<botCommand>> &&commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1741364468;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InternalLinkType;

class affiliateProgramInfo;

class animation;

class botCommand;

class botMenuButton;

class chatAdministratorRights;

class photo;

/**
 * Contains information about a bot.
 */
class botInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The text that is shown on the bot's profile page and is sent together with the link when users share the bot.
  string short_description_;
  /// The text shown in the chat with the bot if the chat is empty.
  string description_;
  /// Photo shown in the chat with the bot if the chat is empty; may be null.
  object_ptr<photo> photo_;
  /// Animation shown in the chat with the bot if the chat is empty; may be null.
  object_ptr<animation> animation_;
  /// Information about a button to show instead of the bot commands menu button; may be null if ordinary bot commands menu must be shown.
  object_ptr<botMenuButton> menu_button_;
  /// List of the bot commands.
  array<object_ptr<botCommand>> commands_;
  /// The HTTP link to the privacy policy of the bot. If empty, then /privacy command must be used if supported by the bot. If the command isn't supported, then https://telegram.org/privacy-tpa must be opened.
  string privacy_policy_url_;
  /// Default administrator rights for adding the bot to basic group and supergroup chats; may be null.
  object_ptr<chatAdministratorRights> default_group_administrator_rights_;
  /// Default administrator rights for adding the bot to channels; may be null.
  object_ptr<chatAdministratorRights> default_channel_administrator_rights_;
  /// Information about the affiliate program of the bot; may be null if none.
  object_ptr<affiliateProgramInfo> affiliate_program_;
  /// Default light background color for bot Web Apps; -1 if not specified.
  int32 web_app_background_light_color_;
  /// Default dark background color for bot Web Apps; -1 if not specified.
  int32 web_app_background_dark_color_;
  /// Default light header color for bot Web Apps; -1 if not specified.
  int32 web_app_header_light_color_;
  /// Default dark header color for bot Web Apps; -1 if not specified.
  int32 web_app_header_dark_color_;
  /// True, if the bot's revenue statistics are available to the current user.
  bool can_get_revenue_statistics_;
  /// True, if the bot can manage emoji status of the current user.
  bool can_manage_emoji_status_;
  /// True, if the bot has media previews.
  bool has_media_previews_;
  /// The internal link, which can be used to edit bot commands; may be null.
  object_ptr<InternalLinkType> edit_commands_link_;
  /// The internal link, which can be used to edit bot description; may be null.
  object_ptr<InternalLinkType> edit_description_link_;
  /// The internal link, which can be used to edit the photo or animation shown in the chat with the bot if the chat is empty; may be null.
  object_ptr<InternalLinkType> edit_description_media_link_;
  /// The internal link, which can be used to edit bot settings; may be null.
  object_ptr<InternalLinkType> edit_settings_link_;

  /**
   * Contains information about a bot.
   */
  botInfo();

  /**
   * Contains information about a bot.
   *
   * \param[in] short_description_ The text that is shown on the bot's profile page and is sent together with the link when users share the bot.
   * \param[in] description_ The text shown in the chat with the bot if the chat is empty.
   * \param[in] photo_ Photo shown in the chat with the bot if the chat is empty; may be null.
   * \param[in] animation_ Animation shown in the chat with the bot if the chat is empty; may be null.
   * \param[in] menu_button_ Information about a button to show instead of the bot commands menu button; may be null if ordinary bot commands menu must be shown.
   * \param[in] commands_ List of the bot commands.
   * \param[in] privacy_policy_url_ The HTTP link to the privacy policy of the bot. If empty, then /privacy command must be used if supported by the bot. If the command isn't supported, then https://telegram.org/privacy-tpa must be opened.
   * \param[in] default_group_administrator_rights_ Default administrator rights for adding the bot to basic group and supergroup chats; may be null.
   * \param[in] default_channel_administrator_rights_ Default administrator rights for adding the bot to channels; may be null.
   * \param[in] affiliate_program_ Information about the affiliate program of the bot; may be null if none.
   * \param[in] web_app_background_light_color_ Default light background color for bot Web Apps; -1 if not specified.
   * \param[in] web_app_background_dark_color_ Default dark background color for bot Web Apps; -1 if not specified.
   * \param[in] web_app_header_light_color_ Default light header color for bot Web Apps; -1 if not specified.
   * \param[in] web_app_header_dark_color_ Default dark header color for bot Web Apps; -1 if not specified.
   * \param[in] can_get_revenue_statistics_ True, if the bot's revenue statistics are available to the current user.
   * \param[in] can_manage_emoji_status_ True, if the bot can manage emoji status of the current user.
   * \param[in] has_media_previews_ True, if the bot has media previews.
   * \param[in] edit_commands_link_ The internal link, which can be used to edit bot commands; may be null.
   * \param[in] edit_description_link_ The internal link, which can be used to edit bot description; may be null.
   * \param[in] edit_description_media_link_ The internal link, which can be used to edit the photo or animation shown in the chat with the bot if the chat is empty; may be null.
   * \param[in] edit_settings_link_ The internal link, which can be used to edit bot settings; may be null.
   */
  botInfo(string const &short_description_, string const &description_, object_ptr<photo> &&photo_, object_ptr<animation> &&animation_, object_ptr<botMenuButton> &&menu_button_, array<object_ptr<botCommand>> &&commands_, string const &privacy_policy_url_, object_ptr<chatAdministratorRights> &&default_group_administrator_rights_, object_ptr<chatAdministratorRights> &&default_channel_administrator_rights_, object_ptr<affiliateProgramInfo> &&affiliate_program_, int32 web_app_background_light_color_, int32 web_app_background_dark_color_, int32 web_app_header_light_color_, int32 web_app_header_dark_color_, bool can_get_revenue_statistics_, bool can_manage_emoji_status_, bool has_media_previews_, object_ptr<InternalLinkType> &&edit_commands_link_, object_ptr<InternalLinkType> &&edit_description_link_, object_ptr<InternalLinkType> &&edit_description_media_link_, object_ptr<InternalLinkType> &&edit_settings_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 759416187;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class StoryContent;

/**
 * Describes media previews of a bot.
 */
class botMediaPreview final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the preview was added or changed last time.
  int32 date_;
  /// Content of the preview.
  object_ptr<StoryContent> content_;

  /**
   * Describes media previews of a bot.
   */
  botMediaPreview();

  /**
   * Describes media previews of a bot.
   *
   * \param[in] date_ Point in time (Unix timestamp) when the preview was added or changed last time.
   * \param[in] content_ Content of the preview.
   */
  botMediaPreview(int32 date_, object_ptr<StoryContent> &&content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1632264984;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class botMediaPreview;

/**
 * Contains a list of media previews of a bot for the given language and the list of languages for which the bot has dedicated previews.
 */
class botMediaPreviewInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of media previews.
  array<object_ptr<botMediaPreview>> previews_;
  /// List of language codes for which the bot has dedicated previews.
  array<string> language_codes_;

  /**
   * Contains a list of media previews of a bot for the given language and the list of languages for which the bot has dedicated previews.
   */
  botMediaPreviewInfo();

  /**
   * Contains a list of media previews of a bot for the given language and the list of languages for which the bot has dedicated previews.
   *
   * \param[in] previews_ List of media previews.
   * \param[in] language_codes_ List of language codes for which the bot has dedicated previews.
   */
  botMediaPreviewInfo(array<object_ptr<botMediaPreview>> &&previews_, array<string> &&language_codes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -284783012;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class botMediaPreview;

/**
 * Contains a list of media previews of a bot.
 */
class botMediaPreviews final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of media previews.
  array<object_ptr<botMediaPreview>> previews_;

  /**
   * Contains a list of media previews of a bot.
   */
  botMediaPreviews();

  /**
   * Contains a list of media previews of a bot.
   *
   * \param[in] previews_ List of media previews.
   */
  explicit botMediaPreviews(array<object_ptr<botMediaPreview>> &&previews_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1787720586;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a button to be shown instead of bot commands menu button.
 */
class botMenuButton final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text of the button.
  string text_;
  /// URL of a Web App to open when the button is pressed. If the link is of the type internalLinkTypeWebApp, then it must be processed accordingly. Otherwise, the link must be passed to openWebApp.
  string url_;

  /**
   * Describes a button to be shown instead of bot commands menu button.
   */
  botMenuButton();

  /**
   * Describes a button to be shown instead of bot commands menu button.
   *
   * \param[in] text_ Text of the button.
   * \param[in] url_ URL of a Web App to open when the button is pressed. If the link is of the type internalLinkTypeWebApp, then it must be processed accordingly. Otherwise, the link must be passed to openWebApp.
   */
  botMenuButton(string const &text_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -944407322;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class webApp;

/**
 * This class is an abstract base class.
 * Describes a reason why a bot was allowed to write messages to the current user.
 */
class BotWriteAccessAllowReason: public Object {
 public:
};

/**
 * The user connected a website by logging in using Telegram Login Widget on it.
 */
class botWriteAccessAllowReasonConnectedWebsite final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Domain name of the connected website.
  string domain_name_;

  /**
   * The user connected a website by logging in using Telegram Login Widget on it.
   */
  botWriteAccessAllowReasonConnectedWebsite();

  /**
   * The user connected a website by logging in using Telegram Login Widget on it.
   *
   * \param[in] domain_name_ Domain name of the connected website.
   */
  explicit botWriteAccessAllowReasonConnectedWebsite(string const &domain_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2016325603;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user added the bot to attachment or side menu using toggleBotIsAddedToAttachmentMenu.
 */
class botWriteAccessAllowReasonAddedToAttachmentMenu final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user added the bot to attachment or side menu using toggleBotIsAddedToAttachmentMenu.
   */
  botWriteAccessAllowReasonAddedToAttachmentMenu();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2104795235;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user launched a Web App using getWebAppLinkUrl.
 */
class botWriteAccessAllowReasonLaunchedWebApp final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about the Web App.
  object_ptr<webApp> web_app_;

  /**
   * The user launched a Web App using getWebAppLinkUrl.
   */
  botWriteAccessAllowReasonLaunchedWebApp();

  /**
   * The user launched a Web App using getWebAppLinkUrl.
   *
   * \param[in] web_app_ Information about the Web App.
   */
  explicit botWriteAccessAllowReasonLaunchedWebApp(object_ptr<webApp> &&web_app_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -240843561;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user accepted bot's request to send messages with allowBotToSendMessages.
 */
class botWriteAccessAllowReasonAcceptedRequest final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user accepted bot's request to send messages with allowBotToSendMessages.
   */
  botWriteAccessAllowReasonAcceptedRequest();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1983497220;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes conditions for sending of away messages by a Telegram Business account.
 */
class BusinessAwayMessageSchedule: public Object {
 public:
};

/**
 * Send away messages always.
 */
class businessAwayMessageScheduleAlways final : public BusinessAwayMessageSchedule {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Send away messages always.
   */
  businessAwayMessageScheduleAlways();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -910564679;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Send away messages outside of the business opening hours.
 */
class businessAwayMessageScheduleOutsideOfOpeningHours final : public BusinessAwayMessageSchedule {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Send away messages outside of the business opening hours.
   */
  businessAwayMessageScheduleOutsideOfOpeningHours();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -968630506;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Send away messages only in the specified time span.
 */
class businessAwayMessageScheduleCustom final : public BusinessAwayMessageSchedule {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the away messages will start to be sent.
  int32 start_date_;
  /// Point in time (Unix timestamp) when the away messages will stop to be sent.
  int32 end_date_;

  /**
   * Send away messages only in the specified time span.
   */
  businessAwayMessageScheduleCustom();

  /**
   * Send away messages only in the specified time span.
   *
   * \param[in] start_date_ Point in time (Unix timestamp) when the away messages will start to be sent.
   * \param[in] end_date_ Point in time (Unix timestamp) when the away messages will stop to be sent.
   */
  businessAwayMessageScheduleCustom(int32 start_date_, int32 end_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1967108654;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class BusinessAwayMessageSchedule;

class businessRecipients;

/**
 * Describes settings for messages that are automatically sent by a Telegram Business account when it is away.
 */
class businessAwayMessageSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique quick reply shortcut identifier for the away messages.
  int32 shortcut_id_;
  /// Chosen recipients of the away messages.
  object_ptr<businessRecipients> recipients_;
  /// Settings used to check whether the current user is away.
  object_ptr<BusinessAwayMessageSchedule> schedule_;
  /// True, if the messages must not be sent if the account was online in the last 10 minutes.
  bool offline_only_;

  /**
   * Describes settings for messages that are automatically sent by a Telegram Business account when it is away.
   */
  businessAwayMessageSettings();

  /**
   * Describes settings for messages that are automatically sent by a Telegram Business account when it is away.
   *
   * \param[in] shortcut_id_ Unique quick reply shortcut identifier for the away messages.
   * \param[in] recipients_ Chosen recipients of the away messages.
   * \param[in] schedule_ Settings used to check whether the current user is away.
   * \param[in] offline_only_ True, if the messages must not be sent if the account was online in the last 10 minutes.
   */
  businessAwayMessageSettings(int32 shortcut_id_, object_ptr<businessRecipients> &&recipients_, object_ptr<BusinessAwayMessageSchedule> &&schedule_, bool offline_only_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 353084137;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a business bot that manages the chat.
 */
class businessBotManageBar final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the bot.
  int53 bot_user_id_;
  /// URL to be opened to manage the bot.
  string manage_url_;
  /// True, if the bot is paused. Use toggleBusinessConnectedBotChatIsPaused to change the value of the field.
  bool is_bot_paused_;
  /// True, if the bot can reply.
  bool can_bot_reply_;

  /**
   * Contains information about a business bot that manages the chat.
   */
  businessBotManageBar();

  /**
   * Contains information about a business bot that manages the chat.
   *
   * \param[in] bot_user_id_ User identifier of the bot.
   * \param[in] manage_url_ URL to be opened to manage the bot.
   * \param[in] is_bot_paused_ True, if the bot is paused. Use toggleBusinessConnectedBotChatIsPaused to change the value of the field.
   * \param[in] can_bot_reply_ True, if the bot can reply.
   */
  businessBotManageBar(int53 bot_user_id_, string const &manage_url_, bool is_bot_paused_, bool can_bot_reply_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -311399806;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class formattedText;

/**
 * Contains information about a business chat link.
 */
class businessChatLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The HTTPS link.
  string link_;
  /// Message draft text that will be added to the input field.
  object_ptr<formattedText> text_;
  /// Link title.
  string title_;
  /// Number of times the link was used.
  int32 view_count_;

  /**
   * Contains information about a business chat link.
   */
  businessChatLink();

  /**
   * Contains information about a business chat link.
   *
   * \param[in] link_ The HTTPS link.
   * \param[in] text_ Message draft text that will be added to the input field.
   * \param[in] title_ Link title.
   * \param[in] view_count_ Number of times the link was used.
   */
  businessChatLink(string const &link_, object_ptr<formattedText> &&text_, string const &title_, int32 view_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1902539901;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class formattedText;

/**
 * Contains information about a business chat link.
 */
class businessChatLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the private chat that created the link.
  int53 chat_id_;
  /// Message draft text that must be added to the input field.
  object_ptr<formattedText> text_;

  /**
   * Contains information about a business chat link.
   */
  businessChatLinkInfo();

  /**
   * Contains information about a business chat link.
   *
   * \param[in] chat_id_ Identifier of the private chat that created the link.
   * \param[in] text_ Message draft text that must be added to the input field.
   */
  businessChatLinkInfo(int53 chat_id_, object_ptr<formattedText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -864865105;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class businessChatLink;

/**
 * Contains a list of business chat links created by the user.
 */
class businessChatLinks final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of links.
  array<object_ptr<businessChatLink>> links_;

  /**
   * Contains a list of business chat links created by the user.
   */
  businessChatLinks();

  /**
   * Contains a list of business chat links created by the user.
   *
   * \param[in] links_ List of links.
   */
  explicit businessChatLinks(array<object_ptr<businessChatLink>> &&links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 79067036;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class businessRecipients;

/**
 * Describes a bot connected to a business account.
 */
class businessConnectedBot final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the bot.
  int53 bot_user_id_;
  /// Private chats that will be accessible to the bot.
  object_ptr<businessRecipients> recipients_;
  /// True, if the bot can send messages to the private chats; false otherwise.
  bool can_reply_;

  /**
   * Describes a bot connected to a business account.
   */
  businessConnectedBot();

  /**
   * Describes a bot connected to a business account.
   *
   * \param[in] bot_user_id_ User identifier of the bot.
   * \param[in] recipients_ Private chats that will be accessible to the bot.
   * \param[in] can_reply_ True, if the bot can send messages to the private chats; false otherwise.
   */
  businessConnectedBot(int53 bot_user_id_, object_ptr<businessRecipients> &&recipients_, bool can_reply_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -330241321;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a connection of the bot with a business account.
 */
class businessConnection final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the connection.
  string id_;
  /// Identifier of the business user that created the connection.
  int53 user_id_;
  /// Chat identifier of the private chat with the user.
  int53 user_chat_id_;
  /// Point in time (Unix timestamp) when the connection was established.
  int32 date_;
  /// True, if the bot can send messages to the connected user; false otherwise.
  bool can_reply_;
  /// True, if the connection is enabled; false otherwise.
  bool is_enabled_;

  /**
   * Describes a connection of the bot with a business account.
   */
  businessConnection();

  /**
   * Describes a connection of the bot with a business account.
   *
   * \param[in] id_ Unique identifier of the connection.
   * \param[in] user_id_ Identifier of the business user that created the connection.
   * \param[in] user_chat_id_ Chat identifier of the private chat with the user.
   * \param[in] date_ Point in time (Unix timestamp) when the connection was established.
   * \param[in] can_reply_ True, if the bot can send messages to the connected user; false otherwise.
   * \param[in] is_enabled_ True, if the connection is enabled; false otherwise.
   */
  businessConnection(string const &id_, int53 user_id_, int53 user_chat_id_, int32 date_, bool can_reply_, bool is_enabled_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1144447540;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a feature available to Business user accounts.
 */
class BusinessFeature: public Object {
 public:
};

/**
 * The ability to set location.
 */
class businessFeatureLocation final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to set location.
   */
  businessFeatureLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1064304004;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to set opening hours.
 */
class businessFeatureOpeningHours final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to set opening hours.
   */
  businessFeatureOpeningHours();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 461054701;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to use quick replies.
 */
class businessFeatureQuickReplies final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to use quick replies.
   */
  businessFeatureQuickReplies();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1674048894;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to set up a greeting message.
 */
class businessFeatureGreetingMessage final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to set up a greeting message.
   */
  businessFeatureGreetingMessage();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1789424756;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to set up an away message.
 */
class businessFeatureAwayMessage final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to set up an away message.
   */
  businessFeatureAwayMessage();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1090119901;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to create links to the business account with predefined message text.
 */
class businessFeatureAccountLinks final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to create links to the business account with predefined message text.
   */
  businessFeatureAccountLinks();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1878693646;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to customize start page.
 */
class businessFeatureStartPage final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to customize start page.
   */
  businessFeatureStartPage();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 401471457;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to connect a bot to the account.
 */
class businessFeatureBots final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to connect a bot to the account.
   */
  businessFeatureBots();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 275084773;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to show an emoji status along with the business name.
 */
class businessFeatureEmojiStatus final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to show an emoji status along with the business name.
   */
  businessFeatureEmojiStatus();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -846282523;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The ability to display folder names for each chat in the chat list.
 */
class businessFeatureChatFolderTags final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The ability to display folder names for each chat in the chat list.
   */
  businessFeatureChatFolderTags();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -543880918;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Allowed to use many additional features for stories.
 */
class businessFeatureUpgradedStories final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Allowed to use many additional features for stories.
   */
  businessFeatureUpgradedStories();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1812245550;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class BusinessFeature;

class animation;

/**
 * Describes a promotion animation for a Business feature.
 */
class businessFeaturePromotionAnimation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Business feature.
  object_ptr<BusinessFeature> feature_;
  /// Promotion animation for the feature.
  object_ptr<animation> animation_;

  /**
   * Describes a promotion animation for a Business feature.
   */
  businessFeaturePromotionAnimation();

  /**
   * Describes a promotion animation for a Business feature.
   *
   * \param[in] feature_ Business feature.
   * \param[in] animation_ Promotion animation for the feature.
   */
  businessFeaturePromotionAnimation(object_ptr<BusinessFeature> &&feature_, object_ptr<animation> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2047174666;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class BusinessFeature;

/**
 * Contains information about features, available to Business user accounts.
 */
class businessFeatures final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The list of available business features.
  array<object_ptr<BusinessFeature>> features_;

  /**
   * Contains information about features, available to Business user accounts.
   */
  businessFeatures();

  /**
   * Contains information about features, available to Business user accounts.
   *
   * \param[in] features_ The list of available business features.
   */
  explicit businessFeatures(array<object_ptr<BusinessFeature>> &&features_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1532468184;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class businessRecipients;

/**
 * Describes settings for greeting messages that are automatically sent by a Telegram Business account as response to incoming messages in an inactive private chat.
 */
class businessGreetingMessageSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique quick reply shortcut identifier for the greeting messages.
  int32 shortcut_id_;
  /// Chosen recipients of the greeting messages.
  object_ptr<businessRecipients> recipients_;
  /// The number of days after which a chat will be considered as inactive; currently, must be on of 7, 14, 21, or 28.
  int32 inactivity_days_;

  /**
   * Describes settings for greeting messages that are automatically sent by a Telegram Business account as response to incoming messages in an inactive private chat.
   */
  businessGreetingMessageSettings();

  /**
   * Describes settings for greeting messages that are automatically sent by a Telegram Business account as response to incoming messages in an inactive private chat.
   *
   * \param[in] shortcut_id_ Unique quick reply shortcut identifier for the greeting messages.
   * \param[in] recipients_ Chosen recipients of the greeting messages.
   * \param[in] inactivity_days_ The number of days after which a chat will be considered as inactive; currently, must be on of 7, 14, 21, or 28.
   */
  businessGreetingMessageSettings(int32 shortcut_id_, object_ptr<businessRecipients> &&recipients_, int32 inactivity_days_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1689140754;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class businessAwayMessageSettings;

class businessGreetingMessageSettings;

class businessLocation;

class businessOpeningHours;

class businessStartPage;

/**
 * Contains information about a Telegram Business account.
 */
class businessInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Location of the business; may be null if none.
  object_ptr<businessLocation> location_;
  /// Opening hours of the business; may be null if none. The hours are guaranteed to be valid and has already been split by week days.
  object_ptr<businessOpeningHours> opening_hours_;
  /// Opening hours of the business in the local time; may be null if none. The hours are guaranteed to be valid and has already been split by week days. Local time zone identifier will be empty. An updateUserFullInfo update is not triggered when value of this field changes.
  object_ptr<businessOpeningHours> local_opening_hours_;
  /// Time left before the business will open the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
  int32 next_open_in_;
  /// Time left before the business will close the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
  int32 next_close_in_;
  /// The greeting message; may be null if none or the Business account is not of the current user.
  object_ptr<businessGreetingMessageSettings> greeting_message_settings_;
  /// The away message; may be null if none or the Business account is not of the current user.
  object_ptr<businessAwayMessageSettings> away_message_settings_;
  /// Information about start page of the account; may be null if none.
  object_ptr<businessStartPage> start_page_;

  /**
   * Contains information about a Telegram Business account.
   */
  businessInfo();

  /**
   * Contains information about a Telegram Business account.
   *
   * \param[in] location_ Location of the business; may be null if none.
   * \param[in] opening_hours_ Opening hours of the business; may be null if none. The hours are guaranteed to be valid and has already been split by week days.
   * \param[in] local_opening_hours_ Opening hours of the business in the local time; may be null if none. The hours are guaranteed to be valid and has already been split by week days. Local time zone identifier will be empty. An updateUserFullInfo update is not triggered when value of this field changes.
   * \param[in] next_open_in_ Time left before the business will open the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
   * \param[in] next_close_in_ Time left before the business will close the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
   * \param[in] greeting_message_settings_ The greeting message; may be null if none or the Business account is not of the current user.
   * \param[in] away_message_settings_ The away message; may be null if none or the Business account is not of the current user.
   * \param[in] start_page_ Information about start page of the account; may be null if none.
   */
  businessInfo(object_ptr<businessLocation> &&location_, object_ptr<businessOpeningHours> &&opening_hours_, object_ptr<businessOpeningHours> &&local_opening_hours_, int32 next_open_in_, int32 next_close_in_, object_ptr<businessGreetingMessageSettings> &&greeting_message_settings_, object_ptr<businessAwayMessageSettings> &&away_message_settings_, object_ptr<businessStartPage> &&start_page_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1428179342;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class location;

/**
 * Represents a location of a business.
 */
class businessLocation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The location; may be null if not specified.
  object_ptr<location> location_;
  /// Location address; 1-96 characters.
  string address_;

  /**
   * Represents a location of a business.
   */
  businessLocation();

  /**
   * Represents a location of a business.
   *
   * \param[in] location_ The location; may be null if not specified.
   * \param[in] address_ Location address; 1-96 characters.
   */
  businessLocation(object_ptr<location> &&location_, string const &address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1084969126;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class message;

/**
 * Describes a message from a business account as received by a bot.
 */
class businessMessage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The message.
  object_ptr<message> message_;
  /// Message that is replied by the message in the same chat; may be null if none.
  object_ptr<message> reply_to_message_;

  /**
   * Describes a message from a business account as received by a bot.
   */
  businessMessage();

  /**
   * Describes a message from a business account as received by a bot.
   *
   * \param[in] message_ The message.
   * \param[in] reply_to_message_ Message that is replied by the message in the same chat; may be null if none.
   */
  businessMessage(object_ptr<message> &&message_, object_ptr<message> &&reply_to_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -94353850;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class businessMessage;

/**
 * Contains a list of messages from a business account as received by a bot.
 */
class businessMessages final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of business messages.
  array<object_ptr<businessMessage>> messages_;

  /**
   * Contains a list of messages from a business account as received by a bot.
   */
  businessMessages();

  /**
   * Contains a list of messages from a business account as received by a bot.
   *
   * \param[in] messages_ List of business messages.
   */
  explicit businessMessages(array<object_ptr<businessMessage>> &&messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -764562473;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class businessOpeningHoursInterval;

/**
 * Describes opening hours of a business.
 */
class businessOpeningHours final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique time zone identifier.
  string time_zone_id_;
  /// Intervals of the time when the business is open.
  array<object_ptr<businessOpeningHoursInterval>> opening_hours_;

  /**
   * Describes opening hours of a business.
   */
  businessOpeningHours();

  /**
   * Describes opening hours of a business.
   *
   * \param[in] time_zone_id_ Unique time zone identifier.
   * \param[in] opening_hours_ Intervals of the time when the business is open.
   */
  businessOpeningHours(string const &time_zone_id_, array<object_ptr<businessOpeningHoursInterval>> &&opening_hours_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 816603700;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an interval of time when the business is open.
 */
class businessOpeningHoursInterval final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The minute's sequence number in a week, starting on Monday, marking the start of the time interval during which the business is open; 0-7*24*60.
  int32 start_minute_;
  /// The minute's sequence number in a week, starting on Monday, marking the end of the time interval during which the business is open; 1-8*24*60.
  int32 end_minute_;

  /**
   * Describes an interval of time when the business is open.
   */
  businessOpeningHoursInterval();

  /**
   * Describes an interval of time when the business is open.
   *
   * \param[in] start_minute_ The minute's sequence number in a week, starting on Monday, marking the start of the time interval during which the business is open; 0-7*24*60.
   * \param[in] end_minute_ The minute's sequence number in a week, starting on Monday, marking the end of the time interval during which the business is open; 1-8*24*60.
   */
  businessOpeningHoursInterval(int32 start_minute_, int32 end_minute_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1108322732;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes private chats chosen for automatic interaction with a business.
 */
class businessRecipients final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifiers of selected private chats.
  array<int53> chat_ids_;
  /// Identifiers of private chats that are always excluded; for businessConnectedBot only.
  array<int53> excluded_chat_ids_;
  /// True, if all existing private chats are selected.
  bool select_existing_chats_;
  /// True, if all new private chats are selected.
  bool select_new_chats_;
  /// True, if all private chats with contacts are selected.
  bool select_contacts_;
  /// True, if all private chats with non-contacts are selected.
  bool select_non_contacts_;
  /// If true, then all private chats except the selected are chosen. Otherwise, only the selected chats are chosen.
  bool exclude_selected_;

  /**
   * Describes private chats chosen for automatic interaction with a business.
   */
  businessRecipients();

  /**
   * Describes private chats chosen for automatic interaction with a business.
   *
   * \param[in] chat_ids_ Identifiers of selected private chats.
   * \param[in] excluded_chat_ids_ Identifiers of private chats that are always excluded; for businessConnectedBot only.
   * \param[in] select_existing_chats_ True, if all existing private chats are selected.
   * \param[in] select_new_chats_ True, if all new private chats are selected.
   * \param[in] select_contacts_ True, if all private chats with contacts are selected.
   * \param[in] select_non_contacts_ True, if all private chats with non-contacts are selected.
   * \param[in] exclude_selected_ If true, then all private chats except the selected are chosen. Otherwise, only the selected chats are chosen.
   */
  businessRecipients(array<int53> &&chat_ids_, array<int53> &&excluded_chat_ids_, bool select_existing_chats_, bool select_new_chats_, bool select_contacts_, bool select_non_contacts_, bool exclude_selected_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 868656909;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class sticker;

/**
 * Describes settings for a business account start page.
 */
class businessStartPage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Title text of the start page.
  string title_;
  /// Message text of the start page.
  string message_;
  /// Greeting sticker of the start page; may be null if none.
  object_ptr<sticker> sticker_;

  /**
   * Describes settings for a business account start page.
   */
  businessStartPage();

  /**
   * Describes settings for a business account start page.
   *
   * \param[in] title_ Title text of the start page.
   * \param[in] message_ Message text of the start page.
   * \param[in] sticker_ Greeting sticker of the start page; may be null if none.
   */
  businessStartPage(string const &title_, string const &message_, object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1616709681;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class CallState;

/**
 * Describes a call.
 */
class call final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Call identifier, not persistent.
  int32 id_;
  /// User identifier of the other call participant.
  int53 user_id_;
  /// True, if the call is outgoing.
  bool is_outgoing_;
  /// True, if the call is a video call.
  bool is_video_;
  /// Call state.
  object_ptr<CallState> state_;

  /**
   * Describes a call.
   */
  call();

  /**
   * Describes a call.
   *
   * \param[in] id_ Call identifier, not persistent.
   * \param[in] user_id_ User identifier of the other call participant.
   * \param[in] is_outgoing_ True, if the call is outgoing.
   * \param[in] is_video_ True, if the call is a video call.
   * \param[in] state_ Call state.
   */
  call(int32 id_, int53 user_id_, bool is_outgoing_, bool is_video_, object_ptr<CallState> &&state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 920360804;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the reason why a call was discarded.
 */
class CallDiscardReason: public Object {
 public:
};

/**
 * The call wasn't discarded, or the reason is unknown.
 */
class callDiscardReasonEmpty final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call wasn't discarded, or the reason is unknown.
   */
  callDiscardReasonEmpty();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1258917949;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended before the conversation started. It was canceled by the caller or missed by the other party.
 */
class callDiscardReasonMissed final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call was ended before the conversation started. It was canceled by the caller or missed by the other party.
   */
  callDiscardReasonMissed();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1680358012;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended before the conversation started. It was declined by the other party.
 */
class callDiscardReasonDeclined final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call was ended before the conversation started. It was declined by the other party.
   */
  callDiscardReasonDeclined();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1729926094;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended during the conversation because the users were disconnected.
 */
class callDiscardReasonDisconnected final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call was ended during the conversation because the users were disconnected.
   */
  callDiscardReasonDisconnected();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1342872670;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call was ended because one of the parties hung up.
 */
class callDiscardReasonHungUp final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call was ended because one of the parties hung up.
   */
  callDiscardReasonHungUp();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 438216166;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the call identifier.
 */
class callId final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Call identifier.
  int32 id_;

  /**
   * Contains the call identifier.
   */
  callId();

  /**
   * Contains the call identifier.
   *
   * \param[in] id_ Call identifier.
   */
  explicit callId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 65717769;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the exact type of problem with a call.
 */
class CallProblem: public Object {
 public:
};

/**
 * The user heard their own voice.
 */
class callProblemEcho final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user heard their own voice.
   */
  callProblemEcho();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 801116548;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user heard background noise.
 */
class callProblemNoise final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user heard background noise.
   */
  callProblemNoise();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1053065359;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The other side kept disappearing.
 */
class callProblemInterruptions final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The other side kept disappearing.
   */
  callProblemInterruptions();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119493218;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The speech was distorted.
 */
class callProblemDistortedSpeech final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The speech was distorted.
   */
  callProblemDistortedSpeech();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 379960581;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user couldn't hear the other side.
 */
class callProblemSilentLocal final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user couldn't hear the other side.
   */
  callProblemSilentLocal();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 253652790;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The other side couldn't hear the user.
 */
class callProblemSilentRemote final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The other side couldn't hear the user.
   */
  callProblemSilentRemote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 573634714;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call ended unexpectedly.
 */
class callProblemDropped final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call ended unexpectedly.
   */
  callProblemDropped();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1207311487;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The video was distorted.
 */
class callProblemDistortedVideo final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The video was distorted.
   */
  callProblemDistortedVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 385245706;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The video was pixelated.
 */
class callProblemPixelatedVideo final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The video was pixelated.
   */
  callProblemPixelatedVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2115315411;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Specifies the supported call protocols.
 */
class callProtocol final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if UDP peer-to-peer connections are supported.
  bool udp_p2p_;
  /// True, if connection through UDP reflectors is supported.
  bool udp_reflector_;
  /// The minimum supported API layer; use 65.
  int32 min_layer_;
  /// The maximum supported API layer; use 92.
  int32 max_layer_;
  /// List of supported tgcalls versions.
  array<string> library_versions_;

  /**
   * Specifies the supported call protocols.
   */
  callProtocol();

  /**
   * Specifies the supported call protocols.
   *
   * \param[in] udp_p2p_ True, if UDP peer-to-peer connections are supported.
   * \param[in] udp_reflector_ True, if connection through UDP reflectors is supported.
   * \param[in] min_layer_ The minimum supported API layer; use 65.
   * \param[in] max_layer_ The maximum supported API layer; use 92.
   * \param[in] library_versions_ List of supported tgcalls versions.
   */
  callProtocol(bool udp_p2p_, bool udp_reflector_, int32 min_layer_, int32 max_layer_, array<string> &&library_versions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1075562897;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class CallServerType;

/**
 * Describes a server for relaying call data.
 */
class callServer final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Server identifier.
  int64 id_;
  /// Server IPv4 address.
  string ip_address_;
  /// Server IPv6 address.
  string ipv6_address_;
  /// Server port number.
  int32 port_;
  /// Server type.
  object_ptr<CallServerType> type_;

  /**
   * Describes a server for relaying call data.
   */
  callServer();

  /**
   * Describes a server for relaying call data.
   *
   * \param[in] id_ Server identifier.
   * \param[in] ip_address_ Server IPv4 address.
   * \param[in] ipv6_address_ Server IPv6 address.
   * \param[in] port_ Server port number.
   * \param[in] type_ Server type.
   */
  callServer(int64 id_, string const &ip_address_, string const &ipv6_address_, int32 port_, object_ptr<CallServerType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1865932695;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of call server.
 */
class CallServerType: public Object {
 public:
};

/**
 * A Telegram call reflector.
 */
class callServerTypeTelegramReflector final : public CallServerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A peer tag to be used with the reflector.
  bytes peer_tag_;
  /// True, if the server uses TCP instead of UDP.
  bool is_tcp_;

  /**
   * A Telegram call reflector.
   */
  callServerTypeTelegramReflector();

  /**
   * A Telegram call reflector.
   *
   * \param[in] peer_tag_ A peer tag to be used with the reflector.
   * \param[in] is_tcp_ True, if the server uses TCP instead of UDP.
   */
  callServerTypeTelegramReflector(bytes const &peer_tag_, bool is_tcp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 850343189;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A WebRTC server.
 */
class callServerTypeWebrtc final : public CallServerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username to be used for authentication.
  string username_;
  /// Authentication password.
  string password_;
  /// True, if the server supports TURN.
  bool supports_turn_;
  /// True, if the server supports STUN.
  bool supports_stun_;

  /**
   * A WebRTC server.
   */
  callServerTypeWebrtc();

  /**
   * A WebRTC server.
   *
   * \param[in] username_ Username to be used for authentication.
   * \param[in] password_ Authentication password.
   * \param[in] supports_turn_ True, if the server supports TURN.
   * \param[in] supports_stun_ True, if the server supports STUN.
   */
  callServerTypeWebrtc(string const &username_, string const &password_, bool supports_turn_, bool supports_stun_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1250622821;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class CallDiscardReason;

class callProtocol;

class callServer;

class error;

/**
 * This class is an abstract base class.
 * Describes the current call state.
 */
class CallState: public Object {
 public:
};

/**
 * The call is pending, waiting to be accepted by a user.
 */
class callStatePending final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if the call has already been created by the server.
  bool is_created_;
  /// True, if the call has already been received by the other party.
  bool is_received_;

  /**
   * The call is pending, waiting to be accepted by a user.
   */
  callStatePending();

  /**
   * The call is pending, waiting to be accepted by a user.
   *
   * \param[in] is_created_ True, if the call has already been created by the server.
   * \param[in] is_received_ True, if the call has already been received by the other party.
   */
  callStatePending(bool is_created_, bool is_received_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1073048620;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call has been answered and encryption keys are being exchanged.
 */
class callStateExchangingKeys final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call has been answered and encryption keys are being exchanged.
   */
  callStateExchangingKeys();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1848149403;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call is ready to use.
 */
class callStateReady final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Call protocols supported by the other call participant.
  object_ptr<callProtocol> protocol_;
  /// List of available call servers.
  array<object_ptr<callServer>> servers_;
  /// A JSON-encoded call config.
  string config_;
  /// Call encryption key.
  bytes encryption_key_;
  /// Encryption key fingerprint represented as 4 emoji.
  array<string> emojis_;
  /// True, if peer-to-peer connection is allowed by users privacy settings.
  bool allow_p2p_;
  /// Custom JSON-encoded call parameters to be passed to tgcalls.
  string custom_parameters_;

  /**
   * The call is ready to use.
   */
  callStateReady();

  /**
   * The call is ready to use.
   *
   * \param[in] protocol_ Call protocols supported by the other call participant.
   * \param[in] servers_ List of available call servers.
   * \param[in] config_ A JSON-encoded call config.
   * \param[in] encryption_key_ Call encryption key.
   * \param[in] emojis_ Encryption key fingerprint represented as 4 emoji.
   * \param[in] allow_p2p_ True, if peer-to-peer connection is allowed by users privacy settings.
   * \param[in] custom_parameters_ Custom JSON-encoded call parameters to be passed to tgcalls.
   */
  callStateReady(object_ptr<callProtocol> &&protocol_, array<object_ptr<callServer>> &&servers_, string const &config_, bytes const &encryption_key_, array<string> &&emojis_, bool allow_p2p_, string const &custom_parameters_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 731619651;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call is hanging up after discardCall has been called.
 */
class callStateHangingUp final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The call is hanging up after discardCall has been called.
   */
  callStateHangingUp();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2133790038;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call has ended successfully.
 */
class callStateDiscarded final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The reason why the call has ended.
  object_ptr<CallDiscardReason> reason_;
  /// True, if the call rating must be sent to the server.
  bool need_rating_;
  /// True, if the call debug information must be sent to the server.
  bool need_debug_information_;
  /// True, if the call log must be sent to the server.
  bool need_log_;

  /**
   * The call has ended successfully.
   */
  callStateDiscarded();

  /**
   * The call has ended successfully.
   *
   * \param[in] reason_ The reason why the call has ended.
   * \param[in] need_rating_ True, if the call rating must be sent to the server.
   * \param[in] need_debug_information_ True, if the call debug information must be sent to the server.
   * \param[in] need_log_ True, if the call log must be sent to the server.
   */
  callStateDiscarded(object_ptr<CallDiscardReason> &&reason_, bool need_rating_, bool need_debug_information_, bool need_log_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1394310213;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The call has ended with an error.
 */
class callStateError final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Error. An error with the code 4005000 will be returned if an outgoing call is missed because of an expired timeout.
  object_ptr<error> error_;

  /**
   * The call has ended with an error.
   */
  callStateError();

  /**
   * The call has ended with an error.
   *
   * \param[in] error_ Error. An error with the code 4005000 will be returned if an outgoing call is missed because of an expired timeout.
   */
  explicit callStateError(object_ptr<error> &&error_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -975215467;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a bot's answer to a callback query.
 */
class callbackQueryAnswer final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text of the answer.
  string text_;
  /// True, if an alert must be shown to the user instead of a toast notification.
  bool show_alert_;
  /// URL to be opened.
  string url_;

  /**
   * Contains a bot's answer to a callback query.
   */
  callbackQueryAnswer();

  /**
   * Contains a bot's answer to a callback query.
   *
   * \param[in] text_ Text of the answer.
   * \param[in] show_alert_ True, if an alert must be shown to the user instead of a toast notification.
   * \param[in] url_ URL to be opened.
   */
  callbackQueryAnswer(string const &text_, bool show_alert_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 360867933;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a payload of a callback query.
 */
class CallbackQueryPayload: public Object {
 public:
};

/**
 * The payload for a general callback button.
 */
class callbackQueryPayloadData final : public CallbackQueryPayload {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Data that was attached to the callback button.
  bytes data_;

  /**
   * The payload for a general callback button.
   */
  callbackQueryPayloadData();

  /**
   * The payload for a general callback button.
   *
   * \param[in] data_ Data that was attached to the callback button.
   */
  explicit callbackQueryPayloadData(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1977729946;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The payload for a callback button requiring password.
 */
class callbackQueryPayloadDataWithPassword final : public CallbackQueryPayload {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The 2-step verification password for the current user.
  string password_;
  /// Data that was attached to the callback button.
  bytes data_;

  /**
   * The payload for a callback button requiring password.
   */
  callbackQueryPayloadDataWithPassword();

  /**
   * The payload for a callback button requiring password.
   *
   * \param[in] password_ The 2-step verification password for the current user.
   * \param[in] data_ Data that was attached to the callback button.
   */
  callbackQueryPayloadDataWithPassword(string const &password_, bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1340266738;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The payload for a game callback button.
 */
class callbackQueryPayloadGame final : public CallbackQueryPayload {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A short name of the game that was attached to the callback button.
  string game_short_name_;

  /**
   * The payload for a game callback button.
   */
  callbackQueryPayloadGame();

  /**
   * The payload for a game callback button.
   *
   * \param[in] game_short_name_ A short name of the game that was attached to the callback button.
   */
  explicit callbackQueryPayloadGame(string const &game_short_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1303571512;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes result of canSendMessageToUser.
 */
class CanSendMessageToUserResult: public Object {
 public:
};

/**
 * The user can be messaged.
 */
class canSendMessageToUserResultOk final : public CanSendMessageToUserResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user can be messaged.
   */
  canSendMessageToUserResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1530583042;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user can't be messaged, because they are deleted or unknown.
 */
class canSendMessageToUserResultUserIsDeleted final : public CanSendMessageToUserResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user can't be messaged, because they are deleted or unknown.
   */
  canSendMessageToUserResultUserIsDeleted();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1944639903;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user can't be messaged, because they restrict new chats with non-contacts.
 */
class canSendMessageToUserResultUserRestrictsNewChats final : public CanSendMessageToUserResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user can't be messaged, because they restrict new chats with non-contacts.
   */
  canSendMessageToUserResultUserRestrictsNewChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1929699797;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether the current user can send a story in the specific chat.
 */
class CanSendStoryResult: public Object {
 public:
};

/**
 * A story can be sent.
 */
class canSendStoryResultOk final : public CanSendStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A story can be sent.
   */
  canSendStoryResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1346171133;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user must subscribe to Telegram Premium to be able to post stories.
 */
class canSendStoryResultPremiumNeeded final : public CanSendStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user must subscribe to Telegram Premium to be able to post stories.
   */
  canSendStoryResultPremiumNeeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1451220585;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat must be boosted first by Telegram Premium subscribers to post more stories. Call getChatBoostStatus to get current boost status of the chat.
 */
class canSendStoryResultBoostNeeded final : public CanSendStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The chat must be boosted first by Telegram Premium subscribers to post more stories. Call getChatBoostStatus to get current boost status of the chat.
   */
  canSendStoryResultBoostNeeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1637816017;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The limit for the number of active stories exceeded. The user can buy Telegram Premium, delete an active story, or wait for the oldest story to expire.
 */
class canSendStoryResultActiveStoryLimitExceeded final : public CanSendStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The limit for the number of active stories exceeded. The user can buy Telegram Premium, delete an active story, or wait for the oldest story to expire.
   */
  canSendStoryResultActiveStoryLimitExceeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1344689450;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The weekly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
 */
class canSendStoryResultWeeklyLimitExceeded final : public CanSendStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Time left before the user can send the next story.
  int32 retry_after_;

  /**
   * The weekly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   */
  canSendStoryResultWeeklyLimitExceeded();

  /**
   * The weekly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   *
   * \param[in] retry_after_ Time left before the user can send the next story.
   */
  explicit canSendStoryResultWeeklyLimitExceeded(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 323068088;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The monthly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
 */
class canSendStoryResultMonthlyLimitExceeded final : public CanSendStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Time left before the user can send the next story.
  int32 retry_after_;

  /**
   * The monthly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   */
  canSendStoryResultMonthlyLimitExceeded();

  /**
   * The monthly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   *
   * \param[in] retry_after_ Time left before the user can send the next story.
   */
  explicit canSendStoryResultMonthlyLimitExceeded(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -578665771;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether the current session can be used to transfer a chat ownership to another user.
 */
class CanTransferOwnershipResult: public Object {
 public:
};

/**
 * The session can be used.
 */
class canTransferOwnershipResultOk final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The session can be used.
   */
  canTransferOwnershipResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -89881021;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The 2-step verification needs to be enabled first.
 */
class canTransferOwnershipResultPasswordNeeded final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The 2-step verification needs to be enabled first.
   */
  canTransferOwnershipResultPasswordNeeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1548372703;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The 2-step verification was enabled recently, user needs to wait.
 */
class canTransferOwnershipResultPasswordTooFresh final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Time left before the session can be used to transfer ownership of a chat, in seconds.
  int32 retry_after_;

  /**
   * The 2-step verification was enabled recently, user needs to wait.
   */
  canTransferOwnershipResultPasswordTooFresh();

  /**
   * The 2-step verification was enabled recently, user needs to wait.
   *
   * \param[in] retry_after_ Time left before the session can be used to transfer ownership of a chat, in seconds.
   */
  explicit canTransferOwnershipResultPasswordTooFresh(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 811440913;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The session was created recently, user needs to wait.
 */
class canTransferOwnershipResultSessionTooFresh final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Time left before the session can be used to transfer ownership of a chat, in seconds.
  int32 retry_after_;

  /**
   * The session was created recently, user needs to wait.
   */
  canTransferOwnershipResultSessionTooFresh();

  /**
   * The session was created recently, user needs to wait.
   *
   * \param[in] retry_after_ Time left before the session can be used to transfer ownership of a chat, in seconds.
   */
  explicit canTransferOwnershipResultSessionTooFresh(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 984664289;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class BlockList;

class ChatActionBar;

class ChatAvailableReactions;

class ChatList;

class ChatType;

class MessageSender;

class businessBotManageBar;

class chatBackground;

class chatJoinRequestsInfo;

class chatNotificationSettings;

class chatPermissions;

class chatPhotoInfo;

class chatPosition;

class draftMessage;

class emojiStatus;

class message;

class videoChat;

/**
 * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
 */
class chat final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat unique identifier.
  int53 id_;
  /// Type of the chat.
  object_ptr<ChatType> type_;
  /// Chat title.
  string title_;
  /// Chat photo; may be null.
  object_ptr<chatPhotoInfo> photo_;
  /// Identifier of the accent color for message sender name, and backgrounds of chat photo, reply header, and link preview.
  int32 accent_color_id_;
  /// Identifier of a custom emoji to be shown on the reply header and link preview background for messages sent by the chat; 0 if none.
  int64 background_custom_emoji_id_;
  /// Identifier of the profile accent color for the chat's profile; -1 if none.
  int32 profile_accent_color_id_;
  /// Identifier of a custom emoji to be shown on the background of the chat's profile; 0 if none.
  int64 profile_background_custom_emoji_id_;
  /// Actions that non-administrator chat members are allowed to take in the chat.
  object_ptr<chatPermissions> permissions_;
  /// Last message in the chat; may be null if none or unknown.
  object_ptr<message> last_message_;
  /// Positions of the chat in chat lists.
  array<object_ptr<chatPosition>> positions_;
  /// Chat lists to which the chat belongs. A chat can have a non-zero position in a chat list even it doesn't belong to the chat list and have no position in a chat list even it belongs to the chat list.
  array<object_ptr<ChatList>> chat_lists_;
  /// Identifier of a user or chat that is selected to send messages in the chat; may be null if the user can't change message sender.
  object_ptr<MessageSender> message_sender_id_;
  /// Block list to which the chat is added; may be null if none.
  object_ptr<BlockList> block_list_;
  /// True, if chat content can't be saved locally, forwarded, or copied.
  bool has_protected_content_;
  /// True, if translation of all messages in the chat must be suggested to the user.
  bool is_translatable_;
  /// True, if the chat is marked as unread.
  bool is_marked_as_unread_;
  /// True, if the chat is a forum supergroup that must be shown in the &quot;View as topics&quot; mode, or Saved Messages chat that must be shown in the &quot;View as chats&quot;.
  bool view_as_topics_;
  /// True, if the chat has scheduled messages.
  bool has_scheduled_messages_;
  /// True, if the chat messages can be deleted only for the current user while other users will continue to see the messages.
  bool can_be_deleted_only_for_self_;
  /// True, if the chat messages can be deleted for all users.
  bool can_be_deleted_for_all_users_;
  /// True, if the chat can be reported to Telegram moderators through reportChat or reportChatPhoto.
  bool can_be_reported_;
  /// Default value of the disable_notification parameter, used when a message is sent to the chat.
  bool default_disable_notification_;
  /// Number of unread messages in the chat.
  int32 unread_count_;
  /// Identifier of the last read incoming message.
  int53 last_read_inbox_message_id_;
  /// Identifier of the last read outgoing message.
  int53 last_read_outbox_message_id_;
  /// Number of unread messages with a mention/reply in the chat.
  int32 unread_mention_count_;
  /// Number of messages with unread reactions in the chat.
  int32 unread_reaction_count_;
  /// Notification settings for the chat.
  object_ptr<chatNotificationSettings> notification_settings_;
  /// Types of reaction, available in the chat.
  object_ptr<ChatAvailableReactions> available_reactions_;
  /// Current message auto-delete or self-destruct timer setting for the chat, in seconds; 0 if disabled. Self-destruct timer in secret chats starts after the message or its content is viewed. Auto-delete timer in other chats starts from the send date.
  int32 message_auto_delete_time_;
  /// Emoji status to be shown along with chat title; may be null.
  object_ptr<emojiStatus> emoji_status_;
  /// Background set for the chat; may be null if none.
  object_ptr<chatBackground> background_;
  /// If non-empty, name of a theme, set for the chat.
  string theme_name_;
  /// Information about actions which must be possible to do through the chat action bar; may be null if none.
  object_ptr<ChatActionBar> action_bar_;
  /// Information about bar for managing a business bot in the chat; may be null if none.
  object_ptr<businessBotManageBar> business_bot_manage_bar_;
  /// Information about video chat of the chat.
  object_ptr<videoChat> video_chat_;
  /// Information about pending join requests; may be null if none.
  object_ptr<chatJoinRequestsInfo> pending_join_requests_;
  /// Identifier of the message from which reply markup needs to be used; 0 if there is no default custom reply markup in the chat.
  int53 reply_markup_message_id_;
  /// A draft of a message in the chat; may be null if none.
  object_ptr<draftMessage> draft_message_;
  /// Application-specific data associated with the chat. (For example, the chat scroll position or local chat notification settings can be stored here.) Persistent if the message database is used.
  string client_data_;

  /**
   * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
   */
  chat();

  /**
   * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
   *
   * \param[in] id_ Chat unique identifier.
   * \param[in] type_ Type of the chat.
   * \param[in] title_ Chat title.
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] accent_color_id_ Identifier of the accent color for message sender name, and backgrounds of chat photo, reply header, and link preview.
   * \param[in] background_custom_emoji_id_ Identifier of a custom emoji to be shown on the reply header and link preview background for messages sent by the chat; 0 if none.
   * \param[in] profile_accent_color_id_ Identifier of the profile accent color for the chat's profile; -1 if none.
   * \param[in] profile_background_custom_emoji_id_ Identifier of a custom emoji to be shown on the background of the chat's profile; 0 if none.
   * \param[in] permissions_ Actions that non-administrator chat members are allowed to take in the chat.
   * \param[in] last_message_ Last message in the chat; may be null if none or unknown.
   * \param[in] positions_ Positions of the chat in chat lists.
   * \param[in] chat_lists_ Chat lists to which the chat belongs. A chat can have a non-zero position in a chat list even it doesn't belong to the chat list and have no position in a chat list even it belongs to the chat list.
   * \param[in] message_sender_id_ Identifier of a user or chat that is selected to send messages in the chat; may be null if the user can't change message sender.
   * \param[in] block_list_ Block list to which the chat is added; may be null if none.
   * \param[in] has_protected_content_ True, if chat content can't be saved locally, forwarded, or copied.
   * \param[in] is_translatable_ True, if translation of all messages in the chat must be suggested to the user.
   * \param[in] is_marked_as_unread_ True, if the chat is marked as unread.
   * \param[in] view_as_topics_ True, if the chat is a forum supergroup that must be shown in the &quot;View as topics&quot; mode, or Saved Messages chat that must be shown in the &quot;View as chats&quot;.
   * \param[in] has_scheduled_messages_ True, if the chat has scheduled messages.
   * \param[in] can_be_deleted_only_for_self_ True, if the chat messages can be deleted only for the current user while other users will continue to see the messages.
   * \param[in] can_be_deleted_for_all_users_ True, if the chat messages can be deleted for all users.
   * \param[in] can_be_reported_ True, if the chat can be reported to Telegram moderators through reportChat or reportChatPhoto.
   * \param[in] default_disable_notification_ Default value of the disable_notification parameter, used when a message is sent to the chat.
   * \param[in] unread_count_ Number of unread messages in the chat.
   * \param[in] last_read_inbox_message_id_ Identifier of the last read incoming message.
   * \param[in] last_read_outbox_message_id_ Identifier of the last read outgoing message.
   * \param[in] unread_mention_count_ Number of unread messages with a mention/reply in the chat.
   * \param[in] unread_reaction_count_ Number of messages with unread reactions in the chat.
   * \param[in] notification_settings_ Notification settings for the chat.
   * \param[in] available_reactions_ Types of reaction, available in the chat.
   * \param[in] message_auto_delete_time_ Current message auto-delete or self-destruct timer setting for the chat, in seconds; 0 if disabled. Self-destruct timer in secret chats starts after the message or its content is viewed. Auto-delete timer in other chats starts from the send date.
   * \param[in] emoji_status_ Emoji status to be shown along with chat title; may be null.
   * \param[in] background_ Background set for the chat; may be null if none.
   * \param[in] theme_name_ If non-empty, name of a theme, set for the chat.
   * \param[in] action_bar_ Information about actions which must be possible to do through the chat action bar; may be null if none.
   * \param[in] business_bot_manage_bar_ Information about bar for managing a business bot in the chat; may be null if none.
   * \param[in] video_chat_ Information about video chat of the chat.
   * \param[in] pending_join_requests_ Information about pending join requests; may be null if none.
   * \param[in] reply_markup_message_id_ Identifier of the message from which reply markup needs to be used; 0 if there is no default custom reply markup in the chat.
   * \param[in] draft_message_ A draft of a message in the chat; may be null if none.
   * \param[in] client_data_ Application-specific data associated with the chat. (For example, the chat scroll position or local chat notification settings can be stored here.) Persistent if the message database is used.
   */
  chat(int53 id_, object_ptr<ChatType> &&type_, string const &title_, object_ptr<chatPhotoInfo> &&photo_, int32 accent_color_id_, int64 background_custom_emoji_id_, int32 profile_accent_color_id_, int64 profile_background_custom_emoji_id_, object_ptr<chatPermissions> &&permissions_, object_ptr<message> &&last_message_, array<object_ptr<chatPosition>> &&positions_, array<object_ptr<ChatList>> &&chat_lists_, object_ptr<MessageSender> &&message_sender_id_, object_ptr<BlockList> &&block_list_, bool has_protected_content_, bool is_translatable_, bool is_marked_as_unread_, bool view_as_topics_, bool has_scheduled_messages_, bool can_be_deleted_only_for_self_, bool can_be_deleted_for_all_users_, bool can_be_reported_, bool default_disable_notification_, int32 unread_count_, int53 last_read_inbox_message_id_, int53 last_read_outbox_message_id_, int32 unread_mention_count_, int32 unread_reaction_count_, object_ptr<chatNotificationSettings> &&notification_settings_, object_ptr<ChatAvailableReactions> &&available_reactions_, int32 message_auto_delete_time_, object_ptr<emojiStatus> &&emoji_status_, object_ptr<chatBackground> &&background_, string const &theme_name_, object_ptr<ChatActionBar> &&action_bar_, object_ptr<businessBotManageBar> &&business_bot_manage_bar_, object_ptr<videoChat> &&video_chat_, object_ptr<chatJoinRequestsInfo> &&pending_join_requests_, int53 reply_markup_message_id_, object_ptr<draftMessage> &&draft_message_, string const &client_data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 830601369;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the different types of activity in a chat.
 */
class ChatAction: public Object {
 public:
};

/**
 * The user is typing a message.
 */
class chatActionTyping final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is typing a message.
   */
  chatActionTyping();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 380122167;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is recording a video.
 */
class chatActionRecordingVideo final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is recording a video.
   */
  chatActionRecordingVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 216553362;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a video.
 */
class chatActionUploadingVideo final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a video.
   */
  chatActionUploadingVideo();

  /**
   * The user is uploading a video.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVideo(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1234185270;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is recording a voice note.
 */
class chatActionRecordingVoiceNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is recording a voice note.
   */
  chatActionRecordingVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -808850058;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a voice note.
 */
class chatActionUploadingVoiceNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a voice note.
   */
  chatActionUploadingVoiceNote();

  /**
   * The user is uploading a voice note.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVoiceNote(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -613643666;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a photo.
 */
class chatActionUploadingPhoto final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a photo.
   */
  chatActionUploadingPhoto();

  /**
   * The user is uploading a photo.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingPhoto(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 654240583;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a document.
 */
class chatActionUploadingDocument final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a document.
   */
  chatActionUploadingDocument();

  /**
   * The user is uploading a document.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingDocument(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 167884362;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is picking a sticker to send.
 */
class chatActionChoosingSticker final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is picking a sticker to send.
   */
  chatActionChoosingSticker();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 372753697;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is picking a location or venue to send.
 */
class chatActionChoosingLocation final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is picking a location or venue to send.
   */
  chatActionChoosingLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2017893596;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is picking a contact to send.
 */
class chatActionChoosingContact final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is picking a contact to send.
   */
  chatActionChoosingContact();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1222507496;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has started to play a game.
 */
class chatActionStartPlayingGame final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user has started to play a game.
   */
  chatActionStartPlayingGame();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -865884164;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is recording a video note.
 */
class chatActionRecordingVideoNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is recording a video note.
   */
  chatActionRecordingVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 16523393;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is uploading a video note.
 */
class chatActionUploadingVideoNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Upload progress, as a percentage.
  int32 progress_;

  /**
   * The user is uploading a video note.
   */
  chatActionUploadingVideoNote();

  /**
   * The user is uploading a video note.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVideoNote(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1172364918;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is watching animations sent by the other party by clicking on an animated emoji.
 */
class chatActionWatchingAnimations final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The animated emoji.
  string emoji_;

  /**
   * The user is watching animations sent by the other party by clicking on an animated emoji.
   */
  chatActionWatchingAnimations();

  /**
   * The user is watching animations sent by the other party by clicking on an animated emoji.
   *
   * \param[in] emoji_ The animated emoji.
   */
  explicit chatActionWatchingAnimations(string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2052990641;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has canceled the previous action.
 */
class chatActionCancel final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user has canceled the previous action.
   */
  chatActionCancel();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1160523958;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes actions which must be possible to do through a chat action bar.
 */
class ChatActionBar: public Object {
 public:
};

/**
 * The chat can be reported as spam using the method reportChat with an empty option_id and message_ids. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
 */
class chatActionBarReportSpam final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
  bool can_unarchive_;

  /**
   * The chat can be reported as spam using the method reportChat with an empty option_id and message_ids. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   */
  chatActionBarReportSpam();

  /**
   * The chat can be reported as spam using the method reportChat with an empty option_id and message_ids. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   *
   * \param[in] can_unarchive_ If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
   */
  explicit chatActionBarReportSpam(bool can_unarchive_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1312758246;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a recently created group chat to which new members can be invited.
 */
class chatActionBarInviteMembers final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The chat is a recently created group chat to which new members can be invited.
   */
  chatActionBarInviteMembers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1985313904;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method setMessageSenderBlockList, or the other user can be added to the contact list using the method addContact. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
 */
class chatActionBarReportAddBlock final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
  bool can_unarchive_;

  /**
   * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method setMessageSenderBlockList, or the other user can be added to the contact list using the method addContact. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   */
  chatActionBarReportAddBlock();

  /**
   * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method setMessageSenderBlockList, or the other user can be added to the contact list using the method addContact. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   *
   * \param[in] can_unarchive_ If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
   */
  explicit chatActionBarReportAddBlock(bool can_unarchive_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1451980246;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private or secret chat and the other user can be added to the contact list using the method addContact.
 */
class chatActionBarAddContact final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The chat is a private or secret chat and the other user can be added to the contact list using the method addContact.
   */
  chatActionBarAddContact();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -733325295;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private or secret chat with a mutual contact and the user's phone number can be shared with the other user using the method sharePhoneNumber.
 */
class chatActionBarSharePhoneNumber final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The chat is a private or secret chat with a mutual contact and the user's phone number can be shared with the other user using the method sharePhoneNumber.
   */
  chatActionBarSharePhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 35188697;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat is a private chat with an administrator of a chat to which the user sent join request.
 */
class chatActionBarJoinRequest final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Title of the chat to which the join request was sent.
  string title_;
  /// True, if the join request was sent to a channel chat.
  bool is_channel_;
  /// Point in time (Unix timestamp) when the join request was sent.
  int32 request_date_;

  /**
   * The chat is a private chat with an administrator of a chat to which the user sent join request.
   */
  chatActionBarJoinRequest();

  /**
   * The chat is a private chat with an administrator of a chat to which the user sent join request.
   *
   * \param[in] title_ Title of the chat to which the join request was sent.
   * \param[in] is_channel_ True, if the join request was sent to a channel chat.
   * \param[in] request_date_ Point in time (Unix timestamp) when the join request was sent.
   */
  chatActionBarJoinRequest(string const &title_, bool is_channel_, int32 request_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1037140744;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class StoryList;

class storyInfo;

/**
 * Describes active stories posted by a chat.
 */
class chatActiveStories final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the chat that posted the stories.
  int53 chat_id_;
  /// Identifier of the story list in which the stories are shown; may be null if the stories aren't shown in a story list.
  object_ptr<StoryList> list_;
  /// A parameter used to determine order of the stories in the story list; 0 if the stories doesn't need to be shown in the story list. Stories must be sorted by the pair (order, story_sender_chat_id) in descending order.
  int53 order_;
  /// Identifier of the last read active story.
  int32 max_read_story_id_;
  /// Basic information about the stories; use getStory to get full information about the stories. The stories are in chronological order (i.e., in order of increasing story identifiers).
  array<object_ptr<storyInfo>> stories_;

  /**
   * Describes active stories posted by a chat.
   */
  chatActiveStories();

  /**
   * Describes active stories posted by a chat.
   *
   * \param[in] chat_id_ Identifier of the chat that posted the stories.
   * \param[in] list_ Identifier of the story list in which the stories are shown; may be null if the stories aren't shown in a story list.
   * \param[in] order_ A parameter used to determine order of the stories in the story list; 0 if the stories doesn't need to be shown in the story list. Stories must be sorted by the pair (order, story_sender_chat_id) in descending order.
   * \param[in] max_read_story_id_ Identifier of the last read active story.
   * \param[in] stories_ Basic information about the stories; use getStory to get full information about the stories. The stories are in chronological order (i.e., in order of increasing story identifiers).
   */
  chatActiveStories(int53 chat_id_, object_ptr<StoryList> &&list_, int53 order_, int32 max_read_story_id_, array<object_ptr<storyInfo>> &&stories_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1398869529;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a chat administrator.
 */
class chatAdministrator final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the administrator.
  int53 user_id_;
  /// Custom title of the administrator.
  string custom_title_;
  /// True, if the user is the owner of the chat.
  bool is_owner_;

  /**
   * Contains information about a chat administrator.
   */
  chatAdministrator();

  /**
   * Contains information about a chat administrator.
   *
   * \param[in] user_id_ User identifier of the administrator.
   * \param[in] custom_title_ Custom title of the administrator.
   * \param[in] is_owner_ True, if the user is the owner of the chat.
   */
  chatAdministrator(int53 user_id_, string const &custom_title_, bool is_owner_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1920449836;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes rights of the administrator.
 */
class chatAdministratorRights final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if the administrator can access the chat event log, get boost list, see hidden supergroup and channel members, report supergroup spam messages and ignore slow mode. Implied by any other privilege; applicable to supergroups and channels only.
  bool can_manage_chat_;
  /// True, if the administrator can change the chat title, photo, and other settings.
  bool can_change_info_;
  /// True, if the administrator can create channel posts or view channel statistics; applicable to channels only.
  bool can_post_messages_;
  /// True, if the administrator can edit messages of other users and pin messages; applicable to channels only.
  bool can_edit_messages_;
  /// True, if the administrator can delete messages of other users.
  bool can_delete_messages_;
  /// True, if the administrator can invite new users to the chat.
  bool can_invite_users_;
  /// True, if the administrator can restrict, ban, or unban chat members or view supergroup statistics; always true for channels.
  bool can_restrict_members_;
  /// True, if the administrator can pin messages; applicable to basic groups and supergroups only.
  bool can_pin_messages_;
  /// True, if the administrator can create, rename, close, reopen, hide, and unhide forum topics; applicable to forum supergroups only.
  bool can_manage_topics_;
  /// True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them.
  bool can_promote_members_;
  /// True, if the administrator can manage video chats.
  bool can_manage_video_chats_;
  /// True, if the administrator can create new chat stories, or edit and delete posted stories; applicable to supergroups and channels only.
  bool can_post_stories_;
  /// True, if the administrator can edit stories posted by other users, post stories to the chat page, pin chat stories, and access story archive; applicable to supergroups and channels only.
  bool can_edit_stories_;
  /// True, if the administrator can delete stories posted by other users; applicable to supergroups and channels only.
  bool can_delete_stories_;
  /// True, if the administrator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
  bool is_anonymous_;

  /**
   * Describes rights of the administrator.
   */
  chatAdministratorRights();

  /**
   * Describes rights of the administrator.
   *
   * \param[in] can_manage_chat_ True, if the administrator can access the chat event log, get boost list, see hidden supergroup and channel members, report supergroup spam messages and ignore slow mode. Implied by any other privilege; applicable to supergroups and channels only.
   * \param[in] can_change_info_ True, if the administrator can change the chat title, photo, and other settings.
   * \param[in] can_post_messages_ True, if the administrator can create channel posts or view channel statistics; applicable to channels only.
   * \param[in] can_edit_messages_ True, if the administrator can edit messages of other users and pin messages; applicable to channels only.
   * \param[in] can_delete_messages_ True, if the administrator can delete messages of other users.
   * \param[in] can_invite_users_ True, if the administrator can invite new users to the chat.
   * \param[in] can_restrict_members_ True, if the administrator can restrict, ban, or unban chat members or view supergroup statistics; always true for channels.
   * \param[in] can_pin_messages_ True, if the administrator can pin messages; applicable to basic groups and supergroups only.
   * \param[in] can_manage_topics_ True, if the administrator can create, rename, close, reopen, hide, and unhide forum topics; applicable to forum supergroups only.
   * \param[in] can_promote_members_ True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them.
   * \param[in] can_manage_video_chats_ True, if the administrator can manage video chats.
   * \param[in] can_post_stories_ True, if the administrator can create new chat stories, or edit and delete posted stories; applicable to supergroups and channels only.
   * \param[in] can_edit_stories_ True, if the administrator can edit stories posted by other users, post stories to the chat page, pin chat stories, and access story archive; applicable to supergroups and channels only.
   * \param[in] can_delete_stories_ True, if the administrator can delete stories posted by other users; applicable to supergroups and channels only.
   * \param[in] is_anonymous_ True, if the administrator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
   */
  chatAdministratorRights(bool can_manage_chat_, bool can_change_info_, bool can_post_messages_, bool can_edit_messages_, bool can_delete_messages_, bool can_invite_users_, bool can_restrict_members_, bool can_pin_messages_, bool can_manage_topics_, bool can_promote_members_, bool can_manage_video_chats_, bool can_post_stories_, bool can_edit_stories_, bool can_delete_stories_, bool is_anonymous_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1599049796;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatAdministrator;

/**
 * Represents a list of chat administrators.
 */
class chatAdministrators final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A list of chat administrators.
  array<object_ptr<chatAdministrator>> administrators_;

  /**
   * Represents a list of chat administrators.
   */
  chatAdministrators();

  /**
   * Represents a list of chat administrators.
   *
   * \param[in] administrators_ A list of chat administrators.
   */
  explicit chatAdministrators(array<object_ptr<chatAdministrator>> &&administrators_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2126186435;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class affiliateProgramParameters;

/**
 * Describes an affiliate program that was connected to a chat.
 */
class chatAffiliateProgram final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The link that can be used to refer users if the program is still active.
  string url_;
  /// User identifier of the bot created the program.
  int53 bot_user_id_;
  /// The parameters of the affiliate program.
  object_ptr<affiliateProgramParameters> parameters_;
  /// Point in time (Unix timestamp) when the affiliate program was connected.
  int32 connection_date_;
  /// True, if the program was canceled by the bot, or disconnected by the chat owner and isn't available anymore.
  bool is_disconnected_;
  /// The number of users that used the affiliate program.
  int64 user_count_;
  /// The number of Telegram Stars that were earned by the affiliate program.
  int64 revenue_star_count_;

  /**
   * Describes an affiliate program that was connected to a chat.
   */
  chatAffiliateProgram();

  /**
   * Describes an affiliate program that was connected to a chat.
   *
   * \param[in] url_ The link that can be used to refer users if the program is still active.
   * \param[in] bot_user_id_ User identifier of the bot created the program.
   * \param[in] parameters_ The parameters of the affiliate program.
   * \param[in] connection_date_ Point in time (Unix timestamp) when the affiliate program was connected.
   * \param[in] is_disconnected_ True, if the program was canceled by the bot, or disconnected by the chat owner and isn't available anymore.
   * \param[in] user_count_ The number of users that used the affiliate program.
   * \param[in] revenue_star_count_ The number of Telegram Stars that were earned by the affiliate program.
   */
  chatAffiliateProgram(string const &url_, int53 bot_user_id_, object_ptr<affiliateProgramParameters> &&parameters_, int32 connection_date_, bool is_disconnected_, int64 user_count_, int64 revenue_star_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1415835338;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatAffiliateProgram;

/**
 * Represents a list of affiliate programs that were connected to a chat.
 */
class chatAffiliatePrograms final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The total number of affiliate programs that were connected to the chat.
  int32 total_count_;
  /// The list of connected affiliate programs.
  array<object_ptr<chatAffiliateProgram>> programs_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Represents a list of affiliate programs that were connected to a chat.
   */
  chatAffiliatePrograms();

  /**
   * Represents a list of affiliate programs that were connected to a chat.
   *
   * \param[in] total_count_ The total number of affiliate programs that were connected to the chat.
   * \param[in] programs_ The list of connected affiliate programs.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  chatAffiliatePrograms(int32 total_count_, array<object_ptr<chatAffiliateProgram>> &&programs_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1107955123;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ReactionType;

/**
 * This class is an abstract base class.
 * Describes reactions available in the chat.
 */
class ChatAvailableReactions: public Object {
 public:
};

/**
 * All reactions are available in the chat, excluding the paid reaction and custom reactions in channel chats.
 */
class chatAvailableReactionsAll final : public ChatAvailableReactions {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The maximum allowed number of reactions per message; 1-11.
  int32 max_reaction_count_;

  /**
   * All reactions are available in the chat, excluding the paid reaction and custom reactions in channel chats.
   */
  chatAvailableReactionsAll();

  /**
   * All reactions are available in the chat, excluding the paid reaction and custom reactions in channel chats.
   *
   * \param[in] max_reaction_count_ The maximum allowed number of reactions per message; 1-11.
   */
  explicit chatAvailableReactionsAll(int32 max_reaction_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 694160279;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Only specific reactions are available in the chat.
 */
class chatAvailableReactionsSome final : public ChatAvailableReactions {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The list of reactions.
  array<object_ptr<ReactionType>> reactions_;
  /// The maximum allowed number of reactions per message; 1-11.
  int32 max_reaction_count_;

  /**
   * Only specific reactions are available in the chat.
   */
  chatAvailableReactionsSome();

  /**
   * Only specific reactions are available in the chat.
   *
   * \param[in] reactions_ The list of reactions.
   * \param[in] max_reaction_count_ The maximum allowed number of reactions per message; 1-11.
   */
  chatAvailableReactionsSome(array<object_ptr<ReactionType>> &&reactions_, int32 max_reaction_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 152513153;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class background;

/**
 * Describes a background set for a specific chat.
 */
class chatBackground final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The background.
  object_ptr<background> background_;
  /// Dimming of the background in dark themes, as a percentage; 0-100. Applied only to Wallpaper and Fill types of background.
  int32 dark_theme_dimming_;

  /**
   * Describes a background set for a specific chat.
   */
  chatBackground();

  /**
   * Describes a background set for a specific chat.
   *
   * \param[in] background_ The background.
   * \param[in] dark_theme_dimming_ Dimming of the background in dark themes, as a percentage; 0-100. Applied only to Wallpaper and Fill types of background.
   */
  chatBackground(object_ptr<background> &&background_, int32 dark_theme_dimming_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1653152104;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatBoostSource;

/**
 * Describes a boost applied to a chat.
 */
class chatBoost final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the boost.
  string id_;
  /// The number of identical boosts applied.
  int32 count_;
  /// Source of the boost.
  object_ptr<ChatBoostSource> source_;
  /// Point in time (Unix timestamp) when the chat was boosted.
  int32 start_date_;
  /// Point in time (Unix timestamp) when the boost will expire.
  int32 expiration_date_;

  /**
   * Describes a boost applied to a chat.
   */
  chatBoost();

  /**
   * Describes a boost applied to a chat.
   *
   * \param[in] id_ Unique identifier of the boost.
   * \param[in] count_ The number of identical boosts applied.
   * \param[in] source_ Source of the boost.
   * \param[in] start_date_ Point in time (Unix timestamp) when the chat was boosted.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the boost will expire.
   */
  chatBoost(string const &id_, int32 count_, object_ptr<ChatBoostSource> &&source_, int32 start_date_, int32 expiration_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1765815118;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatBoostLevelFeatures;

/**
 * Contains a list of features available on the first chat boost levels.
 */
class chatBoostFeatures final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The list of features.
  array<object_ptr<chatBoostLevelFeatures>> features_;
  /// The minimum boost level required to set custom emoji for profile background.
  int32 min_profile_background_custom_emoji_boost_level_;
  /// The minimum boost level required to set custom emoji for reply header and link preview background; for channel chats only.
  int32 min_background_custom_emoji_boost_level_;
  /// The minimum boost level required to set emoji status.
  int32 min_emoji_status_boost_level_;
  /// The minimum boost level required to set a chat theme background as chat background.
  int32 min_chat_theme_background_boost_level_;
  /// The minimum boost level required to set custom chat background.
  int32 min_custom_background_boost_level_;
  /// The minimum boost level required to set custom emoji sticker set for the chat; for supergroup chats only.
  int32 min_custom_emoji_sticker_set_boost_level_;
  /// The minimum boost level allowing to recognize speech in video note and voice note messages for non-Premium users; for supergroup chats only.
  int32 min_speech_recognition_boost_level_;
  /// The minimum boost level allowing to disable sponsored messages in the chat; for channel chats only.
  int32 min_sponsored_message_disable_boost_level_;

  /**
   * Contains a list of features available on the first chat boost levels.
   */
  chatBoostFeatures();

  /**
   * Contains a list of features available on the first chat boost levels.
   *
   * \param[in] features_ The list of features.
   * \param[in] min_profile_background_custom_emoji_boost_level_ The minimum boost level required to set custom emoji for profile background.
   * \param[in] min_background_custom_emoji_boost_level_ The minimum boost level required to set custom emoji for reply header and link preview background; for channel chats only.
   * \param[in] min_emoji_status_boost_level_ The minimum boost level required to set emoji status.
   * \param[in] min_chat_theme_background_boost_level_ The minimum boost level required to set a chat theme background as chat background.
   * \param[in] min_custom_background_boost_level_ The minimum boost level required to set custom chat background.
   * \param[in] min_custom_emoji_sticker_set_boost_level_ The minimum boost level required to set custom emoji sticker set for the chat; for supergroup chats only.
   * \param[in] min_speech_recognition_boost_level_ The minimum boost level allowing to recognize speech in video note and voice note messages for non-Premium users; for supergroup chats only.
   * \param[in] min_sponsored_message_disable_boost_level_ The minimum boost level allowing to disable sponsored messages in the chat; for channel chats only.
   */
  chatBoostFeatures(array<object_ptr<chatBoostLevelFeatures>> &&features_, int32 min_profile_background_custom_emoji_boost_level_, int32 min_background_custom_emoji_boost_level_, int32 min_emoji_status_boost_level_, int32 min_chat_theme_background_boost_level_, int32 min_custom_background_boost_level_, int32 min_custom_emoji_sticker_set_boost_level_, int32 min_speech_recognition_boost_level_, int32 min_sponsored_message_disable_boost_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 866182642;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of features available on a specific chat boost level.
 */
class chatBoostLevelFeatures final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Target chat boost level.
  int32 level_;
  /// Number of stories that the chat can publish daily.
  int32 story_per_day_count_;
  /// Number of custom emoji reactions that can be added to the list of available reactions.
  int32 custom_emoji_reaction_count_;
  /// Number of custom colors for chat title.
  int32 title_color_count_;
  /// Number of custom colors for profile photo background.
  int32 profile_accent_color_count_;
  /// True, if custom emoji for profile background can be set.
  bool can_set_profile_background_custom_emoji_;
  /// Number of custom colors for background of empty chat photo, replies to messages and link previews.
  int32 accent_color_count_;
  /// True, if custom emoji for reply header and link preview background can be set.
  bool can_set_background_custom_emoji_;
  /// True, if emoji status can be set.
  bool can_set_emoji_status_;
  /// Number of chat theme backgrounds that can be set as chat background.
  int32 chat_theme_background_count_;
  /// True, if custom background can be set in the chat for all users.
  bool can_set_custom_background_;
  /// True, if custom emoji sticker set can be set for the chat.
  bool can_set_custom_emoji_sticker_set_;
  /// True, if speech recognition can be used for video note and voice note messages by all users.
  bool can_recognize_speech_;
  /// True, if sponsored messages can be disabled in the chat.
  bool can_disable_sponsored_messages_;

  /**
   * Contains a list of features available on a specific chat boost level.
   */
  chatBoostLevelFeatures();

  /**
   * Contains a list of features available on a specific chat boost level.
   *
   * \param[in] level_ Target chat boost level.
   * \param[in] story_per_day_count_ Number of stories that the chat can publish daily.
   * \param[in] custom_emoji_reaction_count_ Number of custom emoji reactions that can be added to the list of available reactions.
   * \param[in] title_color_count_ Number of custom colors for chat title.
   * \param[in] profile_accent_color_count_ Number of custom colors for profile photo background.
   * \param[in] can_set_profile_background_custom_emoji_ True, if custom emoji for profile background can be set.
   * \param[in] accent_color_count_ Number of custom colors for background of empty chat photo, replies to messages and link previews.
   * \param[in] can_set_background_custom_emoji_ True, if custom emoji for reply header and link preview background can be set.
   * \param[in] can_set_emoji_status_ True, if emoji status can be set.
   * \param[in] chat_theme_background_count_ Number of chat theme backgrounds that can be set as chat background.
   * \param[in] can_set_custom_background_ True, if custom background can be set in the chat for all users.
   * \param[in] can_set_custom_emoji_sticker_set_ True, if custom emoji sticker set can be set for the chat.
   * \param[in] can_recognize_speech_ True, if speech recognition can be used for video note and voice note messages by all users.
   * \param[in] can_disable_sponsored_messages_ True, if sponsored messages can be disabled in the chat.
   */
  chatBoostLevelFeatures(int32 level_, int32 story_per_day_count_, int32 custom_emoji_reaction_count_, int32 title_color_count_, int32 profile_accent_color_count_, bool can_set_profile_background_custom_emoji_, int32 accent_color_count_, bool can_set_background_custom_emoji_, bool can_set_emoji_status_, int32 chat_theme_background_count_, bool can_set_custom_background_, bool can_set_custom_emoji_sticker_set_, bool can_recognize_speech_, bool can_disable_sponsored_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -189458156;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains an HTTPS link to boost a chat.
 */
class chatBoostLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The link.
  string link_;
  /// True, if the link will work for non-members of the chat.
  bool is_public_;

  /**
   * Contains an HTTPS link to boost a chat.
   */
  chatBoostLink();

  /**
   * Contains an HTTPS link to boost a chat.
   *
   * \param[in] link_ The link.
   * \param[in] is_public_ True, if the link will work for non-members of the chat.
   */
  chatBoostLink(string const &link_, bool is_public_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1253999503;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a link to boost a chat.
 */
class chatBoostLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if the link will work for non-members of the chat.
  bool is_public_;
  /// Identifier of the chat to which the link points; 0 if the chat isn't found.
  int53 chat_id_;

  /**
   * Contains information about a link to boost a chat.
   */
  chatBoostLinkInfo();

  /**
   * Contains information about a link to boost a chat.
   *
   * \param[in] is_public_ True, if the link will work for non-members of the chat.
   * \param[in] chat_id_ Identifier of the chat to which the link points; 0 if the chat isn't found.
   */
  chatBoostLinkInfo(bool is_public_, int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -602785660;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a slot for chat boost.
 */
class chatBoostSlot final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the slot.
  int32 slot_id_;
  /// Identifier of the currently boosted chat; 0 if none.
  int53 currently_boosted_chat_id_;
  /// Point in time (Unix timestamp) when the chat was boosted; 0 if none.
  int32 start_date_;
  /// Point in time (Unix timestamp) when the boost will expire.
  int32 expiration_date_;
  /// Point in time (Unix timestamp) after which the boost can be used for another chat.
  int32 cooldown_until_date_;

  /**
   * Describes a slot for chat boost.
   */
  chatBoostSlot();

  /**
   * Describes a slot for chat boost.
   *
   * \param[in] slot_id_ Unique identifier of the slot.
   * \param[in] currently_boosted_chat_id_ Identifier of the currently boosted chat; 0 if none.
   * \param[in] start_date_ Point in time (Unix timestamp) when the chat was boosted; 0 if none.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the boost will expire.
   * \param[in] cooldown_until_date_ Point in time (Unix timestamp) after which the boost can be used for another chat.
   */
  chatBoostSlot(int32 slot_id_, int53 currently_boosted_chat_id_, int32 start_date_, int32 expiration_date_, int32 cooldown_until_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 123206343;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatBoostSlot;

/**
 * Contains a list of chat boost slots.
 */
class chatBoostSlots final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of boost slots.
  array<object_ptr<chatBoostSlot>> slots_;

  /**
   * Contains a list of chat boost slots.
   */
  chatBoostSlots();

  /**
   * Contains a list of chat boost slots.
   *
   * \param[in] slots_ List of boost slots.
   */
  explicit chatBoostSlots(array<object_ptr<chatBoostSlot>> &&slots_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1014966293;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes source of a chat boost.
 */
class ChatBoostSource: public Object {
 public:
};

/**
 * The chat created a Telegram Premium gift code for a user.
 */
class chatBoostSourceGiftCode final : public ChatBoostSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of a user, for which the gift code was created.
  int53 user_id_;
  /// The created Telegram Premium gift code, which is known only if this is a gift code for the current user, or it has already been claimed.
  string gift_code_;

  /**
   * The chat created a Telegram Premium gift code for a user.
   */
  chatBoostSourceGiftCode();

  /**
   * The chat created a Telegram Premium gift code for a user.
   *
   * \param[in] user_id_ Identifier of a user, for which the gift code was created.
   * \param[in] gift_code_ The created Telegram Premium gift code, which is known only if this is a gift code for the current user, or it has already been claimed.
   */
  chatBoostSourceGiftCode(int53 user_id_, string const &gift_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -98299206;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat created a giveaway.
 */
class chatBoostSourceGiveaway final : public ChatBoostSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of a user that won in the giveaway; 0 if none.
  int53 user_id_;
  /// The created Telegram Premium gift code if it was used by the user or can be claimed by the current user; an empty string otherwise; for Telegram Premium giveways only.
  string gift_code_;
  /// Number of Telegram Stars distributed among winners of the giveaway.
  int53 star_count_;
  /// Identifier of the corresponding giveaway message; can be an identifier of a deleted message.
  int53 giveaway_message_id_;
  /// True, if the winner for the corresponding giveaway prize wasn't chosen, because there were not enough participants.
  bool is_unclaimed_;

  /**
   * The chat created a giveaway.
   */
  chatBoostSourceGiveaway();

  /**
   * The chat created a giveaway.
   *
   * \param[in] user_id_ Identifier of a user that won in the giveaway; 0 if none.
   * \param[in] gift_code_ The created Telegram Premium gift code if it was used by the user or can be claimed by the current user; an empty string otherwise; for Telegram Premium giveways only.
   * \param[in] star_count_ Number of Telegram Stars distributed among winners of the giveaway.
   * \param[in] giveaway_message_id_ Identifier of the corresponding giveaway message; can be an identifier of a deleted message.
   * \param[in] is_unclaimed_ True, if the winner for the corresponding giveaway prize wasn't chosen, because there were not enough participants.
   */
  chatBoostSourceGiveaway(int53 user_id_, string const &gift_code_, int53 star_count_, int53 giveaway_message_id_, bool is_unclaimed_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1918145690;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A user with Telegram Premium subscription or gifted Telegram Premium boosted the chat.
 */
class chatBoostSourcePremium final : public ChatBoostSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the user.
  int53 user_id_;

  /**
   * A user with Telegram Premium subscription or gifted Telegram Premium boosted the chat.
   */
  chatBoostSourcePremium();

  /**
   * A user with Telegram Premium subscription or gifted Telegram Premium boosted the chat.
   *
   * \param[in] user_id_ Identifier of the user.
   */
  explicit chatBoostSourcePremium(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 972011;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class prepaidGiveaway;

/**
 * Describes current boost status of a chat.
 */
class chatBoostStatus final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// An HTTP URL, which can be used to boost the chat.
  string boost_url_;
  /// Identifiers of boost slots of the current user applied to the chat.
  array<int32> applied_slot_ids_;
  /// Current boost level of the chat.
  int32 level_;
  /// The number of boosts received by the chat from created Telegram Premium gift codes and giveaways; always 0 if the current user isn't an administrator in the chat.
  int32 gift_code_boost_count_;
  /// The number of boosts received by the chat.
  int32 boost_count_;
  /// The number of boosts added to reach the current level.
  int32 current_level_boost_count_;
  /// The number of boosts needed to reach the next level; 0 if the next level isn't available.
  int32 next_level_boost_count_;
  /// Approximate number of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
  int32 premium_member_count_;
  /// A percentage of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
  double premium_member_percentage_;
  /// The list of prepaid giveaways available for the chat; only for chat administrators.
  array<object_ptr<prepaidGiveaway>> prepaid_giveaways_;

  /**
   * Describes current boost status of a chat.
   */
  chatBoostStatus();

  /**
   * Describes current boost status of a chat.
   *
   * \param[in] boost_url_ An HTTP URL, which can be used to boost the chat.
   * \param[in] applied_slot_ids_ Identifiers of boost slots of the current user applied to the chat.
   * \param[in] level_ Current boost level of the chat.
   * \param[in] gift_code_boost_count_ The number of boosts received by the chat from created Telegram Premium gift codes and giveaways; always 0 if the current user isn't an administrator in the chat.
   * \param[in] boost_count_ The number of boosts received by the chat.
   * \param[in] current_level_boost_count_ The number of boosts added to reach the current level.
   * \param[in] next_level_boost_count_ The number of boosts needed to reach the next level; 0 if the next level isn't available.
   * \param[in] premium_member_count_ Approximate number of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
   * \param[in] premium_member_percentage_ A percentage of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
   * \param[in] prepaid_giveaways_ The list of prepaid giveaways available for the chat; only for chat administrators.
   */
  chatBoostStatus(string const &boost_url_, array<int32> &&applied_slot_ids_, int32 level_, int32 gift_code_boost_count_, int32 boost_count_, int32 current_level_boost_count_, int32 next_level_boost_count_, int32 premium_member_count_, double premium_member_percentage_, array<object_ptr<prepaidGiveaway>> &&prepaid_giveaways_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1050332618;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatEventAction;

class MessageSender;

/**
 * Represents a chat event.
 */
class chatEvent final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat event identifier.
  int64 id_;
  /// Point in time (Unix timestamp) when the event happened.
  int32 date_;
  /// Identifier of the user or chat who performed the action.
  object_ptr<MessageSender> member_id_;
  /// The action.
  object_ptr<ChatEventAction> action_;

  /**
   * Represents a chat event.
   */
  chatEvent();

  /**
   * Represents a chat event.
   *
   * \param[in] id_ Chat event identifier.
   * \param[in] date_ Point in time (Unix timestamp) when the event happened.
   * \param[in] member_id_ Identifier of the user or chat who performed the action.
   * \param[in] action_ The action.
   */
  chatEvent(int64 id_, int32 date_, object_ptr<MessageSender> &&member_id_, object_ptr<ChatEventAction> &&action_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -652102704;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatAvailableReactions;

class ChatMemberStatus;

class MessageSender;

class chatBackground;

class chatInviteLink;

class chatLocation;

class chatPermissions;

class chatPhoto;

class emojiStatus;

class forumTopicInfo;

class message;

/**
 * This class is an abstract base class.
 * Represents a chat event.
 */
class ChatEventAction: public Object {
 public:
};

/**
 * A message was edited.
 */
class chatEventMessageEdited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The original message before the edit.
  object_ptr<message> old_message_;
  /// The message after it was edited.
  object_ptr<message> new_message_;

  /**
   * A message was edited.
   */
  chatEventMessageEdited();

  /**
   * A message was edited.
   *
   * \param[in] old_message_ The original message before the edit.
   * \param[in] new_message_ The message after it was edited.
   */
  chatEventMessageEdited(object_ptr<message> &&old_message_, object_ptr<message> &&new_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -430967304;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message was deleted.
 */
class chatEventMessageDeleted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Deleted message.
  object_ptr<message> message_;
  /// True, if the message deletion can be reported via reportSupergroupAntiSpamFalsePositive.
  bool can_report_anti_spam_false_positive_;

  /**
   * A message was deleted.
   */
  chatEventMessageDeleted();

  /**
   * A message was deleted.
   *
   * \param[in] message_ Deleted message.
   * \param[in] can_report_anti_spam_false_positive_ True, if the message deletion can be reported via reportSupergroupAntiSpamFalsePositive.
   */
  chatEventMessageDeleted(object_ptr<message> &&message_, bool can_report_anti_spam_false_positive_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 935316851;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message was pinned.
 */
class chatEventMessagePinned final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Pinned message.
  object_ptr<message> message_;

  /**
   * A message was pinned.
   */
  chatEventMessagePinned();

  /**
   * A message was pinned.
   *
   * \param[in] message_ Pinned message.
   */
  explicit chatEventMessagePinned(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 438742298;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message was unpinned.
 */
class chatEventMessageUnpinned final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unpinned message.
  object_ptr<message> message_;

  /**
   * A message was unpinned.
   */
  chatEventMessageUnpinned();

  /**
   * A message was unpinned.
   *
   * \param[in] message_ Unpinned message.
   */
  explicit chatEventMessageUnpinned(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -376161513;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A poll in a message was stopped.
 */
class chatEventPollStopped final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The message with the poll.
  object_ptr<message> message_;

  /**
   * A poll in a message was stopped.
   */
  chatEventPollStopped();

  /**
   * A poll in a message was stopped.
   *
   * \param[in] message_ The message with the poll.
   */
  explicit chatEventPollStopped(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2009893861;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member joined the chat.
 */
class chatEventMemberJoined final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A new member joined the chat.
   */
  chatEventMemberJoined();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -235468508;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member joined the chat via an invite link.
 */
class chatEventMemberJoinedByInviteLink final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Invite link used to join the chat.
  object_ptr<chatInviteLink> invite_link_;
  /// True, if the user has joined the chat using an invite link for a chat folder.
  bool via_chat_folder_invite_link_;

  /**
   * A new member joined the chat via an invite link.
   */
  chatEventMemberJoinedByInviteLink();

  /**
   * A new member joined the chat via an invite link.
   *
   * \param[in] invite_link_ Invite link used to join the chat.
   * \param[in] via_chat_folder_invite_link_ True, if the user has joined the chat using an invite link for a chat folder.
   */
  chatEventMemberJoinedByInviteLink(object_ptr<chatInviteLink> &&invite_link_, bool via_chat_folder_invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1445536390;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new member was accepted to the chat by an administrator.
 */
class chatEventMemberJoinedByRequest final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the chat administrator, approved user join request.
  int53 approver_user_id_;
  /// Invite link used to join the chat; may be null.
  object_ptr<chatInviteLink> invite_link_;

  /**
   * A new member was accepted to the chat by an administrator.
   */
  chatEventMemberJoinedByRequest();

  /**
   * A new member was accepted to the chat by an administrator.
   *
   * \param[in] approver_user_id_ User identifier of the chat administrator, approved user join request.
   * \param[in] invite_link_ Invite link used to join the chat; may be null.
   */
  chatEventMemberJoinedByRequest(int53 approver_user_id_, object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1647804865;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new chat member was invited.
 */
class chatEventMemberInvited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New member user identifier.
  int53 user_id_;
  /// New member status.
  object_ptr<ChatMemberStatus> status_;

  /**
   * A new chat member was invited.
   */
  chatEventMemberInvited();

  /**
   * A new chat member was invited.
   *
   * \param[in] user_id_ New member user identifier.
   * \param[in] status_ New member status.
   */
  chatEventMemberInvited(int53 user_id_, object_ptr<ChatMemberStatus> &&status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 953663433;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A member left the chat.
 */
class chatEventMemberLeft final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A member left the chat.
   */
  chatEventMemberLeft();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -948420593;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
 */
class chatEventMemberPromoted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Affected chat member user identifier.
  int53 user_id_;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;

  /**
   * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
   */
  chatEventMemberPromoted();

  /**
   * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
   *
   * \param[in] user_id_ Affected chat member user identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberPromoted(int53 user_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 525297761;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
 */
class chatEventMemberRestricted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Affected chat member identifier.
  object_ptr<MessageSender> member_id_;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;

  /**
   * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
   */
  chatEventMemberRestricted();

  /**
   * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
   *
   * \param[in] member_id_ Affected chat member identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberRestricted(object_ptr<MessageSender> &&member_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1603608069;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat member extended their subscription to the chat.
 */
class chatEventMemberSubscriptionExtended final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Affected chat member user identifier.
  int53 user_id_;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;

  /**
   * A chat member extended their subscription to the chat.
   */
  chatEventMemberSubscriptionExtended();

  /**
   * A chat member extended their subscription to the chat.
   *
   * \param[in] user_id_ Affected chat member user identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberSubscriptionExtended(int53 user_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1141198846;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat available reactions were changed.
 */
class chatEventAvailableReactionsChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous chat available reactions.
  object_ptr<ChatAvailableReactions> old_available_reactions_;
  /// New chat available reactions.
  object_ptr<ChatAvailableReactions> new_available_reactions_;

  /**
   * The chat available reactions were changed.
   */
  chatEventAvailableReactionsChanged();

  /**
   * The chat available reactions were changed.
   *
   * \param[in] old_available_reactions_ Previous chat available reactions.
   * \param[in] new_available_reactions_ New chat available reactions.
   */
  chatEventAvailableReactionsChanged(object_ptr<ChatAvailableReactions> &&old_available_reactions_, object_ptr<ChatAvailableReactions> &&new_available_reactions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1749491521;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat background was changed.
 */
class chatEventBackgroundChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous background; may be null if none.
  object_ptr<chatBackground> old_background_;
  /// New background; may be null if none.
  object_ptr<chatBackground> new_background_;

  /**
   * The chat background was changed.
   */
  chatEventBackgroundChanged();

  /**
   * The chat background was changed.
   *
   * \param[in] old_background_ Previous background; may be null if none.
   * \param[in] new_background_ New background; may be null if none.
   */
  chatEventBackgroundChanged(object_ptr<chatBackground> &&old_background_, object_ptr<chatBackground> &&new_background_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1225953992;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat description was changed.
 */
class chatEventDescriptionChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous chat description.
  string old_description_;
  /// New chat description.
  string new_description_;

  /**
   * The chat description was changed.
   */
  chatEventDescriptionChanged();

  /**
   * The chat description was changed.
   *
   * \param[in] old_description_ Previous chat description.
   * \param[in] new_description_ New chat description.
   */
  chatEventDescriptionChanged(string const &old_description_, string const &new_description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 39112478;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat emoji status was changed.
 */
class chatEventEmojiStatusChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous emoji status; may be null if none.
  object_ptr<emojiStatus> old_emoji_status_;
  /// New emoji status; may be null if none.
  object_ptr<emojiStatus> new_emoji_status_;

  /**
   * The chat emoji status was changed.
   */
  chatEventEmojiStatusChanged();

  /**
   * The chat emoji status was changed.
   *
   * \param[in] old_emoji_status_ Previous emoji status; may be null if none.
   * \param[in] new_emoji_status_ New emoji status; may be null if none.
   */
  chatEventEmojiStatusChanged(object_ptr<emojiStatus> &&old_emoji_status_, object_ptr<emojiStatus> &&new_emoji_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2081850594;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The linked chat of a supergroup was changed.
 */
class chatEventLinkedChatChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous supergroup linked chat identifier.
  int53 old_linked_chat_id_;
  /// New supergroup linked chat identifier.
  int53 new_linked_chat_id_;

  /**
   * The linked chat of a supergroup was changed.
   */
  chatEventLinkedChatChanged();

  /**
   * The linked chat of a supergroup was changed.
   *
   * \param[in] old_linked_chat_id_ Previous supergroup linked chat identifier.
   * \param[in] new_linked_chat_id_ New supergroup linked chat identifier.
   */
  chatEventLinkedChatChanged(int53 old_linked_chat_id_, int53 new_linked_chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1797419439;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The supergroup location was changed.
 */
class chatEventLocationChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous location; may be null.
  object_ptr<chatLocation> old_location_;
  /// New location; may be null.
  object_ptr<chatLocation> new_location_;

  /**
   * The supergroup location was changed.
   */
  chatEventLocationChanged();

  /**
   * The supergroup location was changed.
   *
   * \param[in] old_location_ Previous location; may be null.
   * \param[in] new_location_ New location; may be null.
   */
  chatEventLocationChanged(object_ptr<chatLocation> &&old_location_, object_ptr<chatLocation> &&new_location_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -405930674;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The message auto-delete timer was changed.
 */
class chatEventMessageAutoDeleteTimeChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous value of message_auto_delete_time.
  int32 old_message_auto_delete_time_;
  /// New value of message_auto_delete_time.
  int32 new_message_auto_delete_time_;

  /**
   * The message auto-delete timer was changed.
   */
  chatEventMessageAutoDeleteTimeChanged();

  /**
   * The message auto-delete timer was changed.
   *
   * \param[in] old_message_auto_delete_time_ Previous value of message_auto_delete_time.
   * \param[in] new_message_auto_delete_time_ New value of message_auto_delete_time.
   */
  chatEventMessageAutoDeleteTimeChanged(int32 old_message_auto_delete_time_, int32 new_message_auto_delete_time_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 17317668;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat permissions were changed.
 */
class chatEventPermissionsChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous chat permissions.
  object_ptr<chatPermissions> old_permissions_;
  /// New chat permissions.
  object_ptr<chatPermissions> new_permissions_;

  /**
   * The chat permissions were changed.
   */
  chatEventPermissionsChanged();

  /**
   * The chat permissions were changed.
   *
   * \param[in] old_permissions_ Previous chat permissions.
   * \param[in] new_permissions_ New chat permissions.
   */
  chatEventPermissionsChanged(object_ptr<chatPermissions> &&old_permissions_, object_ptr<chatPermissions> &&new_permissions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1311557720;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat photo was changed.
 */
class chatEventPhotoChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous chat photo value; may be null.
  object_ptr<chatPhoto> old_photo_;
  /// New chat photo value; may be null.
  object_ptr<chatPhoto> new_photo_;

  /**
   * The chat photo was changed.
   */
  chatEventPhotoChanged();

  /**
   * The chat photo was changed.
   *
   * \param[in] old_photo_ Previous chat photo value; may be null.
   * \param[in] new_photo_ New chat photo value; may be null.
   */
  chatEventPhotoChanged(object_ptr<chatPhoto> &&old_photo_, object_ptr<chatPhoto> &&new_photo_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -811572541;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The slow_mode_delay setting of a supergroup was changed.
 */
class chatEventSlowModeDelayChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous value of slow_mode_delay, in seconds.
  int32 old_slow_mode_delay_;
  /// New value of slow_mode_delay, in seconds.
  int32 new_slow_mode_delay_;

  /**
   * The slow_mode_delay setting of a supergroup was changed.
   */
  chatEventSlowModeDelayChanged();

  /**
   * The slow_mode_delay setting of a supergroup was changed.
   *
   * \param[in] old_slow_mode_delay_ Previous value of slow_mode_delay, in seconds.
   * \param[in] new_slow_mode_delay_ New value of slow_mode_delay, in seconds.
   */
  chatEventSlowModeDelayChanged(int32 old_slow_mode_delay_, int32 new_slow_mode_delay_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1653195765;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The supergroup sticker set was changed.
 */
class chatEventStickerSetChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous identifier of the chat sticker set; 0 if none.
  int64 old_sticker_set_id_;
  /// New identifier of the chat sticker set; 0 if none.
  int64 new_sticker_set_id_;

  /**
   * The supergroup sticker set was changed.
   */
  chatEventStickerSetChanged();

  /**
   * The supergroup sticker set was changed.
   *
   * \param[in] old_sticker_set_id_ Previous identifier of the chat sticker set; 0 if none.
   * \param[in] new_sticker_set_id_ New identifier of the chat sticker set; 0 if none.
   */
  chatEventStickerSetChanged(int64 old_sticker_set_id_, int64 new_sticker_set_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1243130481;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The supergroup sticker set with allowed custom emoji was changed.
 */
class chatEventCustomEmojiStickerSetChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous identifier of the chat sticker set; 0 if none.
  int64 old_sticker_set_id_;
  /// New identifier of the chat sticker set; 0 if none.
  int64 new_sticker_set_id_;

  /**
   * The supergroup sticker set with allowed custom emoji was changed.
   */
  chatEventCustomEmojiStickerSetChanged();

  /**
   * The supergroup sticker set with allowed custom emoji was changed.
   *
   * \param[in] old_sticker_set_id_ Previous identifier of the chat sticker set; 0 if none.
   * \param[in] new_sticker_set_id_ New identifier of the chat sticker set; 0 if none.
   */
  chatEventCustomEmojiStickerSetChanged(int64 old_sticker_set_id_, int64 new_sticker_set_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 118244123;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat title was changed.
 */
class chatEventTitleChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous chat title.
  string old_title_;
  /// New chat title.
  string new_title_;

  /**
   * The chat title was changed.
   */
  chatEventTitleChanged();

  /**
   * The chat title was changed.
   *
   * \param[in] old_title_ Previous chat title.
   * \param[in] new_title_ New chat title.
   */
  chatEventTitleChanged(string const &old_title_, string const &new_title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1134103250;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat editable username was changed.
 */
class chatEventUsernameChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous chat username.
  string old_username_;
  /// New chat username.
  string new_username_;

  /**
   * The chat editable username was changed.
   */
  chatEventUsernameChanged();

  /**
   * The chat editable username was changed.
   *
   * \param[in] old_username_ Previous chat username.
   * \param[in] new_username_ New chat username.
   */
  chatEventUsernameChanged(string const &old_username_, string const &new_username_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1728558443;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat active usernames were changed.
 */
class chatEventActiveUsernamesChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous list of active usernames.
  array<string> old_usernames_;
  /// New list of active usernames.
  array<string> new_usernames_;

  /**
   * The chat active usernames were changed.
   */
  chatEventActiveUsernamesChanged();

  /**
   * The chat active usernames were changed.
   *
   * \param[in] old_usernames_ Previous list of active usernames.
   * \param[in] new_usernames_ New list of active usernames.
   */
  chatEventActiveUsernamesChanged(array<string> &&old_usernames_, array<string> &&new_usernames_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1508790810;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat accent color or background custom emoji were changed.
 */
class chatEventAccentColorChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous identifier of chat accent color.
  int32 old_accent_color_id_;
  /// Previous identifier of the custom emoji; 0 if none.
  int64 old_background_custom_emoji_id_;
  /// New identifier of chat accent color.
  int32 new_accent_color_id_;
  /// New identifier of the custom emoji; 0 if none.
  int64 new_background_custom_emoji_id_;

  /**
   * The chat accent color or background custom emoji were changed.
   */
  chatEventAccentColorChanged();

  /**
   * The chat accent color or background custom emoji were changed.
   *
   * \param[in] old_accent_color_id_ Previous identifier of chat accent color.
   * \param[in] old_background_custom_emoji_id_ Previous identifier of the custom emoji; 0 if none.
   * \param[in] new_accent_color_id_ New identifier of chat accent color.
   * \param[in] new_background_custom_emoji_id_ New identifier of the custom emoji; 0 if none.
   */
  chatEventAccentColorChanged(int32 old_accent_color_id_, int64 old_background_custom_emoji_id_, int32 new_accent_color_id_, int64 new_background_custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -427591885;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat's profile accent color or profile background custom emoji were changed.
 */
class chatEventProfileAccentColorChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous identifier of chat's profile accent color; -1 if none.
  int32 old_profile_accent_color_id_;
  /// Previous identifier of the custom emoji; 0 if none.
  int64 old_profile_background_custom_emoji_id_;
  /// New identifier of chat's profile accent color; -1 if none.
  int32 new_profile_accent_color_id_;
  /// New identifier of the custom emoji; 0 if none.
  int64 new_profile_background_custom_emoji_id_;

  /**
   * The chat's profile accent color or profile background custom emoji were changed.
   */
  chatEventProfileAccentColorChanged();

  /**
   * The chat's profile accent color or profile background custom emoji were changed.
   *
   * \param[in] old_profile_accent_color_id_ Previous identifier of chat's profile accent color; -1 if none.
   * \param[in] old_profile_background_custom_emoji_id_ Previous identifier of the custom emoji; 0 if none.
   * \param[in] new_profile_accent_color_id_ New identifier of chat's profile accent color; -1 if none.
   * \param[in] new_profile_background_custom_emoji_id_ New identifier of the custom emoji; 0 if none.
   */
  chatEventProfileAccentColorChanged(int32 old_profile_accent_color_id_, int64 old_profile_background_custom_emoji_id_, int32 new_profile_accent_color_id_, int64 new_profile_background_custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1514612124;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The has_protected_content setting of a channel was toggled.
 */
class chatEventHasProtectedContentToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of has_protected_content.
  bool has_protected_content_;

  /**
   * The has_protected_content setting of a channel was toggled.
   */
  chatEventHasProtectedContentToggled();

  /**
   * The has_protected_content setting of a channel was toggled.
   *
   * \param[in] has_protected_content_ New value of has_protected_content.
   */
  explicit chatEventHasProtectedContentToggled(bool has_protected_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -184270335;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The can_invite_users permission of a supergroup chat was toggled.
 */
class chatEventInvitesToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of can_invite_users permission.
  bool can_invite_users_;

  /**
   * The can_invite_users permission of a supergroup chat was toggled.
   */
  chatEventInvitesToggled();

  /**
   * The can_invite_users permission of a supergroup chat was toggled.
   *
   * \param[in] can_invite_users_ New value of can_invite_users permission.
   */
  explicit chatEventInvitesToggled(bool can_invite_users_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -62548373;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The is_all_history_available setting of a supergroup was toggled.
 */
class chatEventIsAllHistoryAvailableToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of is_all_history_available.
  bool is_all_history_available_;

  /**
   * The is_all_history_available setting of a supergroup was toggled.
   */
  chatEventIsAllHistoryAvailableToggled();

  /**
   * The is_all_history_available setting of a supergroup was toggled.
   *
   * \param[in] is_all_history_available_ New value of is_all_history_available.
   */
  explicit chatEventIsAllHistoryAvailableToggled(bool is_all_history_available_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1599063019;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The has_aggressive_anti_spam_enabled setting of a supergroup was toggled.
 */
class chatEventHasAggressiveAntiSpamEnabledToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of has_aggressive_anti_spam_enabled.
  bool has_aggressive_anti_spam_enabled_;

  /**
   * The has_aggressive_anti_spam_enabled setting of a supergroup was toggled.
   */
  chatEventHasAggressiveAntiSpamEnabledToggled();

  /**
   * The has_aggressive_anti_spam_enabled setting of a supergroup was toggled.
   *
   * \param[in] has_aggressive_anti_spam_enabled_ New value of has_aggressive_anti_spam_enabled.
   */
  explicit chatEventHasAggressiveAntiSpamEnabledToggled(bool has_aggressive_anti_spam_enabled_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -125348094;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The sign_messages setting of a channel was toggled.
 */
class chatEventSignMessagesToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of sign_messages.
  bool sign_messages_;

  /**
   * The sign_messages setting of a channel was toggled.
   */
  chatEventSignMessagesToggled();

  /**
   * The sign_messages setting of a channel was toggled.
   *
   * \param[in] sign_messages_ New value of sign_messages.
   */
  explicit chatEventSignMessagesToggled(bool sign_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1313265634;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The show_message_sender setting of a channel was toggled.
 */
class chatEventShowMessageSenderToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of show_message_sender.
  bool show_message_sender_;

  /**
   * The show_message_sender setting of a channel was toggled.
   */
  chatEventShowMessageSenderToggled();

  /**
   * The show_message_sender setting of a channel was toggled.
   *
   * \param[in] show_message_sender_ New value of show_message_sender.
   */
  explicit chatEventShowMessageSenderToggled(bool show_message_sender_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -794343453;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat invite link was edited.
 */
class chatEventInviteLinkEdited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Previous information about the invite link.
  object_ptr<chatInviteLink> old_invite_link_;
  /// New information about the invite link.
  object_ptr<chatInviteLink> new_invite_link_;

  /**
   * A chat invite link was edited.
   */
  chatEventInviteLinkEdited();

  /**
   * A chat invite link was edited.
   *
   * \param[in] old_invite_link_ Previous information about the invite link.
   * \param[in] new_invite_link_ New information about the invite link.
   */
  chatEventInviteLinkEdited(object_ptr<chatInviteLink> &&old_invite_link_, object_ptr<chatInviteLink> &&new_invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -460190366;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A chat invite link was revoked.
 */
class chatEventInviteLinkRevoked final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The invite link.
  object_ptr<chatInviteLink> invite_link_;

  /**
   * A chat invite link was revoked.
   */
  chatEventInviteLinkRevoked();

  /**
   * A chat invite link was revoked.
   *
   * \param[in] invite_link_ The invite link.
   */
  explicit chatEventInviteLinkRevoked(object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1579417629;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A revoked chat invite link was deleted.
 */
class chatEventInviteLinkDeleted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The invite link.
  object_ptr<chatInviteLink> invite_link_;

  /**
   * A revoked chat invite link was deleted.
   */
  chatEventInviteLinkDeleted();

  /**
   * A revoked chat invite link was deleted.
   *
   * \param[in] invite_link_ The invite link.
   */
  explicit chatEventInviteLinkDeleted(object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1394974361;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat was created.
 */
class chatEventVideoChatCreated final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;

  /**
   * A video chat was created.
   */
  chatEventVideoChatCreated();

  /**
   * A video chat was created.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   */
  explicit chatEventVideoChatCreated(int32 group_call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1822853755;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat was ended.
 */
class chatEventVideoChatEnded final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;

  /**
   * A video chat was ended.
   */
  chatEventVideoChatEnded();

  /**
   * A video chat was ended.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   */
  explicit chatEventVideoChatEnded(int32 group_call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1630039112;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The mute_new_participants setting of a video chat was toggled.
 */
class chatEventVideoChatMuteNewParticipantsToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of the mute_new_participants setting.
  bool mute_new_participants_;

  /**
   * The mute_new_participants setting of a video chat was toggled.
   */
  chatEventVideoChatMuteNewParticipantsToggled();

  /**
   * The mute_new_participants setting of a video chat was toggled.
   *
   * \param[in] mute_new_participants_ New value of the mute_new_participants setting.
   */
  explicit chatEventVideoChatMuteNewParticipantsToggled(bool mute_new_participants_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -126547970;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat participant was muted or unmuted.
 */
class chatEventVideoChatParticipantIsMutedToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the affected group call participant.
  object_ptr<MessageSender> participant_id_;
  /// New value of is_muted.
  bool is_muted_;

  /**
   * A video chat participant was muted or unmuted.
   */
  chatEventVideoChatParticipantIsMutedToggled();

  /**
   * A video chat participant was muted or unmuted.
   *
   * \param[in] participant_id_ Identifier of the affected group call participant.
   * \param[in] is_muted_ New value of is_muted.
   */
  chatEventVideoChatParticipantIsMutedToggled(object_ptr<MessageSender> &&participant_id_, bool is_muted_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 521165047;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video chat participant volume level was changed.
 */
class chatEventVideoChatParticipantVolumeLevelChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the affected group call participant.
  object_ptr<MessageSender> participant_id_;
  /// New value of volume_level; 1-20000 in hundreds of percents.
  int32 volume_level_;

  /**
   * A video chat participant volume level was changed.
   */
  chatEventVideoChatParticipantVolumeLevelChanged();

  /**
   * A video chat participant volume level was changed.
   *
   * \param[in] participant_id_ Identifier of the affected group call participant.
   * \param[in] volume_level_ New value of volume_level; 1-20000 in hundreds of percents.
   */
  chatEventVideoChatParticipantVolumeLevelChanged(object_ptr<MessageSender> &&participant_id_, int32 volume_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1131385534;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The is_forum setting of a channel was toggled.
 */
class chatEventIsForumToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New value of is_forum.
  bool is_forum_;

  /**
   * The is_forum setting of a channel was toggled.
   */
  chatEventIsForumToggled();

  /**
   * The is_forum setting of a channel was toggled.
   *
   * \param[in] is_forum_ New value of is_forum.
   */
  explicit chatEventIsForumToggled(bool is_forum_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1516491033;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A new forum topic was created.
 */
class chatEventForumTopicCreated final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about the topic.
  object_ptr<forumTopicInfo> topic_info_;

  /**
   * A new forum topic was created.
   */
  chatEventForumTopicCreated();

  /**
   * A new forum topic was created.
   *
   * \param[in] topic_info_ Information about the topic.
   */
  explicit chatEventForumTopicCreated(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2005269314;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A forum topic was edited.
 */
class chatEventForumTopicEdited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Old information about the topic.
  object_ptr<forumTopicInfo> old_topic_info_;
  /// New information about the topic.
  object_ptr<forumTopicInfo> new_topic_info_;

  /**
   * A forum topic was edited.
   */
  chatEventForumTopicEdited();

  /**
   * A forum topic was edited.
   *
   * \param[in] old_topic_info_ Old information about the topic.
   * \param[in] new_topic_info_ New information about the topic.
   */
  chatEventForumTopicEdited(object_ptr<forumTopicInfo> &&old_topic_info_, object_ptr<forumTopicInfo> &&new_topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1624910860;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A forum topic was closed or reopened.
 */
class chatEventForumTopicToggleIsClosed final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New information about the topic.
  object_ptr<forumTopicInfo> topic_info_;

  /**
   * A forum topic was closed or reopened.
   */
  chatEventForumTopicToggleIsClosed();

  /**
   * A forum topic was closed or reopened.
   *
   * \param[in] topic_info_ New information about the topic.
   */
  explicit chatEventForumTopicToggleIsClosed(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -962704070;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The General forum topic was hidden or unhidden.
 */
class chatEventForumTopicToggleIsHidden final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// New information about the topic.
  object_ptr<forumTopicInfo> topic_info_;

  /**
   * The General forum topic was hidden or unhidden.
   */
  chatEventForumTopicToggleIsHidden();

  /**
   * The General forum topic was hidden or unhidden.
   *
   * \param[in] topic_info_ New information about the topic.
   */
  explicit chatEventForumTopicToggleIsHidden(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1609175250;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A forum topic was deleted.
 */
class chatEventForumTopicDeleted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about the topic.
  object_ptr<forumTopicInfo> topic_info_;

  /**
   * A forum topic was deleted.
   */
  chatEventForumTopicDeleted();

  /**
   * A forum topic was deleted.
   *
   * \param[in] topic_info_ Information about the topic.
   */
  explicit chatEventForumTopicDeleted(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1332795123;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A pinned forum topic was changed.
 */
class chatEventForumTopicPinned final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about the old pinned topic; may be null.
  object_ptr<forumTopicInfo> old_topic_info_;
  /// Information about the new pinned topic; may be null.
  object_ptr<forumTopicInfo> new_topic_info_;

  /**
   * A pinned forum topic was changed.
   */
  chatEventForumTopicPinned();

  /**
   * A pinned forum topic was changed.
   *
   * \param[in] old_topic_info_ Information about the old pinned topic; may be null.
   * \param[in] new_topic_info_ Information about the new pinned topic; may be null.
   */
  chatEventForumTopicPinned(object_ptr<forumTopicInfo> &&old_topic_info_, object_ptr<forumTopicInfo> &&new_topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2143626222;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a set of filters used to obtain a chat event log.
 */
class chatEventLogFilters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if message edits need to be returned.
  bool message_edits_;
  /// True, if message deletions need to be returned.
  bool message_deletions_;
  /// True, if pin/unpin events need to be returned.
  bool message_pins_;
  /// True, if members joining events need to be returned.
  bool member_joins_;
  /// True, if members leaving events need to be returned.
  bool member_leaves_;
  /// True, if invited member events need to be returned.
  bool member_invites_;
  /// True, if member promotion/demotion events need to be returned.
  bool member_promotions_;
  /// True, if member restricted/unrestricted/banned/unbanned events need to be returned.
  bool member_restrictions_;
  /// True, if changes in chat information need to be returned.
  bool info_changes_;
  /// True, if changes in chat settings need to be returned.
  bool setting_changes_;
  /// True, if changes to invite links need to be returned.
  bool invite_link_changes_;
  /// True, if video chat actions need to be returned.
  bool video_chat_changes_;
  /// True, if forum-related actions need to be returned.
  bool forum_changes_;
  /// True, if subscription extensions need to be returned.
  bool subscription_extensions_;

  /**
   * Represents a set of filters used to obtain a chat event log.
   */
  chatEventLogFilters();

  /**
   * Represents a set of filters used to obtain a chat event log.
   *
   * \param[in] message_edits_ True, if message edits need to be returned.
   * \param[in] message_deletions_ True, if message deletions need to be returned.
   * \param[in] message_pins_ True, if pin/unpin events need to be returned.
   * \param[in] member_joins_ True, if members joining events need to be returned.
   * \param[in] member_leaves_ True, if members leaving events need to be returned.
   * \param[in] member_invites_ True, if invited member events need to be returned.
   * \param[in] member_promotions_ True, if member promotion/demotion events need to be returned.
   * \param[in] member_restrictions_ True, if member restricted/unrestricted/banned/unbanned events need to be returned.
   * \param[in] info_changes_ True, if changes in chat information need to be returned.
   * \param[in] setting_changes_ True, if changes in chat settings need to be returned.
   * \param[in] invite_link_changes_ True, if changes to invite links need to be returned.
   * \param[in] video_chat_changes_ True, if video chat actions need to be returned.
   * \param[in] forum_changes_ True, if forum-related actions need to be returned.
   * \param[in] subscription_extensions_ True, if subscription extensions need to be returned.
   */
  chatEventLogFilters(bool message_edits_, bool message_deletions_, bool message_pins_, bool member_joins_, bool member_leaves_, bool member_invites_, bool member_promotions_, bool member_restrictions_, bool info_changes_, bool setting_changes_, bool invite_link_changes_, bool video_chat_changes_, bool forum_changes_, bool subscription_extensions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032965711;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatEvent;

/**
 * Contains a list of chat events.
 */
class chatEvents final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of events.
  array<object_ptr<chatEvent>> events_;

  /**
   * Contains a list of chat events.
   */
  chatEvents();

  /**
   * Contains a list of chat events.
   *
   * \param[in] events_ List of events.
   */
  explicit chatEvents(array<object_ptr<chatEvent>> &&events_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -585329664;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatFolderIcon;

/**
 * Represents a folder for user chats.
 */
class chatFolder final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The title of the folder; 1-12 characters without line feeds.
  string title_;
  /// The chosen icon for the chat folder; may be null. If null, use getChatFolderDefaultIconName to get default icon name for the folder.
  object_ptr<chatFolderIcon> icon_;
  /// The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled. Can't be changed if folder tags are disabled or the current user doesn't have Telegram Premium subscription.
  int32 color_id_;
  /// True, if at least one link has been created for the folder.
  bool is_shareable_;
  /// The chat identifiers of pinned chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
  array<int53> pinned_chat_ids_;
  /// The chat identifiers of always included chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
  array<int53> included_chat_ids_;
  /// The chat identifiers of always excluded chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) always excluded non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
  array<int53> excluded_chat_ids_;
  /// True, if muted chats need to be excluded.
  bool exclude_muted_;
  /// True, if read chats need to be excluded.
  bool exclude_read_;
  /// True, if archived chats need to be excluded.
  bool exclude_archived_;
  /// True, if contacts need to be included.
  bool include_contacts_;
  /// True, if non-contact users need to be included.
  bool include_non_contacts_;
  /// True, if bots need to be included.
  bool include_bots_;
  /// True, if basic groups and supergroups need to be included.
  bool include_groups_;
  /// True, if channels need to be included.
  bool include_channels_;

  /**
   * Represents a folder for user chats.
   */
  chatFolder();

  /**
   * Represents a folder for user chats.
   *
   * \param[in] title_ The title of the folder; 1-12 characters without line feeds.
   * \param[in] icon_ The chosen icon for the chat folder; may be null. If null, use getChatFolderDefaultIconName to get default icon name for the folder.
   * \param[in] color_id_ The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled. Can't be changed if folder tags are disabled or the current user doesn't have Telegram Premium subscription.
   * \param[in] is_shareable_ True, if at least one link has been created for the folder.
   * \param[in] pinned_chat_ids_ The chat identifiers of pinned chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
   * \param[in] included_chat_ids_ The chat identifiers of always included chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
   * \param[in] excluded_chat_ids_ The chat identifiers of always excluded chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) always excluded non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
   * \param[in] exclude_muted_ True, if muted chats need to be excluded.
   * \param[in] exclude_read_ True, if read chats need to be excluded.
   * \param[in] exclude_archived_ True, if archived chats need to be excluded.
   * \param[in] include_contacts_ True, if contacts need to be included.
   * \param[in] include_non_contacts_ True, if non-contact users need to be included.
   * \param[in] include_bots_ True, if bots need to be included.
   * \param[in] include_groups_ True, if basic groups and supergroups need to be included.
   * \param[in] include_channels_ True, if channels need to be included.
   */
  chatFolder(string const &title_, object_ptr<chatFolderIcon> &&icon_, int32 color_id_, bool is_shareable_, array<int53> &&pinned_chat_ids_, array<int53> &&included_chat_ids_, array<int53> &&excluded_chat_ids_, bool exclude_muted_, bool exclude_read_, bool exclude_archived_, bool include_contacts_, bool include_non_contacts_, bool include_bots_, bool include_groups_, bool include_channels_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -474905057;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an icon for a chat folder.
 */
class chatFolderIcon final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The chosen icon name for short folder representation; one of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;, &quot;Airplane&quot;, &quot;Book&quot;, &quot;Light&quot;, &quot;Like&quot;, &quot;Money&quot;, &quot;Note&quot;, &quot;Palette&quot;.
  string name_;

  /**
   * Represents an icon for a chat folder.
   */
  chatFolderIcon();

  /**
   * Represents an icon for a chat folder.
   *
   * \param[in] name_ The chosen icon name for short folder representation; one of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;, &quot;Airplane&quot;, &quot;Book&quot;, &quot;Light&quot;, &quot;Like&quot;, &quot;Money&quot;, &quot;Note&quot;, &quot;Palette&quot;.
   */
  explicit chatFolderIcon(string const &name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -146104090;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatFolderIcon;

/**
 * Contains basic information about a chat folder.
 */
class chatFolderInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique chat folder identifier.
  int32 id_;
  /// The title of the folder; 1-12 characters without line feeds.
  string title_;
  /// The chosen or default icon for the chat folder.
  object_ptr<chatFolderIcon> icon_;
  /// The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled.
  int32 color_id_;
  /// True, if at least one link has been created for the folder.
  bool is_shareable_;
  /// True, if the chat folder has invite links created by the current user.
  bool has_my_invite_links_;

  /**
   * Contains basic information about a chat folder.
   */
  chatFolderInfo();

  /**
   * Contains basic information about a chat folder.
   *
   * \param[in] id_ Unique chat folder identifier.
   * \param[in] title_ The title of the folder; 1-12 characters without line feeds.
   * \param[in] icon_ The chosen or default icon for the chat folder.
   * \param[in] color_id_ The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled.
   * \param[in] is_shareable_ True, if at least one link has been created for the folder.
   * \param[in] has_my_invite_links_ True, if the chat folder has invite links created by the current user.
   */
  chatFolderInfo(int32 id_, string const &title_, object_ptr<chatFolderIcon> &&icon_, int32 color_id_, bool is_shareable_, bool has_my_invite_links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 190948485;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a chat folder invite link.
 */
class chatFolderInviteLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The chat folder invite link.
  string invite_link_;
  /// Name of the link.
  string name_;
  /// Identifiers of chats, included in the link.
  array<int53> chat_ids_;

  /**
   * Contains a chat folder invite link.
   */
  chatFolderInviteLink();

  /**
   * Contains a chat folder invite link.
   *
   * \param[in] invite_link_ The chat folder invite link.
   * \param[in] name_ Name of the link.
   * \param[in] chat_ids_ Identifiers of chats, included in the link.
   */
  chatFolderInviteLink(string const &invite_link_, string const &name_, array<int53> &&chat_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 493969661;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatFolderInfo;

/**
 * Contains information about an invite link to a chat folder.
 */
class chatFolderInviteLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Basic information about the chat folder; chat folder identifier will be 0 if the user didn't have the chat folder yet.
  object_ptr<chatFolderInfo> chat_folder_info_;
  /// Identifiers of the chats from the link, which aren't added to the folder yet.
  array<int53> missing_chat_ids_;
  /// Identifiers of the chats from the link, which are added to the folder already.
  array<int53> added_chat_ids_;

  /**
   * Contains information about an invite link to a chat folder.
   */
  chatFolderInviteLinkInfo();

  /**
   * Contains information about an invite link to a chat folder.
   *
   * \param[in] chat_folder_info_ Basic information about the chat folder; chat folder identifier will be 0 if the user didn't have the chat folder yet.
   * \param[in] missing_chat_ids_ Identifiers of the chats from the link, which aren't added to the folder yet.
   * \param[in] added_chat_ids_ Identifiers of the chats from the link, which are added to the folder already.
   */
  chatFolderInviteLinkInfo(object_ptr<chatFolderInfo> &&chat_folder_info_, array<int53> &&missing_chat_ids_, array<int53> &&added_chat_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119450395;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatFolderInviteLink;

/**
 * Represents a list of chat folder invite links.
 */
class chatFolderInviteLinks final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of the invite links.
  array<object_ptr<chatFolderInviteLink>> invite_links_;

  /**
   * Represents a list of chat folder invite links.
   */
  chatFolderInviteLinks();

  /**
   * Represents a list of chat folder invite links.
   *
   * \param[in] invite_links_ List of the invite links.
   */
  explicit chatFolderInviteLinks(array<object_ptr<chatFolderInviteLink>> &&invite_links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1853351525;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class starSubscriptionPricing;

/**
 * Contains a chat invite link.
 */
class chatInviteLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat invite link.
  string invite_link_;
  /// Name of the link.
  string name_;
  /// User identifier of an administrator created the link.
  int53 creator_user_id_;
  /// Point in time (Unix timestamp) when the link was created.
  int32 date_;
  /// Point in time (Unix timestamp) when the link was last edited; 0 if never or unknown.
  int32 edit_date_;
  /// Point in time (Unix timestamp) when the link will expire; 0 if never.
  int32 expiration_date_;
  /// Information about subscription plan that is applied to the users joining the chat by the link; may be null if the link doesn't require subscription.
  object_ptr<starSubscriptionPricing> subscription_pricing_;
  /// The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited. Always 0 if the link requires approval.
  int32 member_limit_;
  /// Number of chat members, which joined the chat using the link.
  int32 member_count_;
  /// Number of chat members, which joined the chat using the link, but have already left because of expired subscription; for subscription links only.
  int32 expired_member_count_;
  /// Number of pending join requests created using this link.
  int32 pending_join_request_count_;
  /// True, if the link only creates join request. If true, total number of joining members will be unlimited.
  bool creates_join_request_;
  /// True, if the link is primary. Primary invite link can't have name, expiration date, or usage limit. There is exactly one primary invite link for each administrator with can_invite_users right at a given time.
  bool is_primary_;
  /// True, if the link was revoked.
  bool is_revoked_;

  /**
   * Contains a chat invite link.
   */
  chatInviteLink();

  /**
   * Contains a chat invite link.
   *
   * \param[in] invite_link_ Chat invite link.
   * \param[in] name_ Name of the link.
   * \param[in] creator_user_id_ User identifier of an administrator created the link.
   * \param[in] date_ Point in time (Unix timestamp) when the link was created.
   * \param[in] edit_date_ Point in time (Unix timestamp) when the link was last edited; 0 if never or unknown.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the link will expire; 0 if never.
   * \param[in] subscription_pricing_ Information about subscription plan that is applied to the users joining the chat by the link; may be null if the link doesn't require subscription.
   * \param[in] member_limit_ The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited. Always 0 if the link requires approval.
   * \param[in] member_count_ Number of chat members, which joined the chat using the link.
   * \param[in] expired_member_count_ Number of chat members, which joined the chat using the link, but have already left because of expired subscription; for subscription links only.
   * \param[in] pending_join_request_count_ Number of pending join requests created using this link.
   * \param[in] creates_join_request_ True, if the link only creates join request. If true, total number of joining members will be unlimited.
   * \param[in] is_primary_ True, if the link is primary. Primary invite link can't have name, expiration date, or usage limit. There is exactly one primary invite link for each administrator with can_invite_users right at a given time.
   * \param[in] is_revoked_ True, if the link was revoked.
   */
  chatInviteLink(string const &invite_link_, string const &name_, int53 creator_user_id_, int32 date_, int32 edit_date_, int32 expiration_date_, object_ptr<starSubscriptionPricing> &&subscription_pricing_, int32 member_limit_, int32 member_count_, int32 expired_member_count_, int32 pending_join_request_count_, bool creates_join_request_, bool is_primary_, bool is_revoked_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -957651664;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat administrator with a number of active and revoked chat invite links.
 */
class chatInviteLinkCount final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Administrator's user identifier.
  int53 user_id_;
  /// Number of active invite links.
  int32 invite_link_count_;
  /// Number of revoked invite links.
  int32 revoked_invite_link_count_;

  /**
   * Describes a chat administrator with a number of active and revoked chat invite links.
   */
  chatInviteLinkCount();

  /**
   * Describes a chat administrator with a number of active and revoked chat invite links.
   *
   * \param[in] user_id_ Administrator's user identifier.
   * \param[in] invite_link_count_ Number of active invite links.
   * \param[in] revoked_invite_link_count_ Number of revoked invite links.
   */
  chatInviteLinkCount(int53 user_id_, int32 invite_link_count_, int32 revoked_invite_link_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1021999210;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatInviteLinkCount;

/**
 * Contains a list of chat invite link counts.
 */
class chatInviteLinkCounts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of invite link counts.
  array<object_ptr<chatInviteLinkCount>> invite_link_counts_;

  /**
   * Contains a list of chat invite link counts.
   */
  chatInviteLinkCounts();

  /**
   * Contains a list of chat invite link counts.
   *
   * \param[in] invite_link_counts_ List of invite link counts.
   */
  explicit chatInviteLinkCounts(array<object_ptr<chatInviteLinkCount>> &&invite_link_counts_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 920326637;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InviteLinkChatType;

class chatInviteLinkSubscriptionInfo;

class chatPhotoInfo;

/**
 * Contains information about a chat invite link.
 */
class chatInviteLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier of the invite link; 0 if the user has no access to the chat before joining.
  int53 chat_id_;
  /// If non-zero, the amount of time for which read access to the chat will remain available, in seconds.
  int32 accessible_for_;
  /// Type of the chat.
  object_ptr<InviteLinkChatType> type_;
  /// Title of the chat.
  string title_;
  /// Chat photo; may be null.
  object_ptr<chatPhotoInfo> photo_;
  /// Identifier of the accent color for chat title and background of chat photo.
  int32 accent_color_id_;
  /// Chat description.
  string description_;
  /// Number of members in the chat.
  int32 member_count_;
  /// User identifiers of some chat members that may be known to the current user.
  array<int53> member_user_ids_;
  /// Information about subscription plan that must be paid by the user to use the link; may be null if the link doesn't require subscription.
  object_ptr<chatInviteLinkSubscriptionInfo> subscription_info_;
  /// True, if the link only creates join request.
  bool creates_join_request_;
  /// True, if the chat is a public supergroup or channel, i.e. it has a username or it is a location-based supergroup.
  bool is_public_;
  /// True, if the chat is verified.
  bool is_verified_;
  /// True, if many users reported this chat as a scam.
  bool is_scam_;
  /// True, if many users reported this chat as a fake account.
  bool is_fake_;

  /**
   * Contains information about a chat invite link.
   */
  chatInviteLinkInfo();

  /**
   * Contains information about a chat invite link.
   *
   * \param[in] chat_id_ Chat identifier of the invite link; 0 if the user has no access to the chat before joining.
   * \param[in] accessible_for_ If non-zero, the amount of time for which read access to the chat will remain available, in seconds.
   * \param[in] type_ Type of the chat.
   * \param[in] title_ Title of the chat.
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] accent_color_id_ Identifier of the accent color for chat title and background of chat photo.
   * \param[in] description_ Chat description.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] member_user_ids_ User identifiers of some chat members that may be known to the current user.
   * \param[in] subscription_info_ Information about subscription plan that must be paid by the user to use the link; may be null if the link doesn't require subscription.
   * \param[in] creates_join_request_ True, if the link only creates join request.
   * \param[in] is_public_ True, if the chat is a public supergroup or channel, i.e. it has a username or it is a location-based supergroup.
   * \param[in] is_verified_ True, if the chat is verified.
   * \param[in] is_scam_ True, if many users reported this chat as a scam.
   * \param[in] is_fake_ True, if many users reported this chat as a fake account.
   */
  chatInviteLinkInfo(int53 chat_id_, int32 accessible_for_, object_ptr<InviteLinkChatType> &&type_, string const &title_, object_ptr<chatPhotoInfo> &&photo_, int32 accent_color_id_, string const &description_, int32 member_count_, array<int53> &&member_user_ids_, object_ptr<chatInviteLinkSubscriptionInfo> &&subscription_info_, bool creates_join_request_, bool is_public_, bool is_verified_, bool is_scam_, bool is_fake_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2052328938;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a chat member joined a chat via an invite link.
 */
class chatInviteLinkMember final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;
  /// Point in time (Unix timestamp) when the user joined the chat.
  int32 joined_chat_date_;
  /// True, if the user has joined the chat using an invite link for a chat folder.
  bool via_chat_folder_invite_link_;
  /// User identifier of the chat administrator, approved user join request.
  int53 approver_user_id_;

  /**
   * Describes a chat member joined a chat via an invite link.
   */
  chatInviteLinkMember();

  /**
   * Describes a chat member joined a chat via an invite link.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined the chat.
   * \param[in] via_chat_folder_invite_link_ True, if the user has joined the chat using an invite link for a chat folder.
   * \param[in] approver_user_id_ User identifier of the chat administrator, approved user join request.
   */
  chatInviteLinkMember(int53 user_id_, int32 joined_chat_date_, bool via_chat_folder_invite_link_, int53 approver_user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 29156795;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatInviteLinkMember;

/**
 * Contains a list of chat members joined a chat via an invite link.
 */
class chatInviteLinkMembers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of chat members found.
  int32 total_count_;
  /// List of chat members, joined a chat via an invite link.
  array<object_ptr<chatInviteLinkMember>> members_;

  /**
   * Contains a list of chat members joined a chat via an invite link.
   */
  chatInviteLinkMembers();

  /**
   * Contains a list of chat members joined a chat via an invite link.
   *
   * \param[in] total_count_ Approximate total number of chat members found.
   * \param[in] members_ List of chat members, joined a chat via an invite link.
   */
  chatInviteLinkMembers(int32 total_count_, array<object_ptr<chatInviteLinkMember>> &&members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 315635051;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class starSubscriptionPricing;

/**
 * Contains information about subscription plan that must be paid by the user to use a chat invite link.
 */
class chatInviteLinkSubscriptionInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about subscription plan that must be paid by the user to use the link.
  object_ptr<starSubscriptionPricing> pricing_;
  /// True, if the user has already paid for the subscription and can use joinChatByInviteLink to join the subscribed chat again.
  bool can_reuse_;
  /// Identifier of the payment form to use for subscription payment; 0 if the subscription can't be paid.
  int64 form_id_;

  /**
   * Contains information about subscription plan that must be paid by the user to use a chat invite link.
   */
  chatInviteLinkSubscriptionInfo();

  /**
   * Contains information about subscription plan that must be paid by the user to use a chat invite link.
   *
   * \param[in] pricing_ Information about subscription plan that must be paid by the user to use the link.
   * \param[in] can_reuse_ True, if the user has already paid for the subscription and can use joinChatByInviteLink to join the subscribed chat again.
   * \param[in] form_id_ Identifier of the payment form to use for subscription payment; 0 if the subscription can't be paid.
   */
  chatInviteLinkSubscriptionInfo(object_ptr<starSubscriptionPricing> &&pricing_, bool can_reuse_, int64 form_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 953119592;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatInviteLink;

/**
 * Contains a list of chat invite links.
 */
class chatInviteLinks final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of chat invite links found.
  int32 total_count_;
  /// List of invite links.
  array<object_ptr<chatInviteLink>> invite_links_;

  /**
   * Contains a list of chat invite links.
   */
  chatInviteLinks();

  /**
   * Contains a list of chat invite links.
   *
   * \param[in] total_count_ Approximate total number of chat invite links found.
   * \param[in] invite_links_ List of invite links.
   */
  chatInviteLinks(int32 total_count_, array<object_ptr<chatInviteLink>> &&invite_links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 112891427;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a user that sent a join request and waits for administrator approval.
 */
class chatJoinRequest final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;
  /// Point in time (Unix timestamp) when the user sent the join request.
  int32 date_;
  /// A short bio of the user.
  string bio_;

  /**
   * Describes a user that sent a join request and waits for administrator approval.
   */
  chatJoinRequest();

  /**
   * Describes a user that sent a join request and waits for administrator approval.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] date_ Point in time (Unix timestamp) when the user sent the join request.
   * \param[in] bio_ A short bio of the user.
   */
  chatJoinRequest(int53 user_id_, int32 date_, string const &bio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 59341416;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatJoinRequest;

/**
 * Contains a list of requests to join a chat.
 */
class chatJoinRequests final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of requests found.
  int32 total_count_;
  /// List of the requests.
  array<object_ptr<chatJoinRequest>> requests_;

  /**
   * Contains a list of requests to join a chat.
   */
  chatJoinRequests();

  /**
   * Contains a list of requests to join a chat.
   *
   * \param[in] total_count_ Approximate total number of requests found.
   * \param[in] requests_ List of the requests.
   */
  chatJoinRequests(int32 total_count_, array<object_ptr<chatJoinRequest>> &&requests_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1291680519;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about pending join requests for a chat.
 */
class chatJoinRequestsInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Total number of pending join requests.
  int32 total_count_;
  /// Identifiers of at most 3 users sent the newest pending join requests.
  array<int53> user_ids_;

  /**
   * Contains information about pending join requests for a chat.
   */
  chatJoinRequestsInfo();

  /**
   * Contains information about pending join requests for a chat.
   *
   * \param[in] total_count_ Total number of pending join requests.
   * \param[in] user_ids_ Identifiers of at most 3 users sent the newest pending join requests.
   */
  chatJoinRequestsInfo(int32 total_count_, array<int53> &&user_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 888534463;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a list of chats.
 */
class ChatList: public Object {
 public:
};

/**
 * A main list of chats.
 */
class chatListMain final : public ChatList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A main list of chats.
   */
  chatListMain();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -400991316;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A list of chats usually located at the top of the main chat list. Unmuted chats are automatically moved from the Archive to the Main chat list when a new message arrives.
 */
class chatListArchive final : public ChatList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A list of chats usually located at the top of the main chat list. Unmuted chats are automatically moved from the Archive to the Main chat list when a new message arrives.
   */
  chatListArchive();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 362770115;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A list of chats added to a chat folder.
 */
class chatListFolder final : public ChatList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat folder identifier.
  int32 chat_folder_id_;

  /**
   * A list of chats added to a chat folder.
   */
  chatListFolder();

  /**
   * A list of chats added to a chat folder.
   *
   * \param[in] chat_folder_id_ Chat folder identifier.
   */
  explicit chatListFolder(int32 chat_folder_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 385760856;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatList;

/**
 * Contains a list of chat lists.
 */
class chatLists final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of chat lists.
  array<object_ptr<ChatList>> chat_lists_;

  /**
   * Contains a list of chat lists.
   */
  chatLists();

  /**
   * Contains a list of chat lists.
   *
   * \param[in] chat_lists_ List of chat lists.
   */
  explicit chatLists(array<object_ptr<ChatList>> &&chat_lists_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -258292771;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class location;

/**
 * Represents a location to which a chat is connected.
 */
class chatLocation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The location.
  object_ptr<location> location_;
  /// Location address; 1-64 characters, as defined by the chat owner.
  string address_;

  /**
   * Represents a location to which a chat is connected.
   */
  chatLocation();

  /**
   * Represents a location to which a chat is connected.
   *
   * \param[in] location_ The location.
   * \param[in] address_ Location address; 1-64 characters, as defined by the chat owner.
   */
  chatLocation(object_ptr<location> &&location_, string const &address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1566863583;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatMemberStatus;

class MessageSender;

/**
 * Describes a user or a chat as a member of another chat.
 */
class chatMember final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the chat member. Currently, other chats can be only Left or Banned. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels.
  object_ptr<MessageSender> member_id_;
  /// Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown.
  int53 inviter_user_id_;
  /// Point in time (Unix timestamp) when the user joined/was promoted/was banned in the chat.
  int32 joined_chat_date_;
  /// Status of the member in the chat.
  object_ptr<ChatMemberStatus> status_;

  /**
   * Describes a user or a chat as a member of another chat.
   */
  chatMember();

  /**
   * Describes a user or a chat as a member of another chat.
   *
   * \param[in] member_id_ Identifier of the chat member. Currently, other chats can be only Left or Banned. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels.
   * \param[in] inviter_user_id_ Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown.
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined/was promoted/was banned in the chat.
   * \param[in] status_ Status of the member in the chat.
   */
  chatMember(object_ptr<MessageSender> &&member_id_, int53 inviter_user_id_, int32 joined_chat_date_, object_ptr<ChatMemberStatus> &&status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1829953909;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatAdministratorRights;

class chatPermissions;

/**
 * This class is an abstract base class.
 * Provides information about the status of a member in a chat.
 */
class ChatMemberStatus: public Object {
 public:
};

/**
 * The user is the owner of the chat and has all the administrator privileges.
 */
class chatMemberStatusCreator final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A custom title of the owner; 0-16 characters without emoji; applicable to supergroups only.
  string custom_title_;
  /// True, if the creator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
  bool is_anonymous_;
  /// True, if the user is a member of the chat.
  bool is_member_;

  /**
   * The user is the owner of the chat and has all the administrator privileges.
   */
  chatMemberStatusCreator();

  /**
   * The user is the owner of the chat and has all the administrator privileges.
   *
   * \param[in] custom_title_ A custom title of the owner; 0-16 characters without emoji; applicable to supergroups only.
   * \param[in] is_anonymous_ True, if the creator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
   * \param[in] is_member_ True, if the user is a member of the chat.
   */
  chatMemberStatusCreator(string const &custom_title_, bool is_anonymous_, bool is_member_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -160019714;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
 */
class chatMemberStatusAdministrator final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A custom title of the administrator; 0-16 characters without emoji; applicable to supergroups only.
  string custom_title_;
  /// True, if the current user can edit the administrator privileges for the called user.
  bool can_be_edited_;
  /// Rights of the administrator.
  object_ptr<chatAdministratorRights> rights_;

  /**
   * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
   */
  chatMemberStatusAdministrator();

  /**
   * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
   *
   * \param[in] custom_title_ A custom title of the administrator; 0-16 characters without emoji; applicable to supergroups only.
   * \param[in] can_be_edited_ True, if the current user can edit the administrator privileges for the called user.
   * \param[in] rights_ Rights of the administrator.
   */
  chatMemberStatusAdministrator(string const &custom_title_, bool can_be_edited_, object_ptr<chatAdministratorRights> &&rights_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -70024163;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is a member of the chat, without any additional privileges or restrictions.
 */
class chatMemberStatusMember final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the user will be removed from the chat because of the expired subscription; 0 if never. Ignored in setChatMemberStatus.
  int32 member_until_date_;

  /**
   * The user is a member of the chat, without any additional privileges or restrictions.
   */
  chatMemberStatusMember();

  /**
   * The user is a member of the chat, without any additional privileges or restrictions.
   *
   * \param[in] member_until_date_ Point in time (Unix timestamp) when the user will be removed from the chat because of the expired subscription; 0 if never. Ignored in setChatMemberStatus.
   */
  explicit chatMemberStatusMember(int32 member_until_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -32707562;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
 */
class chatMemberStatusRestricted final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if the user is a member of the chat.
  bool is_member_;
  /// Point in time (Unix timestamp) when restrictions will be lifted from the user; 0 if never. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever.
  int32 restricted_until_date_;
  /// User permissions in the chat.
  object_ptr<chatPermissions> permissions_;

  /**
   * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
   */
  chatMemberStatusRestricted();

  /**
   * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
   *
   * \param[in] is_member_ True, if the user is a member of the chat.
   * \param[in] restricted_until_date_ Point in time (Unix timestamp) when restrictions will be lifted from the user; 0 if never. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever.
   * \param[in] permissions_ User permissions in the chat.
   */
  chatMemberStatusRestricted(bool is_member_, int32 restricted_until_date_, object_ptr<chatPermissions> &&permissions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1661432998;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user or the chat is not a chat member.
 */
class chatMemberStatusLeft final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user or the chat is not a chat member.
   */
  chatMemberStatusLeft();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -5815259;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user or the chat was banned (and hence is not a member of the chat). Implies the user can't return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
 */
class chatMemberStatusBanned final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the user will be unbanned; 0 if never. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever. Always 0 in basic groups.
  int32 banned_until_date_;

  /**
   * The user or the chat was banned (and hence is not a member of the chat). Implies the user can't return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
   */
  chatMemberStatusBanned();

  /**
   * The user or the chat was banned (and hence is not a member of the chat). Implies the user can't return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
   *
   * \param[in] banned_until_date_ Point in time (Unix timestamp) when the user will be unbanned; 0 if never. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever. Always 0 in basic groups.
   */
  explicit chatMemberStatusBanned(int32 banned_until_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1653518666;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatMember;

/**
 * Contains a list of chat members.
 */
class chatMembers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of chat members found.
  int32 total_count_;
  /// A list of chat members.
  array<object_ptr<chatMember>> members_;

  /**
   * Contains a list of chat members.
   */
  chatMembers();

  /**
   * Contains a list of chat members.
   *
   * \param[in] total_count_ Approximate total number of chat members found.
   * \param[in] members_ A list of chat members.
   */
  chatMembers(int32 total_count_, array<object_ptr<chatMember>> &&members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -497558622;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Specifies the kind of chat members to return in searchChatMembers.
 */
class ChatMembersFilter: public Object {
 public:
};

/**
 * Returns contacts of the user.
 */
class chatMembersFilterContacts final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Returns contacts of the user.
   */
  chatMembersFilterContacts();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1774485671;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns the owner and administrators.
 */
class chatMembersFilterAdministrators final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Returns the owner and administrators.
   */
  chatMembersFilterAdministrators();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1266893796;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns all chat members, including restricted chat members.
 */
class chatMembersFilterMembers final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Returns all chat members, including restricted chat members.
   */
  chatMembersFilterMembers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 670504342;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns users which can be mentioned in the chat.
 */
class chatMembersFilterMention final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// If non-zero, the identifier of the current message thread.
  int53 message_thread_id_;

  /**
   * Returns users which can be mentioned in the chat.
   */
  chatMembersFilterMention();

  /**
   * Returns users which can be mentioned in the chat.
   *
   * \param[in] message_thread_id_ If non-zero, the identifier of the current message thread.
   */
  explicit chatMembersFilterMention(int53 message_thread_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 856419831;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns users under certain restrictions in the chat; can be used only by administrators in a supergroup.
 */
class chatMembersFilterRestricted final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Returns users under certain restrictions in the chat; can be used only by administrators in a supergroup.
   */
  chatMembersFilterRestricted();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1256282813;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns users banned from the chat; can be used only by administrators in a supergroup or in a channel.
 */
class chatMembersFilterBanned final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Returns users banned from the chat; can be used only by administrators in a supergroup or in a channel.
   */
  chatMembersFilterBanned();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1863102648;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Returns bot members of the chat.
 */
class chatMembersFilterBots final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Returns bot members of the chat.
   */
  chatMembersFilterBots();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1422567288;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class MessageSender;

/**
 * Represents a message sender, which can be used to send messages in a chat.
 */
class chatMessageSender final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The message sender.
  object_ptr<MessageSender> sender_;
  /// True, if Telegram Premium is needed to use the message sender.
  bool needs_premium_;

  /**
   * Represents a message sender, which can be used to send messages in a chat.
   */
  chatMessageSender();

  /**
   * Represents a message sender, which can be used to send messages in a chat.
   *
   * \param[in] sender_ The message sender.
   * \param[in] needs_premium_ True, if Telegram Premium is needed to use the message sender.
   */
  chatMessageSender(object_ptr<MessageSender> &&sender_, bool needs_premium_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 760590010;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatMessageSender;

/**
 * Represents a list of message senders, which can be used to send messages in a chat.
 */
class chatMessageSenders final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of available message senders.
  array<object_ptr<chatMessageSender>> senders_;

  /**
   * Represents a list of message senders, which can be used to send messages in a chat.
   */
  chatMessageSenders();

  /**
   * Represents a list of message senders, which can be used to send messages in a chat.
   *
   * \param[in] senders_ List of available message senders.
   */
  explicit chatMessageSenders(array<object_ptr<chatMessageSender>> &&senders_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1866230970;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about notification settings for a chat or a forum topic.
 */
class chatNotificationSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// If true, the value for the relevant type of chat or the forum chat is used instead of mute_for.
  bool use_default_mute_for_;
  /// Time left before notifications will be unmuted, in seconds.
  int32 mute_for_;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of sound_id.
  bool use_default_sound_;
  /// Identifier of the notification sound to be played for messages; 0 if sound is disabled.
  int64 sound_id_;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of show_preview.
  bool use_default_show_preview_;
  /// True, if message content must be displayed in notifications.
  bool show_preview_;
  /// If true, the value for the relevant type of chat is used instead of mute_stories.
  bool use_default_mute_stories_;
  /// True, if story notifications are disabled for the chat.
  bool mute_stories_;
  /// If true, the value for the relevant type of chat is used instead of story_sound_id.
  bool use_default_story_sound_;
  /// Identifier of the notification sound to be played for stories; 0 if sound is disabled.
  int64 story_sound_id_;
  /// If true, the value for the relevant type of chat is used instead of show_story_sender.
  bool use_default_show_story_sender_;
  /// True, if the sender of stories must be displayed in notifications.
  bool show_story_sender_;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of disable_pinned_message_notifications.
  bool use_default_disable_pinned_message_notifications_;
  /// If true, notifications for incoming pinned messages will be created as for an ordinary unread message.
  bool disable_pinned_message_notifications_;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of disable_mention_notifications.
  bool use_default_disable_mention_notifications_;
  /// If true, notifications for messages with mentions will be created as for an ordinary unread message.
  bool disable_mention_notifications_;

  /**
   * Contains information about notification settings for a chat or a forum topic.
   */
  chatNotificationSettings();

  /**
   * Contains information about notification settings for a chat or a forum topic.
   *
   * \param[in] use_default_mute_for_ If true, the value for the relevant type of chat or the forum chat is used instead of mute_for.
   * \param[in] mute_for_ Time left before notifications will be unmuted, in seconds.
   * \param[in] use_default_sound_ If true, the value for the relevant type of chat or the forum chat is used instead of sound_id.
   * \param[in] sound_id_ Identifier of the notification sound to be played for messages; 0 if sound is disabled.
   * \param[in] use_default_show_preview_ If true, the value for the relevant type of chat or the forum chat is used instead of show_preview.
   * \param[in] show_preview_ True, if message content must be displayed in notifications.
   * \param[in] use_default_mute_stories_ If true, the value for the relevant type of chat is used instead of mute_stories.
   * \param[in] mute_stories_ True, if story notifications are disabled for the chat.
   * \param[in] use_default_story_sound_ If true, the value for the relevant type of chat is used instead of story_sound_id.
   * \param[in] story_sound_id_ Identifier of the notification sound to be played for stories; 0 if sound is disabled.
   * \param[in] use_default_show_story_sender_ If true, the value for the relevant type of chat is used instead of show_story_sender.
   * \param[in] show_story_sender_ True, if the sender of stories must be displayed in notifications.
   * \param[in] use_default_disable_pinned_message_notifications_ If true, the value for the relevant type of chat or the forum chat is used instead of disable_pinned_message_notifications.
   * \param[in] disable_pinned_message_notifications_ If true, notifications for incoming pinned messages will be created as for an ordinary unread message.
   * \param[in] use_default_disable_mention_notifications_ If true, the value for the relevant type of chat or the forum chat is used instead of disable_mention_notifications.
   * \param[in] disable_mention_notifications_ If true, notifications for messages with mentions will be created as for an ordinary unread message.
   */
  chatNotificationSettings(bool use_default_mute_for_, int32 mute_for_, bool use_default_sound_, int64 sound_id_, bool use_default_show_preview_, bool show_preview_, bool use_default_mute_stories_, bool mute_stories_, bool use_default_story_sound_, int64 story_sound_id_, bool use_default_show_story_sender_, bool show_story_sender_, bool use_default_disable_pinned_message_notifications_, bool disable_pinned_message_notifications_, bool use_default_disable_mention_notifications_, bool disable_mention_notifications_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -572779825;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes actions that a user is allowed to take in a chat.
 */
class chatPermissions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// True, if the user can send text messages, contacts, giveaways, giveaway winners, invoices, locations, and venues.
  bool can_send_basic_messages_;
  /// True, if the user can send music files.
  bool can_send_audios_;
  /// True, if the user can send documents.
  bool can_send_documents_;
  /// True, if the user can send photos.
  bool can_send_photos_;
  /// True, if the user can send videos.
  bool can_send_videos_;
  /// True, if the user can send video notes.
  bool can_send_video_notes_;
  /// True, if the user can send voice notes.
  bool can_send_voice_notes_;
  /// True, if the user can send polls.
  bool can_send_polls_;
  /// True, if the user can send animations, games, stickers, and dice and use inline bots.
  bool can_send_other_messages_;
  /// True, if the user may add a link preview to their messages.
  bool can_add_link_previews_;
  /// True, if the user can change the chat title, photo, and other settings.
  bool can_change_info_;
  /// True, if the user can invite new users to the chat.
  bool can_invite_users_;
  /// True, if the user can pin messages.
  bool can_pin_messages_;
  /// True, if the user can create topics.
  bool can_create_topics_;

  /**
   * Describes actions that a user is allowed to take in a chat.
   */
  chatPermissions();

  /**
   * Describes actions that a user is allowed to take in a chat.
   *
   * \param[in] can_send_basic_messages_ True, if the user can send text messages, contacts, giveaways, giveaway winners, invoices, locations, and venues.
   * \param[in] can_send_audios_ True, if the user can send music files.
   * \param[in] can_send_documents_ True, if the user can send documents.
   * \param[in] can_send_photos_ True, if the user can send photos.
   * \param[in] can_send_videos_ True, if the user can send videos.
   * \param[in] can_send_video_notes_ True, if the user can send video notes.
   * \param[in] can_send_voice_notes_ True, if the user can send voice notes.
   * \param[in] can_send_polls_ True, if the user can send polls.
   * \param[in] can_send_other_messages_ True, if the user can send animations, games, stickers, and dice and use inline bots.
   * \param[in] can_add_link_previews_ True, if the user may add a link preview to their messages.
   * \param[in] can_change_info_ True, if the user can change the chat title, photo, and other settings.
   * \param[in] can_invite_users_ True, if the user can invite new users to the chat.
   * \param[in] can_pin_messages_ True, if the user can pin messages.
   * \param[in] can_create_topics_ True, if the user can create topics.
   */
  chatPermissions(bool can_send_basic_messages_, bool can_send_audios_, bool can_send_documents_, bool can_send_photos_, bool can_send_videos_, bool can_send_video_notes_, bool can_send_voice_notes_, bool can_send_polls_, bool can_send_other_messages_, bool can_add_link_previews_, bool can_change_info_, bool can_invite_users_, bool can_pin_messages_, bool can_create_topics_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -118334855;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class animatedChatPhoto;

class chatPhotoSticker;

class minithumbnail;

class photoSize;

/**
 * Describes a chat or user profile photo.
 */
class chatPhoto final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique photo identifier.
  int64 id_;
  /// Point in time (Unix timestamp) when the photo has been added.
  int32 added_date_;
  /// Photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// Available variants of the photo in JPEG format, in different size.
  array<object_ptr<photoSize>> sizes_;
  /// A big (up to 1280x1280) animated variant of the photo in MPEG4 format; may be null.
  object_ptr<animatedChatPhoto> animation_;
  /// A small (160x160) animated variant of the photo in MPEG4 format; may be null even the big animation is available.
  object_ptr<animatedChatPhoto> small_animation_;
  /// Sticker-based version of the chat photo; may be null.
  object_ptr<chatPhotoSticker> sticker_;

  /**
   * Describes a chat or user profile photo.
   */
  chatPhoto();

  /**
   * Describes a chat or user profile photo.
   *
   * \param[in] id_ Unique photo identifier.
   * \param[in] added_date_ Point in time (Unix timestamp) when the photo has been added.
   * \param[in] minithumbnail_ Photo minithumbnail; may be null.
   * \param[in] sizes_ Available variants of the photo in JPEG format, in different size.
   * \param[in] animation_ A big (up to 1280x1280) animated variant of the photo in MPEG4 format; may be null.
   * \param[in] small_animation_ A small (160x160) animated variant of the photo in MPEG4 format; may be null even the big animation is available.
   * \param[in] sticker_ Sticker-based version of the chat photo; may be null.
   */
  chatPhoto(int64 id_, int32 added_date_, object_ptr<minithumbnail> &&minithumbnail_, array<object_ptr<photoSize>> &&sizes_, object_ptr<animatedChatPhoto> &&animation_, object_ptr<animatedChatPhoto> &&small_animation_, object_ptr<chatPhotoSticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1430870201;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

class minithumbnail;

/**
 * Contains basic information about the photo of a chat.
 */
class chatPhotoInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A small (160x160) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
  object_ptr<file> small_;
  /// A big (640x640) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
  object_ptr<file> big_;
  /// Chat photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// True, if the photo has animated variant.
  bool has_animation_;
  /// True, if the photo is visible only for the current user.
  bool is_personal_;

  /**
   * Contains basic information about the photo of a chat.
   */
  chatPhotoInfo();

  /**
   * Contains basic information about the photo of a chat.
   *
   * \param[in] small_ A small (160x160) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
   * \param[in] big_ A big (640x640) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
   * \param[in] minithumbnail_ Chat photo minithumbnail; may be null.
   * \param[in] has_animation_ True, if the photo has animated variant.
   * \param[in] is_personal_ True, if the photo is visible only for the current user.
   */
  chatPhotoInfo(object_ptr<file> &&small_, object_ptr<file> &&big_, object_ptr<minithumbnail> &&minithumbnail_, bool has_animation_, bool is_personal_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 281195686;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class BackgroundFill;

class ChatPhotoStickerType;

/**
 * Information about the sticker, which was used to create the chat photo. The sticker is shown at the center of the photo and occupies at most 67% of it.
 */
class chatPhotoSticker final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of the sticker.
  object_ptr<ChatPhotoStickerType> type_;
  /// The fill to be used as background for the sticker; rotation angle in backgroundFillGradient isn't supported.
  object_ptr<BackgroundFill> background_fill_;

  /**
   * Information about the sticker, which was used to create the chat photo. The sticker is shown at the center of the photo and occupies at most 67% of it.
   */
  chatPhotoSticker();

  /**
   * Information about the sticker, which was used to create the chat photo. The sticker is shown at the center of the photo and occupies at most 67% of it.
   *
   * \param[in] type_ Type of the sticker.
   * \param[in] background_fill_ The fill to be used as background for the sticker; rotation angle in backgroundFillGradient isn't supported.
   */
  chatPhotoSticker(object_ptr<ChatPhotoStickerType> &&type_, object_ptr<BackgroundFill> &&background_fill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1459387485;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes type of sticker, which was used to create a chat photo.
 */
class ChatPhotoStickerType: public Object {
 public:
};

/**
 * Information about the sticker, which was used to create the chat photo.
 */
class chatPhotoStickerTypeRegularOrMask final : public ChatPhotoStickerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Sticker set identifier.
  int64 sticker_set_id_;
  /// Identifier of the sticker in the set.
  int64 sticker_id_;

  /**
   * Information about the sticker, which was used to create the chat photo.
   */
  chatPhotoStickerTypeRegularOrMask();

  /**
   * Information about the sticker, which was used to create the chat photo.
   *
   * \param[in] sticker_set_id_ Sticker set identifier.
   * \param[in] sticker_id_ Identifier of the sticker in the set.
   */
  chatPhotoStickerTypeRegularOrMask(int64 sticker_set_id_, int64 sticker_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -415147620;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Information about the custom emoji, which was used to create the chat photo.
 */
class chatPhotoStickerTypeCustomEmoji final : public ChatPhotoStickerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the custom emoji.
  int64 custom_emoji_id_;

  /**
   * Information about the custom emoji, which was used to create the chat photo.
   */
  chatPhotoStickerTypeCustomEmoji();

  /**
   * Information about the custom emoji, which was used to create the chat photo.
   *
   * \param[in] custom_emoji_id_ Identifier of the custom emoji.
   */
  explicit chatPhotoStickerTypeCustomEmoji(int64 custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -266224943;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatPhoto;

/**
 * Contains a list of chat or user profile photos.
 */
class chatPhotos final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Total number of photos.
  int32 total_count_;
  /// List of photos.
  array<object_ptr<chatPhoto>> photos_;

  /**
   * Contains a list of chat or user profile photos.
   */
  chatPhotos();

  /**
   * Contains a list of chat or user profile photos.
   *
   * \param[in] total_count_ Total number of photos.
   * \param[in] photos_ List of photos.
   */
  chatPhotos(int32 total_count_, array<object_ptr<chatPhoto>> &&photos_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1510699180;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatList;

class ChatSource;

/**
 * Describes a position of a chat in a chat list.
 */
class chatPosition final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The chat list.
  object_ptr<ChatList> list_;
  /// A parameter used to determine order of the chat in the chat list. Chats must be sorted by the pair (order, chat.id) in descending order.
  int64 order_;
  /// True, if the chat is pinned in the chat list.
  bool is_pinned_;
  /// Source of the chat in the chat list; may be null.
  object_ptr<ChatSource> source_;

  /**
   * Describes a position of a chat in a chat list.
   */
  chatPosition();

  /**
   * Describes a position of a chat in a chat list.
   *
   * \param[in] list_ The chat list.
   * \param[in] order_ A parameter used to determine order of the chat in the chat list. Chats must be sorted by the pair (order, chat.id) in descending order.
   * \param[in] is_pinned_ True, if the chat is pinned in the chat list.
   * \param[in] source_ Source of the chat in the chat list; may be null.
   */
  chatPosition(object_ptr<ChatList> &&list_, int64 order_, bool is_pinned_, object_ptr<ChatSource> &&source_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -622557355;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about revenue earned from sponsored messages in a chat.
 */
class chatRevenueAmount final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Cryptocurrency in which revenue is calculated.
  string cryptocurrency_;
  /// Total amount of the cryptocurrency earned, in the smallest units of the cryptocurrency.
  int64 total_amount_;
  /// Amount of the cryptocurrency that isn't withdrawn yet, in the smallest units of the cryptocurrency.
  int64 balance_amount_;
  /// Amount of the cryptocurrency available for withdrawal, in the smallest units of the cryptocurrency.
  int64 available_amount_;
  /// True, if Telegram Stars can be withdrawn now or later.
  bool withdrawal_enabled_;

  /**
   * Contains information about revenue earned from sponsored messages in a chat.
   */
  chatRevenueAmount();

  /**
   * Contains information about revenue earned from sponsored messages in a chat.
   *
   * \param[in] cryptocurrency_ Cryptocurrency in which revenue is calculated.
   * \param[in] total_amount_ Total amount of the cryptocurrency earned, in the smallest units of the cryptocurrency.
   * \param[in] balance_amount_ Amount of the cryptocurrency that isn't withdrawn yet, in the smallest units of the cryptocurrency.
   * \param[in] available_amount_ Amount of the cryptocurrency available for withdrawal, in the smallest units of the cryptocurrency.
   * \param[in] withdrawal_enabled_ True, if Telegram Stars can be withdrawn now or later.
   */
  chatRevenueAmount(string const &cryptocurrency_, int64 total_amount_, int64 balance_amount_, int64 available_amount_, bool withdrawal_enabled_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1505178024;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class StatisticalGraph;

class chatRevenueAmount;

/**
 * A detailed statistics about revenue earned from sponsored messages in a chat.
 */
class chatRevenueStatistics final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A graph containing amount of revenue in a given hour.
  object_ptr<StatisticalGraph> revenue_by_hour_graph_;
  /// A graph containing amount of revenue.
  object_ptr<StatisticalGraph> revenue_graph_;
  /// Amount of earned revenue.
  object_ptr<chatRevenueAmount> revenue_amount_;
  /// Current conversion rate of the cryptocurrency in which revenue is calculated to USD.
  double usd_rate_;

  /**
   * A detailed statistics about revenue earned from sponsored messages in a chat.
   */
  chatRevenueStatistics();

  /**
   * A detailed statistics about revenue earned from sponsored messages in a chat.
   *
   * \param[in] revenue_by_hour_graph_ A graph containing amount of revenue in a given hour.
   * \param[in] revenue_graph_ A graph containing amount of revenue.
   * \param[in] revenue_amount_ Amount of earned revenue.
   * \param[in] usd_rate_ Current conversion rate of the cryptocurrency in which revenue is calculated to USD.
   */
  chatRevenueStatistics(object_ptr<StatisticalGraph> &&revenue_by_hour_graph_, object_ptr<StatisticalGraph> &&revenue_graph_, object_ptr<chatRevenueAmount> &&revenue_amount_, double usd_rate_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1667438779;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatRevenueTransactionType;

/**
 * Contains a chat revenue transactions.
 */
class chatRevenueTransaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Cryptocurrency in which revenue is calculated.
  string cryptocurrency_;
  /// The withdrawn amount, in the smallest units of the cryptocurrency.
  int64 cryptocurrency_amount_;
  /// Type of the transaction.
  object_ptr<ChatRevenueTransactionType> type_;

  /**
   * Contains a chat revenue transactions.
   */
  chatRevenueTransaction();

  /**
   * Contains a chat revenue transactions.
   *
   * \param[in] cryptocurrency_ Cryptocurrency in which revenue is calculated.
   * \param[in] cryptocurrency_amount_ The withdrawn amount, in the smallest units of the cryptocurrency.
   * \param[in] type_ Type of the transaction.
   */
  chatRevenueTransaction(string const &cryptocurrency_, int64 cryptocurrency_amount_, object_ptr<ChatRevenueTransactionType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 80192767;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class RevenueWithdrawalState;

/**
 * This class is an abstract base class.
 * Describes type of transaction for revenue earned from sponsored messages in a chat.
 */
class ChatRevenueTransactionType: public Object {
 public:
};

/**
 * Describes earnings from sponsored messages in a chat in some time frame.
 */
class chatRevenueTransactionTypeEarnings final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the earnings started.
  int32 start_date_;
  /// Point in time (Unix timestamp) when the earnings ended.
  int32 end_date_;

  /**
   * Describes earnings from sponsored messages in a chat in some time frame.
   */
  chatRevenueTransactionTypeEarnings();

  /**
   * Describes earnings from sponsored messages in a chat in some time frame.
   *
   * \param[in] start_date_ Point in time (Unix timestamp) when the earnings started.
   * \param[in] end_date_ Point in time (Unix timestamp) when the earnings ended.
   */
  chatRevenueTransactionTypeEarnings(int32 start_date_, int32 end_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -400776056;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a withdrawal of earnings.
 */
class chatRevenueTransactionTypeWithdrawal final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the earnings withdrawal started.
  int32 withdrawal_date_;
  /// Name of the payment provider.
  string provider_;
  /// State of the withdrawal.
  object_ptr<RevenueWithdrawalState> state_;

  /**
   * Describes a withdrawal of earnings.
   */
  chatRevenueTransactionTypeWithdrawal();

  /**
   * Describes a withdrawal of earnings.
   *
   * \param[in] withdrawal_date_ Point in time (Unix timestamp) when the earnings withdrawal started.
   * \param[in] provider_ Name of the payment provider.
   * \param[in] state_ State of the withdrawal.
   */
  chatRevenueTransactionTypeWithdrawal(int32 withdrawal_date_, string const &provider_, object_ptr<RevenueWithdrawalState> &&state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 252939755;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a refund for failed withdrawal of earnings.
 */
class chatRevenueTransactionTypeRefund final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the transaction was refunded.
  int32 refund_date_;
  /// Name of the payment provider.
  string provider_;

  /**
   * Describes a refund for failed withdrawal of earnings.
   */
  chatRevenueTransactionTypeRefund();

  /**
   * Describes a refund for failed withdrawal of earnings.
   *
   * \param[in] refund_date_ Point in time (Unix timestamp) when the transaction was refunded.
   * \param[in] provider_ Name of the payment provider.
   */
  chatRevenueTransactionTypeRefund(int32 refund_date_, string const &provider_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 302430279;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatRevenueTransaction;

/**
 * Contains a list of chat revenue transactions.
 */
class chatRevenueTransactions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Total number of transactions.
  int32 total_count_;
  /// List of transactions.
  array<object_ptr<chatRevenueTransaction>> transactions_;

  /**
   * Contains a list of chat revenue transactions.
   */
  chatRevenueTransactions();

  /**
   * Contains a list of chat revenue transactions.
   *
   * \param[in] total_count_ Total number of transactions.
   * \param[in] transactions_ List of transactions.
   */
  chatRevenueTransactions(int32 total_count_, array<object_ptr<chatRevenueTransaction>> &&transactions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -553258171;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a reason why an external chat is shown in a chat list.
 */
class ChatSource: public Object {
 public:
};

/**
 * The chat is sponsored by the user's MTProxy server.
 */
class chatSourceMtprotoProxy final : public ChatSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The chat is sponsored by the user's MTProxy server.
   */
  chatSourceMtprotoProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 394074115;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The chat contains a public service announcement.
 */
class chatSourcePublicServiceAnnouncement final : public ChatSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The type of the announcement.
  string type_;
  /// The text of the announcement.
  string text_;

  /**
   * The chat contains a public service announcement.
   */
  chatSourcePublicServiceAnnouncement();

  /**
   * The chat contains a public service announcement.
   *
   * \param[in] type_ The type of the announcement.
   * \param[in] text_ The text of the announcement.
   */
  chatSourcePublicServiceAnnouncement(string const &type_, string const &text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -328571244;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class StatisticalGraph;

class chatStatisticsAdministratorActionsInfo;

class chatStatisticsInteractionInfo;

class chatStatisticsInviterInfo;

class chatStatisticsMessageSenderInfo;

class dateRange;

class statisticalValue;

/**
 * This class is an abstract base class.
 * Contains a detailed statistics about a chat.
 */
class ChatStatistics: public Object {
 public:
};

/**
 * A detailed statistics about a supergroup chat.
 */
class chatStatisticsSupergroup final : public ChatStatistics {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A period to which the statistics applies.
  object_ptr<dateRange> period_;
  /// Number of members in the chat.
  object_ptr<statisticalValue> member_count_;
  /// Number of messages sent to the chat.
  object_ptr<statisticalValue> message_count_;
  /// Number of users who viewed messages in the chat.
  object_ptr<statisticalValue> viewer_count_;
  /// Number of users who sent messages to the chat.
  object_ptr<statisticalValue> sender_count_;
  /// A graph containing number of members in the chat.
  object_ptr<StatisticalGraph> member_count_graph_;
  /// A graph containing number of members joined and left the chat.
  object_ptr<StatisticalGraph> join_graph_;
  /// A graph containing number of new member joins per source.
  object_ptr<StatisticalGraph> join_by_source_graph_;
  /// A graph containing distribution of active users per language.
  object_ptr<StatisticalGraph> language_graph_;
  /// A graph containing distribution of sent messages by content type.
  object_ptr<StatisticalGraph> message_content_graph_;
  /// A graph containing number of different actions in the chat.
  object_ptr<StatisticalGraph> action_graph_;
  /// A graph containing distribution of message views per hour.
  object_ptr<StatisticalGraph> day_graph_;
  /// A graph containing distribution of message views per day of week.
  object_ptr<StatisticalGraph> week_graph_;
  /// List of users sent most messages in the last week.
  array<object_ptr<chatStatisticsMessageSenderInfo>> top_senders_;
  /// List of most active administrators in the last week.
  array<object_ptr<chatStatisticsAdministratorActionsInfo>> top_administrators_;
  /// List of most active inviters of new members in the last week.
  array<object_ptr<chatStatisticsInviterInfo>> top_inviters_;

  /**
   * A detailed statistics about a supergroup chat.
   */
  chatStatisticsSupergroup();

  /**
   * A detailed statistics about a supergroup chat.
   *
   * \param[in] period_ A period to which the statistics applies.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] message_count_ Number of messages sent to the chat.
   * \param[in] viewer_count_ Number of users who viewed messages in the chat.
   * \param[in] sender_count_ Number of users who sent messages to the chat.
   * \param[in] member_count_graph_ A graph containing number of members in the chat.
   * \param[in] join_graph_ A graph containing number of members joined and left the chat.
   * \param[in] join_by_source_graph_ A graph containing number of new member joins per source.
   * \param[in] language_graph_ A graph containing distribution of active users per language.
   * \param[in] message_content_graph_ A graph containing distribution of sent messages by content type.
   * \param[in] action_graph_ A graph containing number of different actions in the chat.
   * \param[in] day_graph_ A graph containing distribution of message views per hour.
   * \param[in] week_graph_ A graph containing distribution of message views per day of week.
   * \param[in] top_senders_ List of users sent most messages in the last week.
   * \param[in] top_administrators_ List of most active administrators in the last week.
   * \param[in] top_inviters_ List of most active inviters of new members in the last week.
   */
  chatStatisticsSupergroup(object_ptr<dateRange> &&period_, object_ptr<statisticalValue> &&member_count_, object_ptr<statisticalValue> &&message_count_, object_ptr<statisticalValue> &&viewer_count_, object_ptr<statisticalValue> &&sender_count_, object_ptr<StatisticalGraph> &&member_count_graph_, object_ptr<StatisticalGraph> &&join_graph_, object_ptr<StatisticalGraph> &&join_by_source_graph_, object_ptr<StatisticalGraph> &&language_graph_, object_ptr<StatisticalGraph> &&message_content_graph_, object_ptr<StatisticalGraph> &&action_graph_, object_ptr<StatisticalGraph> &&day_graph_, object_ptr<StatisticalGraph> &&week_graph_, array<object_ptr<chatStatisticsMessageSenderInfo>> &&top_senders_, array<object_ptr<chatStatisticsAdministratorActionsInfo>> &&top_administrators_, array<object_ptr<chatStatisticsInviterInfo>> &&top_inviters_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -17244633;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A detailed statistics about a channel chat.
 */
class chatStatisticsChannel final : public ChatStatistics {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A period to which the statistics applies.
  object_ptr<dateRange> period_;
  /// Number of members in the chat.
  object_ptr<statisticalValue> member_count_;
  /// Mean number of times the recently sent messages were viewed.
  object_ptr<statisticalValue> mean_message_view_count_;
  /// Mean number of times the recently sent messages were shared.
  object_ptr<statisticalValue> mean_message_share_count_;
  /// Mean number of times reactions were added to the recently sent messages.
  object_ptr<statisticalValue> mean_message_reaction_count_;
  /// Mean number of times the recently sent stories were viewed.
  object_ptr<statisticalValue> mean_story_view_count_;
  /// Mean number of times the recently sent stories were shared.
  object_ptr<statisticalValue> mean_story_share_count_;
  /// Mean number of times reactions were added to the recently sent stories.
  object_ptr<statisticalValue> mean_story_reaction_count_;
  /// A percentage of users with enabled notifications for the chat; 0-100.
  double enabled_notifications_percentage_;
  /// A graph containing number of members in the chat.
  object_ptr<StatisticalGraph> member_count_graph_;
  /// A graph containing number of members joined and left the chat.
  object_ptr<StatisticalGraph> join_graph_;
  /// A graph containing number of members muted and unmuted the chat.
  object_ptr<StatisticalGraph> mute_graph_;
  /// A graph containing number of message views in a given hour in the last two weeks.
  object_ptr<StatisticalGraph> view_count_by_hour_graph_;
  /// A graph containing number of message views per source.
  object_ptr<StatisticalGraph> view_count_by_source_graph_;
  /// A graph containing number of new member joins per source.
  object_ptr<StatisticalGraph> join_by_source_graph_;
  /// A graph containing number of users viewed chat messages per language.
  object_ptr<StatisticalGraph> language_graph_;
  /// A graph containing number of chat message views and shares.
  object_ptr<StatisticalGraph> message_interaction_graph_;
  /// A graph containing number of reactions on messages.
  object_ptr<StatisticalGraph> message_reaction_graph_;
  /// A graph containing number of story views and shares.
  object_ptr<StatisticalGraph> story_interaction_graph_;
  /// A graph containing number of reactions on stories.
  object_ptr<StatisticalGraph> story_reaction_graph_;
  /// A graph containing number of views of associated with the chat instant views.
  object_ptr<StatisticalGraph> instant_view_interaction_graph_;
  /// Detailed statistics about number of views and shares of recently sent messages and stories.
  array<object_ptr<chatStatisticsInteractionInfo>> recent_interactions_;

  /**
   * A detailed statistics about a channel chat.
   */
  chatStatisticsChannel();

  /**
   * A detailed statistics about a channel chat.
   *
   * \param[in] period_ A period to which the statistics applies.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] mean_message_view_count_ Mean number of times the recently sent messages were viewed.
   * \param[in] mean_message_share_count_ Mean number of times the recently sent messages were shared.
   * \param[in] mean_message_reaction_count_ Mean number of times reactions were added to the recently sent messages.
   * \param[in] mean_story_view_count_ Mean number of times the recently sent stories were viewed.
   * \param[in] mean_story_share_count_ Mean number of times the recently sent stories were shared.
   * \param[in] mean_story_reaction_count_ Mean number of times reactions were added to the recently sent stories.
   * \param[in] enabled_notifications_percentage_ A percentage of users with enabled notifications for the chat; 0-100.
   * \param[in] member_count_graph_ A graph containing number of members in the chat.
   * \param[in] join_graph_ A graph containing number of members joined and left the chat.
   * \param[in] mute_graph_ A graph containing number of members muted and unmuted the chat.
   * \param[in] view_count_by_hour_graph_ A graph containing number of message views in a given hour in the last two weeks.
   * \param[in] view_count_by_source_graph_ A graph containing number of message views per source.
   * \param[in] join_by_source_graph_ A graph containing number of new member joins per source.
   * \param[in] language_graph_ A graph containing number of users viewed chat messages per language.
   * \param[in] message_interaction_graph_ A graph containing number of chat message views and shares.
   * \param[in] message_reaction_graph_ A graph containing number of reactions on messages.
   * \param[in] story_interaction_graph_ A graph containing number of story views and shares.
   * \param[in] story_reaction_graph_ A graph containing number of reactions on stories.
   * \param[in] instant_view_interaction_graph_ A graph containing number of views of associated with the chat instant views.
   * \param[in] recent_interactions_ Detailed statistics about number of views and shares of recently sent messages and stories.
   */
  chatStatisticsChannel(object_ptr<dateRange> &&period_, object_ptr<statisticalValue> &&member_count_, object_ptr<statisticalValue> &&mean_message_view_count_, object_ptr<statisticalValue> &&mean_message_share_count_, object_ptr<statisticalValue> &&mean_message_reaction_count_, object_ptr<statisticalValue> &&mean_story_view_count_, object_ptr<statisticalValue> &&mean_story_share_count_, object_ptr<statisticalValue> &&mean_story_reaction_count_, double enabled_notifications_percentage_, object_ptr<StatisticalGraph> &&member_count_graph_, object_ptr<StatisticalGraph> &&join_graph_, object_ptr<StatisticalGraph> &&mute_graph_, object_ptr<StatisticalGraph> &&view_count_by_hour_graph_, object_ptr<StatisticalGraph> &&view_count_by_source_graph_, object_ptr<StatisticalGraph> &&join_by_source_graph_, object_ptr<StatisticalGraph> &&language_graph_, object_ptr<StatisticalGraph> &&message_interaction_graph_, object_ptr<StatisticalGraph> &&message_reaction_graph_, object_ptr<StatisticalGraph> &&story_interaction_graph_, object_ptr<StatisticalGraph> &&story_reaction_graph_, object_ptr<StatisticalGraph> &&instant_view_interaction_graph_, array<object_ptr<chatStatisticsInteractionInfo>> &&recent_interactions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1375151660;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains statistics about administrator actions done by a user.
 */
class chatStatisticsAdministratorActionsInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Administrator user identifier.
  int53 user_id_;
  /// Number of messages deleted by the administrator.
  int32 deleted_message_count_;
  /// Number of users banned by the administrator.
  int32 banned_user_count_;
  /// Number of users restricted by the administrator.
  int32 restricted_user_count_;

  /**
   * Contains statistics about administrator actions done by a user.
   */
  chatStatisticsAdministratorActionsInfo();

  /**
   * Contains statistics about administrator actions done by a user.
   *
   * \param[in] user_id_ Administrator user identifier.
   * \param[in] deleted_message_count_ Number of messages deleted by the administrator.
   * \param[in] banned_user_count_ Number of users banned by the administrator.
   * \param[in] restricted_user_count_ Number of users restricted by the administrator.
   */
  chatStatisticsAdministratorActionsInfo(int53 user_id_, int32 deleted_message_count_, int32 banned_user_count_, int32 restricted_user_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -406467202;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ChatStatisticsObjectType;

/**
 * Contains statistics about interactions with a message sent in the chat or a story sent by the chat.
 */
class chatStatisticsInteractionInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of the object.
  object_ptr<ChatStatisticsObjectType> object_type_;
  /// Number of times the object was viewed.
  int32 view_count_;
  /// Number of times the object was forwarded.
  int32 forward_count_;
  /// Number of times reactions were added to the object.
  int32 reaction_count_;

  /**
   * Contains statistics about interactions with a message sent in the chat or a story sent by the chat.
   */
  chatStatisticsInteractionInfo();

  /**
   * Contains statistics about interactions with a message sent in the chat or a story sent by the chat.
   *
   * \param[in] object_type_ Type of the object.
   * \param[in] view_count_ Number of times the object was viewed.
   * \param[in] forward_count_ Number of times the object was forwarded.
   * \param[in] reaction_count_ Number of times reactions were added to the object.
   */
  chatStatisticsInteractionInfo(object_ptr<ChatStatisticsObjectType> &&object_type_, int32 view_count_, int32 forward_count_, int32 reaction_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1766496909;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains statistics about number of new members invited by a user.
 */
class chatStatisticsInviterInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;
  /// Number of new members invited by the user.
  int32 added_member_count_;

  /**
   * Contains statistics about number of new members invited by a user.
   */
  chatStatisticsInviterInfo();

  /**
   * Contains statistics about number of new members invited by a user.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] added_member_count_ Number of new members invited by the user.
   */
  chatStatisticsInviterInfo(int53 user_id_, int32 added_member_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 629396619;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains statistics about messages sent by a user.
 */
class chatStatisticsMessageSenderInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;
  /// Number of sent messages.
  int32 sent_message_count_;
  /// Average number of characters in sent messages; 0 if unknown.
  int32 average_character_count_;

  /**
   * Contains statistics about messages sent by a user.
   */
  chatStatisticsMessageSenderInfo();

  /**
   * Contains statistics about messages sent by a user.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] sent_message_count_ Number of sent messages.
   * \param[in] average_character_count_ Average number of characters in sent messages; 0 if unknown.
   */
  chatStatisticsMessageSenderInfo(int53 user_id_, int32 sent_message_count_, int32 average_character_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1762295371;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes type of object, for which statistics are provided.
 */
class ChatStatisticsObjectType: public Object {
 public:
};

/**
 * Describes a message sent in the chat.
 */
class chatStatisticsObjectTypeMessage final : public ChatStatisticsObjectType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Message identifier.
  int53 message_id_;

  /**
   * Describes a message sent in the chat.
   */
  chatStatisticsObjectTypeMessage();

  /**
   * Describes a message sent in the chat.
   *
   * \param[in] message_id_ Message identifier.
   */
  explicit chatStatisticsObjectTypeMessage(int53 message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1872700662;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a story sent by the chat.
 */
class chatStatisticsObjectTypeStory final : public ChatStatisticsObjectType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Story identifier.
  int32 story_id_;

  /**
   * Describes a story sent by the chat.
   */
  chatStatisticsObjectTypeStory();

  /**
   * Describes a story sent by the chat.
   *
   * \param[in] story_id_ Story identifier.
   */
  explicit chatStatisticsObjectTypeStory(int32 story_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 364575152;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class themeSettings;

/**
 * Describes a chat theme.
 */
class chatTheme final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Theme name.
  string name_;
  /// Theme settings for a light chat theme.
  object_ptr<themeSettings> light_settings_;
  /// Theme settings for a dark chat theme.
  object_ptr<themeSettings> dark_settings_;

  /**
   * Describes a chat theme.
   */
  chatTheme();

  /**
   * Describes a chat theme.
   *
   * \param[in] name_ Theme name.
   * \param[in] light_settings_ Theme settings for a light chat theme.
   * \param[in] dark_settings_ Theme settings for a dark chat theme.
   */
  chatTheme(string const &name_, object_ptr<themeSettings> &&light_settings_, object_ptr<themeSettings> &&dark_settings_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -113218503;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the type of chat.
 */
class ChatType: public Object {
 public:
};

/**
 * An ordinary chat with a user.
 */
class chatTypePrivate final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;

  /**
   * An ordinary chat with a user.
   */
  chatTypePrivate();

  /**
   * An ordinary chat with a user.
   *
   * \param[in] user_id_ User identifier.
   */
  explicit chatTypePrivate(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1579049844;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A basic group (a chat with 0-200 other users).
 */
class chatTypeBasicGroup final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Basic group identifier.
  int53 basic_group_id_;

  /**
   * A basic group (a chat with 0-200 other users).
   */
  chatTypeBasicGroup();

  /**
   * A basic group (a chat with 0-200 other users).
   *
   * \param[in] basic_group_id_ Basic group identifier.
   */
  explicit chatTypeBasicGroup(int53 basic_group_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 973884508;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A supergroup or channel (with unlimited members).
 */
class chatTypeSupergroup final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Supergroup or channel identifier.
  int53 supergroup_id_;
  /// True, if the supergroup is a channel.
  bool is_channel_;

  /**
   * A supergroup or channel (with unlimited members).
   */
  chatTypeSupergroup();

  /**
   * A supergroup or channel (with unlimited members).
   *
   * \param[in] supergroup_id_ Supergroup or channel identifier.
   * \param[in] is_channel_ True, if the supergroup is a channel.
   */
  chatTypeSupergroup(int53 supergroup_id_, bool is_channel_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1472570774;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A secret chat with a user.
 */
class chatTypeSecret final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Secret chat identifier.
  int32 secret_chat_id_;
  /// User identifier of the other user in the secret chat.
  int53 user_id_;

  /**
   * A secret chat with a user.
   */
  chatTypeSecret();

  /**
   * A secret chat with a user.
   *
   * \param[in] secret_chat_id_ Secret chat identifier.
   * \param[in] user_id_ User identifier of the other user in the secret chat.
   */
  chatTypeSecret(int32 secret_chat_id_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 862366513;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of chats.
 */
class chats final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of chats found.
  int32 total_count_;
  /// List of chat identifiers.
  array<int53> chat_ids_;

  /**
   * Represents a list of chats.
   */
  chats();

  /**
   * Represents a list of chats.
   *
   * \param[in] total_count_ Approximate total number of chats found.
   * \param[in] chat_ids_ List of chat identifiers.
   */
  chats(int32 total_count_, array<int53> &&chat_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1809654812;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether a username can be set for a chat.
 */
class CheckChatUsernameResult: public Object {
 public:
};

/**
 * The username can be set.
 */
class checkChatUsernameResultOk final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The username can be set.
   */
  checkChatUsernameResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1498956964;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The username is invalid.
 */
class checkChatUsernameResultUsernameInvalid final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The username is invalid.
   */
  checkChatUsernameResultUsernameInvalid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -636979370;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The username is occupied.
 */
class checkChatUsernameResultUsernameOccupied final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The username is occupied.
   */
  checkChatUsernameResultUsernameOccupied();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1320892201;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The username can be purchased at https://fragment.com. Information about the username can be received using getCollectibleItemInfo.
 */
class checkChatUsernameResultUsernamePurchasable final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The username can be purchased at https://fragment.com. Information about the username can be received using getCollectibleItemInfo.
   */
  checkChatUsernameResultUsernamePurchasable();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 5885529;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user has too many chats with username, one of them must be made private first.
 */
class checkChatUsernameResultPublicChatsTooMany final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user has too many chats with username, one of them must be made private first.
   */
  checkChatUsernameResultPublicChatsTooMany();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -659264388;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user can't be a member of a public supergroup.
 */
class checkChatUsernameResultPublicGroupsUnavailable final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user can't be a member of a public supergroup.
   */
  checkChatUsernameResultPublicGroupsUnavailable();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -51833641;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether a name can be used for a new sticker set.
 */
class CheckStickerSetNameResult: public Object {
 public:
};

/**
 * The name can be set.
 */
class checkStickerSetNameResultOk final : public CheckStickerSetNameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The name can be set.
   */
  checkStickerSetNameResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1404308904;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The name is invalid.
 */
class checkStickerSetNameResultNameInvalid final : public CheckStickerSetNameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The name is invalid.
   */
  checkStickerSetNameResultNameInvalid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 177992244;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The name is occupied.
 */
class checkStickerSetNameResultNameOccupied final : public CheckStickerSetNameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The name is occupied.
   */
  checkStickerSetNameResultNameOccupied();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1012980872;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class birthdate;

/**
 * Describes a user that had or will have a birthday soon.
 */
class closeBirthdayUser final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;
  /// Birthdate of the user.
  object_ptr<birthdate> birthdate_;

  /**
   * Describes a user that had or will have a birthday soon.
   */
  closeBirthdayUser();

  /**
   * Describes a user that had or will have a birthday soon.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] birthdate_ Birthdate of the user.
   */
  closeBirthdayUser(int53 user_id_, object_ptr<birthdate> &&birthdate_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2147067410;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class VectorPathCommand;

/**
 * Represents a closed vector path. The path begins at the end point of the last command. The coordinate system origin is in the upper-left corner.
 */
class closedVectorPath final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of vector path commands.
  array<object_ptr<VectorPathCommand>> commands_;

  /**
   * Represents a closed vector path. The path begins at the end point of the last command. The coordinate system origin is in the upper-left corner.
   */
  closedVectorPath();

  /**
   * Represents a closed vector path. The path begins at the end point of the last command. The coordinate system origin is in the upper-left corner.
   *
   * \param[in] commands_ List of vector path commands.
   */
  explicit closedVectorPath(array<object_ptr<VectorPathCommand>> &&commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 589951657;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a collectible item and its last purchase.
 */
class collectibleItemInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the item was purchased.
  int32 purchase_date_;
  /// Currency for the paid amount.
  string currency_;
  /// The paid amount, in the smallest units of the currency.
  int53 amount_;
  /// Cryptocurrency used to pay for the item.
  string cryptocurrency_;
  /// The paid amount, in the smallest units of the cryptocurrency.
  int64 cryptocurrency_amount_;
  /// Individual URL for the item on https://fragment.com.
  string url_;

  /**
   * Contains information about a collectible item and its last purchase.
   */
  collectibleItemInfo();

  /**
   * Contains information about a collectible item and its last purchase.
   *
   * \param[in] purchase_date_ Point in time (Unix timestamp) when the item was purchased.
   * \param[in] currency_ Currency for the paid amount.
   * \param[in] amount_ The paid amount, in the smallest units of the currency.
   * \param[in] cryptocurrency_ Cryptocurrency used to pay for the item.
   * \param[in] cryptocurrency_amount_ The paid amount, in the smallest units of the cryptocurrency.
   * \param[in] url_ Individual URL for the item on https://fragment.com.
   */
  collectibleItemInfo(int32 purchase_date_, string const &currency_, int53 amount_, string const &cryptocurrency_, int64 cryptocurrency_amount_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1460640717;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes a collectible item that can be purchased at https://fragment.com.
 */
class CollectibleItemType: public Object {
 public:
};

/**
 * A username.
 */
class collectibleItemTypeUsername final : public CollectibleItemType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The username.
  string username_;

  /**
   * A username.
   */
  collectibleItemTypeUsername();

  /**
   * A username.
   *
   * \param[in] username_ The username.
   */
  explicit collectibleItemTypeUsername(string const &username_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 458680273;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A phone number.
 */
class collectibleItemTypePhoneNumber final : public CollectibleItemType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The phone number.
  string phone_number_;

  /**
   * A phone number.
   */
  collectibleItemTypePhoneNumber();

  /**
   * A phone number.
   *
   * \param[in] phone_number_ The phone number.
   */
  explicit collectibleItemTypePhoneNumber(string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1256251714;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about one website the current user is logged in with Telegram.
 */
class connectedWebsite final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Website identifier.
  int64 id_;
  /// The domain name of the website.
  string domain_name_;
  /// User identifier of a bot linked with the website.
  int53 bot_user_id_;
  /// The version of a browser used to log in.
  string browser_;
  /// Operating system the browser is running on.
  string platform_;
  /// Point in time (Unix timestamp) when the user was logged in.
  int32 log_in_date_;
  /// Point in time (Unix timestamp) when obtained authorization was last used.
  int32 last_active_date_;
  /// IP address from which the user was logged in, in human-readable format.
  string ip_address_;
  /// Human-readable description of a country and a region from which the user was logged in, based on the IP address.
  string location_;

  /**
   * Contains information about one website the current user is logged in with Telegram.
   */
  connectedWebsite();

  /**
   * Contains information about one website the current user is logged in with Telegram.
   *
   * \param[in] id_ Website identifier.
   * \param[in] domain_name_ The domain name of the website.
   * \param[in] bot_user_id_ User identifier of a bot linked with the website.
   * \param[in] browser_ The version of a browser used to log in.
   * \param[in] platform_ Operating system the browser is running on.
   * \param[in] log_in_date_ Point in time (Unix timestamp) when the user was logged in.
   * \param[in] last_active_date_ Point in time (Unix timestamp) when obtained authorization was last used.
   * \param[in] ip_address_ IP address from which the user was logged in, in human-readable format.
   * \param[in] location_ Human-readable description of a country and a region from which the user was logged in, based on the IP address.
   */
  connectedWebsite(int64 id_, string const &domain_name_, int53 bot_user_id_, string const &browser_, string const &platform_, int32 log_in_date_, int32 last_active_date_, string const &ip_address_, string const &location_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1978115978;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class connectedWebsite;

/**
 * Contains a list of websites the current user is logged in with Telegram.
 */
class connectedWebsites final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of connected websites.
  array<object_ptr<connectedWebsite>> websites_;

  /**
   * Contains a list of websites the current user is logged in with Telegram.
   */
  connectedWebsites();

  /**
   * Contains a list of websites the current user is logged in with Telegram.
   *
   * \param[in] websites_ List of connected websites.
   */
  explicit connectedWebsites(array<object_ptr<connectedWebsite>> &&websites_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1727949694;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the current state of the connection to Telegram servers.
 */
class ConnectionState: public Object {
 public:
};

/**
 * Waiting for the network to become available. Use setNetworkType to change the available network type.
 */
class connectionStateWaitingForNetwork final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Waiting for the network to become available. Use setNetworkType to change the available network type.
   */
  connectionStateWaitingForNetwork();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1695405912;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Establishing a connection with a proxy server.
 */
class connectionStateConnectingToProxy final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Establishing a connection with a proxy server.
   */
  connectionStateConnectingToProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -93187239;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Establishing a connection to the Telegram servers.
 */
class connectionStateConnecting final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Establishing a connection to the Telegram servers.
   */
  connectionStateConnecting();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1298400670;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Downloading data expected to be received while the application was offline.
 */
class connectionStateUpdating final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Downloading data expected to be received while the application was offline.
   */
  connectionStateUpdating();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -188104009;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * There is a working connection to the Telegram servers.
 */
class connectionStateReady final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * There is a working connection to the Telegram servers.
   */
  connectionStateReady();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 48608492;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a user contact.
 */
class contact final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Phone number of the user.
  string phone_number_;
  /// First name of the user; 1-255 characters in length.
  string first_name_;
  /// Last name of the user.
  string last_name_;
  /// Additional data about the user in a form of vCard; 0-2048 bytes in length.
  string vcard_;
  /// Identifier of the user, if known; 0 otherwise.
  int53 user_id_;

  /**
   * Describes a user contact.
   */
  contact();

  /**
   * Describes a user contact.
   *
   * \param[in] phone_number_ Phone number of the user.
   * \param[in] first_name_ First name of the user; 1-255 characters in length.
   * \param[in] last_name_ Last name of the user.
   * \param[in] vcard_ Additional data about the user in a form of vCard; 0-2048 bytes in length.
   * \param[in] user_id_ Identifier of the user, if known; 0 otherwise.
   */
  contact(string const &phone_number_, string const &first_name_, string const &last_name_, string const &vcard_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1993844876;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a counter.
 */
class count final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Count.
  int32 count_;

  /**
   * Contains a counter.
   */
  count();

  /**
   * Contains a counter.
   *
   * \param[in] count_ Count.
   */
  explicit count(int32 count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1295577348;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class countryInfo;

/**
 * Contains information about countries.
 */
class countries final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The list of countries.
  array<object_ptr<countryInfo>> countries_;

  /**
   * Contains information about countries.
   */
  countries();

  /**
   * Contains information about countries.
   *
   * \param[in] countries_ The list of countries.
   */
  explicit countries(array<object_ptr<countryInfo>> &&countries_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1854211813;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a country.
 */
class countryInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A two-letter ISO 3166-1 alpha-2 country code.
  string country_code_;
  /// Native name of the country.
  string name_;
  /// English name of the country.
  string english_name_;
  /// True, if the country must be hidden from the list of all countries.
  bool is_hidden_;
  /// List of country calling codes.
  array<string> calling_codes_;

  /**
   * Contains information about a country.
   */
  countryInfo();

  /**
   * Contains information about a country.
   *
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code.
   * \param[in] name_ Native name of the country.
   * \param[in] english_name_ English name of the country.
   * \param[in] is_hidden_ True, if the country must be hidden from the list of all countries.
   * \param[in] calling_codes_ List of country calling codes.
   */
  countryInfo(string const &country_code_, string const &name_, string const &english_name_, bool is_hidden_, array<string> &&calling_codes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1617195722;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class failedToAddMembers;

/**
 * Contains information about a newly created basic group chat.
 */
class createdBasicGroupChat final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier.
  int53 chat_id_;
  /// Information about failed to add members.
  object_ptr<failedToAddMembers> failed_to_add_members_;

  /**
   * Contains information about a newly created basic group chat.
   */
  createdBasicGroupChat();

  /**
   * Contains information about a newly created basic group chat.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] failed_to_add_members_ Information about failed to add members.
   */
  createdBasicGroupChat(int53 chat_id_, object_ptr<failedToAddMembers> &&failed_to_add_members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -20417068;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes the current weather.
 */
class currentWeather final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Temperature, in degree Celsius.
  double temperature_;
  /// Emoji representing the weather.
  string emoji_;

  /**
   * Describes the current weather.
   */
  currentWeather();

  /**
   * Describes the current weather.
   *
   * \param[in] temperature_ Temperature, in degree Celsius.
   * \param[in] emoji_ Emoji representing the weather.
   */
  currentWeather(double temperature_, string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -355555136;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the result of a custom request.
 */
class customRequestResult final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A JSON-serialized result.
  string result_;

  /**
   * Contains the result of a custom request.
   */
  customRequestResult();

  /**
   * Contains the result of a custom request.
   *
   * \param[in] result_ A JSON-serialized result.
   */
  explicit customRequestResult(string const &result_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2009960452;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains database statistics.
 */
class databaseStatistics final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Database statistics in an unspecified human-readable format.
  string statistics_;

  /**
   * Contains database statistics.
   */
  databaseStatistics();

  /**
   * Contains database statistics.
   *
   * \param[in] statistics_ Database statistics in an unspecified human-readable format.
   */
  explicit databaseStatistics(string const &statistics_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1123912880;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a date according to the Gregorian calendar.
 */
class date final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Day of the month; 1-31.
  int32 day_;
  /// Month; 1-12.
  int32 month_;
  /// Year; 1-9999.
  int32 year_;

  /**
   * Represents a date according to the Gregorian calendar.
   */
  date();

  /**
   * Represents a date according to the Gregorian calendar.
   *
   * \param[in] day_ Day of the month; 1-31.
   * \param[in] month_ Month; 1-12.
   * \param[in] year_ Year; 1-9999.
   */
  date(int32 day_, int32 month_, int32 year_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -277956960;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a date range.
 */
class dateRange final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) at which the date range begins.
  int32 start_date_;
  /// Point in time (Unix timestamp) at which the date range ends.
  int32 end_date_;

  /**
   * Represents a date range.
   */
  dateRange();

  /**
   * Represents a date range.
   *
   * \param[in] start_date_ Point in time (Unix timestamp) at which the date range begins.
   * \param[in] end_date_ Point in time (Unix timestamp) at which the date range ends.
   */
  dateRange(int32 start_date_, int32 end_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1360333926;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

/**
 * File with the date it was uploaded.
 */
class datedFile final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The file.
  object_ptr<file> file_;
  /// Point in time (Unix timestamp) when the file was uploaded.
  int32 date_;

  /**
   * File with the date it was uploaded.
   */
  datedFile();

  /**
   * File with the date it was uploaded.
   *
   * \param[in] file_ The file.
   * \param[in] date_ Point in time (Unix timestamp) when the file was uploaded.
   */
  datedFile(object_ptr<file> &&file_, int32 date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1840795491;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class formattedText;

/**
 * Contains information about a tg: deep link.
 */
class deepLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text to be shown to the user.
  object_ptr<formattedText> text_;
  /// True, if the user must be asked to update the application.
  bool need_update_application_;

  /**
   * Contains information about a tg: deep link.
   */
  deepLinkInfo();

  /**
   * Contains information about a tg: deep link.
   *
   * \param[in] text_ Text to be shown to the user.
   * \param[in] need_update_application_ True, if the user must be asked to update the application.
   */
  deepLinkInfo(object_ptr<formattedText> &&text_, bool need_update_application_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1864081662;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents a data needed to subscribe for push notifications through registerDevice method. To use specific push notification service, the correct application platform must be specified and a valid server authentication data must be uploaded at https://my.telegram.org.
 */
class DeviceToken: public Object {
 public:
};

/**
 * A token for Firebase Cloud Messaging.
 */
class deviceTokenFirebaseCloudMessaging final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Device registration token; may be empty to deregister a device.
  string token_;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;

  /**
   * A token for Firebase Cloud Messaging.
   */
  deviceTokenFirebaseCloudMessaging();

  /**
   * A token for Firebase Cloud Messaging.
   *
   * \param[in] token_ Device registration token; may be empty to deregister a device.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenFirebaseCloudMessaging(string const &token_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -797881849;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Apple Push Notification service.
 */
class deviceTokenApplePush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Device token; may be empty to deregister a device.
  string device_token_;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;

  /**
   * A token for Apple Push Notification service.
   */
  deviceTokenApplePush();

  /**
   * A token for Apple Push Notification service.
   *
   * \param[in] device_token_ Device token; may be empty to deregister a device.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   */
  deviceTokenApplePush(string const &device_token_, bool is_app_sandbox_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 387541955;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Apple Push Notification service VoIP notifications.
 */
class deviceTokenApplePushVoIP final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Device token; may be empty to deregister a device.
  string device_token_;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;

  /**
   * A token for Apple Push Notification service VoIP notifications.
   */
  deviceTokenApplePushVoIP();

  /**
   * A token for Apple Push Notification service VoIP notifications.
   *
   * \param[in] device_token_ Device token; may be empty to deregister a device.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenApplePushVoIP(string const &device_token_, bool is_app_sandbox_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 804275689;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Windows Push Notification Services.
 */
class deviceTokenWindowsPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The access token that will be used to send notifications; may be empty to deregister a device.
  string access_token_;

  /**
   * A token for Windows Push Notification Services.
   */
  deviceTokenWindowsPush();

  /**
   * A token for Windows Push Notification Services.
   *
   * \param[in] access_token_ The access token that will be used to send notifications; may be empty to deregister a device.
   */
  explicit deviceTokenWindowsPush(string const &access_token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1410514289;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Microsoft Push Notification Service.
 */
class deviceTokenMicrosoftPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Push notification channel URI; may be empty to deregister a device.
  string channel_uri_;

  /**
   * A token for Microsoft Push Notification Service.
   */
  deviceTokenMicrosoftPush();

  /**
   * A token for Microsoft Push Notification Service.
   *
   * \param[in] channel_uri_ Push notification channel URI; may be empty to deregister a device.
   */
  explicit deviceTokenMicrosoftPush(string const &channel_uri_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1224269900;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Microsoft Push Notification Service VoIP channel.
 */
class deviceTokenMicrosoftPushVoIP final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Push notification channel URI; may be empty to deregister a device.
  string channel_uri_;

  /**
   * A token for Microsoft Push Notification Service VoIP channel.
   */
  deviceTokenMicrosoftPushVoIP();

  /**
   * A token for Microsoft Push Notification Service VoIP channel.
   *
   * \param[in] channel_uri_ Push notification channel URI; may be empty to deregister a device.
   */
  explicit deviceTokenMicrosoftPushVoIP(string const &channel_uri_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -785603759;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for web Push API.
 */
class deviceTokenWebPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
  string endpoint_;
  /// Base64url-encoded P-256 elliptic curve Diffie-Hellman public key.
  string p256dh_base64url_;
  /// Base64url-encoded authentication secret.
  string auth_base64url_;

  /**
   * A token for web Push API.
   */
  deviceTokenWebPush();

  /**
   * A token for web Push API.
   *
   * \param[in] endpoint_ Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
   * \param[in] p256dh_base64url_ Base64url-encoded P-256 elliptic curve Diffie-Hellman public key.
   * \param[in] auth_base64url_ Base64url-encoded authentication secret.
   */
  deviceTokenWebPush(string const &endpoint_, string const &p256dh_base64url_, string const &auth_base64url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1694507273;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Simple Push API for Firefox OS.
 */
class deviceTokenSimplePush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
  string endpoint_;

  /**
   * A token for Simple Push API for Firefox OS.
   */
  deviceTokenSimplePush();

  /**
   * A token for Simple Push API for Firefox OS.
   *
   * \param[in] endpoint_ Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
   */
  explicit deviceTokenSimplePush(string const &endpoint_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 49584736;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Ubuntu Push Client service.
 */
class deviceTokenUbuntuPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Token; may be empty to deregister a device.
  string token_;

  /**
   * A token for Ubuntu Push Client service.
   */
  deviceTokenUbuntuPush();

  /**
   * A token for Ubuntu Push Client service.
   *
   * \param[in] token_ Token; may be empty to deregister a device.
   */
  explicit deviceTokenUbuntuPush(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1782320422;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for BlackBerry Push Service.
 */
class deviceTokenBlackBerryPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Token; may be empty to deregister a device.
  string token_;

  /**
   * A token for BlackBerry Push Service.
   */
  deviceTokenBlackBerryPush();

  /**
   * A token for BlackBerry Push Service.
   *
   * \param[in] token_ Token; may be empty to deregister a device.
   */
  explicit deviceTokenBlackBerryPush(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1559167234;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for Tizen Push Service.
 */
class deviceTokenTizenPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Push service registration identifier; may be empty to deregister a device.
  string reg_id_;

  /**
   * A token for Tizen Push Service.
   */
  deviceTokenTizenPush();

  /**
   * A token for Tizen Push Service.
   *
   * \param[in] reg_id_ Push service registration identifier; may be empty to deregister a device.
   */
  explicit deviceTokenTizenPush(string const &reg_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1359947213;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A token for HUAWEI Push Service.
 */
class deviceTokenHuaweiPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Device registration token; may be empty to deregister a device.
  string token_;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;

  /**
   * A token for HUAWEI Push Service.
   */
  deviceTokenHuaweiPush();

  /**
   * A token for HUAWEI Push Service.
   *
   * \param[in] token_ Device registration token; may be empty to deregister a device.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenHuaweiPush(string const &token_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1989103142;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class sticker;

/**
 * This class is an abstract base class.
 * Contains animated stickers which must be used for dice animation rendering.
 */
class DiceStickers: public Object {
 public:
};

/**
 * A regular animated sticker.
 */
class diceStickersRegular final : public DiceStickers {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The animated sticker with the dice animation.
  object_ptr<sticker> sticker_;

  /**
   * A regular animated sticker.
   */
  diceStickersRegular();

  /**
   * A regular animated sticker.
   *
   * \param[in] sticker_ The animated sticker with the dice animation.
   */
  explicit diceStickersRegular(object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -740299570;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Animated stickers to be combined into a slot machine.
 */
class diceStickersSlotMachine final : public DiceStickers {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The animated sticker with the slot machine background. The background animation must start playing after all reel animations finish.
  object_ptr<sticker> background_;
  /// The animated sticker with the lever animation. The lever animation must play once in the initial dice state.
  object_ptr<sticker> lever_;
  /// The animated sticker with the left reel.
  object_ptr<sticker> left_reel_;
  /// The animated sticker with the center reel.
  object_ptr<sticker> center_reel_;
  /// The animated sticker with the right reel.
  object_ptr<sticker> right_reel_;

  /**
   * Animated stickers to be combined into a slot machine.
   */
  diceStickersSlotMachine();

  /**
   * Animated stickers to be combined into a slot machine.
   *
   * \param[in] background_ The animated sticker with the slot machine background. The background animation must start playing after all reel animations finish.
   * \param[in] lever_ The animated sticker with the lever animation. The lever animation must play once in the initial dice state.
   * \param[in] left_reel_ The animated sticker with the left reel.
   * \param[in] center_reel_ The animated sticker with the center reel.
   * \param[in] right_reel_ The animated sticker with the right reel.
   */
  diceStickersSlotMachine(object_ptr<sticker> &&background_, object_ptr<sticker> &&lever_, object_ptr<sticker> &&left_reel_, object_ptr<sticker> &&center_reel_, object_ptr<sticker> &&right_reel_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -375223124;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class file;

class minithumbnail;

class thumbnail;

/**
 * Describes a document of any type.
 */
class document final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Original name of the file; as defined by the sender.
  string file_name_;
  /// MIME type of the file; as defined by the sender.
  string mime_type_;
  /// Document minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  /// Document thumbnail in JPEG or PNG format (PNG will be used only for background patterns); as defined by the sender; may be null.
  object_ptr<thumbnail> thumbnail_;
  /// File containing the document.
  object_ptr<file> document_;

  /**
   * Describes a document of any type.
   */
  document();

  /**
   * Describes a document of any type.
   *
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ MIME type of the file; as defined by the sender.
   * \param[in] minithumbnail_ Document minithumbnail; may be null.
   * \param[in] thumbnail_ Document thumbnail in JPEG or PNG format (PNG will be used only for background patterns); as defined by the sender; may be null.
   * \param[in] document_ File containing the document.
   */
  document(string const &file_name_, string const &mime_type_, object_ptr<minithumbnail> &&minithumbnail_, object_ptr<thumbnail> &&thumbnail_, object_ptr<file> &&document_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1357271080;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains number of being downloaded and recently downloaded files found.
 */
class downloadedFileCounts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Number of active file downloads found, including paused.
  int32 active_count_;
  /// Number of paused file downloads found.
  int32 paused_count_;
  /// Number of completed file downloads found.
  int32 completed_count_;

  /**
   * Contains number of being downloaded and recently downloaded files found.
   */
  downloadedFileCounts();

  /**
   * Contains number of being downloaded and recently downloaded files found.
   *
   * \param[in] active_count_ Number of active file downloads found, including paused.
   * \param[in] paused_count_ Number of paused file downloads found.
   * \param[in] completed_count_ Number of completed file downloads found.
   */
  downloadedFileCounts(int32 active_count_, int32 paused_count_, int32 completed_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1973999550;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputMessageContent;

class InputMessageReplyTo;

/**
 * Contains information about a message draft.
 */
class draftMessage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about the message to be replied; must be of the type inputMessageReplyToMessage; may be null if none.
  object_ptr<InputMessageReplyTo> reply_to_;
  /// Point in time (Unix timestamp) when the draft was created.
  int32 date_;
  /// Content of the message draft; must be of the type inputMessageText, inputMessageVideoNote, or inputMessageVoiceNote.
  object_ptr<InputMessageContent> input_message_text_;
  /// Identifier of the effect to apply to the message when it is sent; 0 if none.
  int64 effect_id_;

  /**
   * Contains information about a message draft.
   */
  draftMessage();

  /**
   * Contains information about a message draft.
   *
   * \param[in] reply_to_ Information about the message to be replied; must be of the type inputMessageReplyToMessage; may be null if none.
   * \param[in] date_ Point in time (Unix timestamp) when the draft was created.
   * \param[in] input_message_text_ Content of the message draft; must be of the type inputMessageText, inputMessageVideoNote, or inputMessageVoiceNote.
   * \param[in] effect_id_ Identifier of the effect to apply to the message when it is sent; 0 if none.
   */
  draftMessage(object_ptr<InputMessageReplyTo> &&reply_to_, int32 date_, object_ptr<InputMessageContent> &&input_message_text_, int64 effect_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1125328749;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains authentication data for an email address.
 */
class EmailAddressAuthentication: public Object {
 public:
};

/**
 * An authentication code delivered to a user's email address.
 */
class emailAddressAuthenticationCode final : public EmailAddressAuthentication {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The code.
  string code_;

  /**
   * An authentication code delivered to a user's email address.
   */
  emailAddressAuthenticationCode();

  /**
   * An authentication code delivered to a user's email address.
   *
   * \param[in] code_ The code.
   */
  explicit emailAddressAuthenticationCode(string const &code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -993257022;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication token received through Apple ID.
 */
class emailAddressAuthenticationAppleId final : public EmailAddressAuthentication {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The token.
  string token_;

  /**
   * An authentication token received through Apple ID.
   */
  emailAddressAuthenticationAppleId();

  /**
   * An authentication token received through Apple ID.
   *
   * \param[in] token_ The token.
   */
  explicit emailAddressAuthenticationAppleId(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 633948265;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An authentication token received through Google ID.
 */
class emailAddressAuthenticationGoogleId final : public EmailAddressAuthentication {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The token.
  string token_;

  /**
   * An authentication token received through Google ID.
   */
  emailAddressAuthenticationGoogleId();

  /**
   * An authentication token received through Google ID.
   *
   * \param[in] token_ The token.
   */
  explicit emailAddressAuthenticationGoogleId(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -19142846;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Information about the email address authentication code that was sent.
 */
class emailAddressAuthenticationCodeInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Pattern of the email address to which an authentication code was sent.
  string email_address_pattern_;
  /// Length of the code; 0 if unknown.
  int32 length_;

  /**
   * Information about the email address authentication code that was sent.
   */
  emailAddressAuthenticationCodeInfo();

  /**
   * Information about the email address authentication code that was sent.
   *
   * \param[in] email_address_pattern_ Pattern of the email address to which an authentication code was sent.
   * \param[in] length_ Length of the code; 0 if unknown.
   */
  emailAddressAuthenticationCodeInfo(string const &email_address_pattern_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1151066659;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes reset state of an email address.
 */
class EmailAddressResetState: public Object {
 public:
};

/**
 * Email address can be reset after the given period. Call resetAuthenticationEmailAddress to reset it and allow the user to authorize with a code sent to the user's phone number.
 */
class emailAddressResetStateAvailable final : public EmailAddressResetState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Time required to wait before the email address can be reset; 0 if the user is subscribed to Telegram Premium.
  int32 wait_period_;

  /**
   * Email address can be reset after the given period. Call resetAuthenticationEmailAddress to reset it and allow the user to authorize with a code sent to the user's phone number.
   */
  emailAddressResetStateAvailable();

  /**
   * Email address can be reset after the given period. Call resetAuthenticationEmailAddress to reset it and allow the user to authorize with a code sent to the user's phone number.
   *
   * \param[in] wait_period_ Time required to wait before the email address can be reset; 0 if the user is subscribed to Telegram Premium.
   */
  explicit emailAddressResetStateAvailable(int32 wait_period_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1917177600;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Email address reset has already been requested. Call resetAuthenticationEmailAddress to check whether immediate reset is possible.
 */
class emailAddressResetStatePending final : public EmailAddressResetState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Left time before the email address will be reset, in seconds. updateAuthorizationState is not sent when this field changes.
  int32 reset_in_;

  /**
   * Email address reset has already been requested. Call resetAuthenticationEmailAddress to check whether immediate reset is possible.
   */
  emailAddressResetStatePending();

  /**
   * Email address reset has already been requested. Call resetAuthenticationEmailAddress to check whether immediate reset is possible.
   *
   * \param[in] reset_in_ Left time before the email address will be reset, in seconds. updateAuthorizationState is not sent when this field changes.
   */
  explicit emailAddressResetStatePending(int32 reset_in_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1885966805;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class emojiCategory;

/**
 * Represents a list of emoji categories.
 */
class emojiCategories final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of categories.
  array<object_ptr<emojiCategory>> categories_;

  /**
   * Represents a list of emoji categories.
   */
  emojiCategories();

  /**
   * Represents a list of emoji categories.
   *
   * \param[in] categories_ List of categories.
   */
  explicit emojiCategories(array<object_ptr<emojiCategory>> &&categories_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1455387824;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class EmojiCategorySource;

class sticker;

/**
 * Describes an emoji category.
 */
class emojiCategory final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the category.
  string name_;
  /// Custom emoji sticker, which represents icon of the category.
  object_ptr<sticker> icon_;
  /// Source of stickers for the emoji category.
  object_ptr<EmojiCategorySource> source_;
  /// True, if the category must be shown first when choosing a sticker for the start page.
  bool is_greeting_;

  /**
   * Describes an emoji category.
   */
  emojiCategory();

  /**
   * Describes an emoji category.
   *
   * \param[in] name_ Name of the category.
   * \param[in] icon_ Custom emoji sticker, which represents icon of the category.
   * \param[in] source_ Source of stickers for the emoji category.
   * \param[in] is_greeting_ True, if the category must be shown first when choosing a sticker for the start page.
   */
  emojiCategory(string const &name_, object_ptr<sticker> &&icon_, object_ptr<EmojiCategorySource> &&source_, bool is_greeting_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 571335919;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes source of stickers for an emoji category.
 */
class EmojiCategorySource: public Object {
 public:
};

/**
 * The category contains a list of similar emoji to search for in getStickers and searchStickers for stickers, or getInlineQueryResults with the bot getOption(&quot;animation_search_bot_username&quot;) for animations.
 */
class emojiCategorySourceSearch final : public EmojiCategorySource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of emojis to search for.
  array<string> emojis_;

  /**
   * The category contains a list of similar emoji to search for in getStickers and searchStickers for stickers, or getInlineQueryResults with the bot getOption(&quot;animation_search_bot_username&quot;) for animations.
   */
  emojiCategorySourceSearch();

  /**
   * The category contains a list of similar emoji to search for in getStickers and searchStickers for stickers, or getInlineQueryResults with the bot getOption(&quot;animation_search_bot_username&quot;) for animations.
   *
   * \param[in] emojis_ List of emojis to search for.
   */
  explicit emojiCategorySourceSearch(array<string> &&emojis_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -453260262;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The category contains premium stickers that must be found by getPremiumStickers.
 */
class emojiCategorySourcePremium final : public EmojiCategorySource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The category contains premium stickers that must be found by getPremiumStickers.
   */
  emojiCategorySourcePremium();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1932358388;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes type of emoji category.
 */
class EmojiCategoryType: public Object {
 public:
};

/**
 * The category must be used by default (e.g., for custom emoji or animation search).
 */
class emojiCategoryTypeDefault final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The category must be used by default (e.g., for custom emoji or animation search).
   */
  emojiCategoryTypeDefault();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1188782699;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The category must be used by default for regular sticker selection. It may contain greeting emoji category and premium stickers.
 */
class emojiCategoryTypeRegularStickers final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The category must be used by default for regular sticker selection. It may contain greeting emoji category and premium stickers.
   */
  emojiCategoryTypeRegularStickers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1337484846;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The category must be used for emoji status selection.
 */
class emojiCategoryTypeEmojiStatus final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The category must be used for emoji status selection.
   */
  emojiCategoryTypeEmojiStatus();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1381282631;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The category must be used for chat photo emoji selection.
 */
class emojiCategoryTypeChatPhoto final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The category must be used for chat photo emoji selection.
   */
  emojiCategoryTypeChatPhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1059063081;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an emoji with its keyword.
 */
class emojiKeyword final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The emoji.
  string emoji_;
  /// The keyword.
  string keyword_;

  /**
   * Represents an emoji with its keyword.
   */
  emojiKeyword();

  /**
   * Represents an emoji with its keyword.
   *
   * \param[in] emoji_ The emoji.
   * \param[in] keyword_ The keyword.
   */
  emojiKeyword(string const &emoji_, string const &keyword_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2112285985;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class emojiKeyword;

/**
 * Represents a list of emojis with their keywords.
 */
class emojiKeywords final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of emojis with their keywords.
  array<object_ptr<emojiKeyword>> emoji_keywords_;

  /**
   * Represents a list of emojis with their keywords.
   */
  emojiKeywords();

  /**
   * Represents a list of emojis with their keywords.
   *
   * \param[in] emoji_keywords_ List of emojis with their keywords.
   */
  explicit emojiKeywords(array<object_ptr<emojiKeyword>> &&emoji_keywords_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 689723339;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class sticker;

/**
 * Contains information about an emoji reaction.
 */
class emojiReaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text representation of the reaction.
  string emoji_;
  /// Reaction title.
  string title_;
  /// True, if the reaction can be added to new messages and enabled in chats.
  bool is_active_;
  /// Static icon for the reaction.
  object_ptr<sticker> static_icon_;
  /// Appear animation for the reaction.
  object_ptr<sticker> appear_animation_;
  /// Select animation for the reaction.
  object_ptr<sticker> select_animation_;
  /// Activate animation for the reaction.
  object_ptr<sticker> activate_animation_;
  /// Effect animation for the reaction.
  object_ptr<sticker> effect_animation_;
  /// Around animation for the reaction; may be null.
  object_ptr<sticker> around_animation_;
  /// Center animation for the reaction; may be null.
  object_ptr<sticker> center_animation_;

  /**
   * Contains information about an emoji reaction.
   */
  emojiReaction();

  /**
   * Contains information about an emoji reaction.
   *
   * \param[in] emoji_ Text representation of the reaction.
   * \param[in] title_ Reaction title.
   * \param[in] is_active_ True, if the reaction can be added to new messages and enabled in chats.
   * \param[in] static_icon_ Static icon for the reaction.
   * \param[in] appear_animation_ Appear animation for the reaction.
   * \param[in] select_animation_ Select animation for the reaction.
   * \param[in] activate_animation_ Activate animation for the reaction.
   * \param[in] effect_animation_ Effect animation for the reaction.
   * \param[in] around_animation_ Around animation for the reaction; may be null.
   * \param[in] center_animation_ Center animation for the reaction; may be null.
   */
  emojiReaction(string const &emoji_, string const &title_, bool is_active_, object_ptr<sticker> &&static_icon_, object_ptr<sticker> &&appear_animation_, object_ptr<sticker> &&select_animation_, object_ptr<sticker> &&activate_animation_, object_ptr<sticker> &&effect_animation_, object_ptr<sticker> &&around_animation_, object_ptr<sticker> &&center_animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1616063583;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a custom emoji to be shown instead of the Telegram Premium badge.
 */
class emojiStatus final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the custom emoji in stickerFormatTgs format.
  int64 custom_emoji_id_;
  /// Point in time (Unix timestamp) when the status will expire; 0 if never.
  int32 expiration_date_;

  /**
   * Describes a custom emoji to be shown instead of the Telegram Premium badge.
   */
  emojiStatus();

  /**
   * Describes a custom emoji to be shown instead of the Telegram Premium badge.
   *
   * \param[in] custom_emoji_id_ Identifier of the custom emoji in stickerFormatTgs format.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the status will expire; 0 if never.
   */
  emojiStatus(int64 custom_emoji_id_, int32 expiration_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -606529994;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of custom emoji identifiers for emoji statuses.
 */
class emojiStatuses final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The list of custom emoji identifiers.
  array<int64> custom_emoji_ids_;

  /**
   * Contains a list of custom emoji identifiers for emoji statuses.
   */
  emojiStatuses();

  /**
   * Contains a list of custom emoji identifiers for emoji statuses.
   *
   * \param[in] custom_emoji_ids_ The list of custom emoji identifiers.
   */
  explicit emojiStatuses(array<int64> &&custom_emoji_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -377859594;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of emojis.
 */
class emojis final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of emojis.
  array<string> emojis_;

  /**
   * Represents a list of emojis.
   */
  emojis();

  /**
   * Represents a list of emojis.
   *
   * \param[in] emojis_ List of emojis.
   */
  explicit emojis(array<string> &&emojis_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 950339552;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains encrypted Telegram Passport data credentials.
 */
class encryptedCredentials final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The encrypted credentials.
  bytes data_;
  /// The decrypted data hash.
  bytes hash_;
  /// Secret for data decryption, encrypted with the service's public key.
  bytes secret_;

  /**
   * Contains encrypted Telegram Passport data credentials.
   */
  encryptedCredentials();

  /**
   * Contains encrypted Telegram Passport data credentials.
   *
   * \param[in] data_ The encrypted credentials.
   * \param[in] hash_ The decrypted data hash.
   * \param[in] secret_ Secret for data decryption, encrypted with the service's public key.
   */
  encryptedCredentials(bytes const &data_, bytes const &hash_, bytes const &secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1331106766;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class PassportElementType;

class datedFile;

/**
 * Contains information about an encrypted Telegram Passport element; for bots only.
 */
class encryptedPassportElement final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of Telegram Passport element.
  object_ptr<PassportElementType> type_;
  /// Encrypted JSON-encoded data about the user.
  bytes data_;
  /// The front side of an identity document.
  object_ptr<datedFile> front_side_;
  /// The reverse side of an identity document; may be null.
  object_ptr<datedFile> reverse_side_;
  /// Selfie with the document; may be null.
  object_ptr<datedFile> selfie_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<datedFile>> translation_;
  /// List of attached files.
  array<object_ptr<datedFile>> files_;
  /// Unencrypted data, phone number or email address.
  string value_;
  /// Hash of the entire element.
  string hash_;

  /**
   * Contains information about an encrypted Telegram Passport element; for bots only.
   */
  encryptedPassportElement();

  /**
   * Contains information about an encrypted Telegram Passport element; for bots only.
   *
   * \param[in] type_ Type of Telegram Passport element.
   * \param[in] data_ Encrypted JSON-encoded data about the user.
   * \param[in] front_side_ The front side of an identity document.
   * \param[in] reverse_side_ The reverse side of an identity document; may be null.
   * \param[in] selfie_ Selfie with the document; may be null.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   * \param[in] files_ List of attached files.
   * \param[in] value_ Unencrypted data, phone number or email address.
   * \param[in] hash_ Hash of the entire element.
   */
  encryptedPassportElement(object_ptr<PassportElementType> &&type_, bytes const &data_, object_ptr<datedFile> &&front_side_, object_ptr<datedFile> &&reverse_side_, object_ptr<datedFile> &&selfie_, array<object_ptr<datedFile>> &&translation_, array<object_ptr<datedFile>> &&files_, string const &value_, string const &hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2002386193;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An object of this type can be returned on every function call, in case of an error.
 */
class error final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Error code; subject to future changes. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user.
  int32 code_;
  /// Error message; subject to future changes.
  string message_;

  /**
   * An object of this type can be returned on every function call, in case of an error.
   */
  error();

  /**
   * An object of this type can be returned on every function call, in case of an error.
   *
   * \param[in] code_ Error code; subject to future changes. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user.
   * \param[in] message_ Error message; subject to future changes.
   */
  error(int32 code_, string const &message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1679978726;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class formattedText;

/**
 * Describes a fact-check added to the message by an independent checker.
 */
class factCheck final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text of the fact-check.
  object_ptr<formattedText> text_;
  /// A two-letter ISO 3166-1 alpha-2 country code of the country for which the fact-check is shown.
  string country_code_;

  /**
   * Describes a fact-check added to the message by an independent checker.
   */
  factCheck();

  /**
   * Describes a fact-check added to the message by an independent checker.
   *
   * \param[in] text_ Text of the fact-check.
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code of the country for which the fact-check is shown.
   */
  factCheck(object_ptr<formattedText> &&text_, string const &country_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1048184552;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains information about a user that has failed to be added to a chat.
 */
class failedToAddMember final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;
  /// True, if subscription to Telegram Premium would have allowed to add the user to the chat.
  bool premium_would_allow_invite_;
  /// True, if subscription to Telegram Premium is required to send the user chat invite link.
  bool premium_required_to_send_messages_;

  /**
   * Contains information about a user that has failed to be added to a chat.
   */
  failedToAddMember();

  /**
   * Contains information about a user that has failed to be added to a chat.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] premium_would_allow_invite_ True, if subscription to Telegram Premium would have allowed to add the user to the chat.
   * \param[in] premium_required_to_send_messages_ True, if subscription to Telegram Premium is required to send the user chat invite link.
   */
  failedToAddMember(int53 user_id_, bool premium_would_allow_invite_, bool premium_required_to_send_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -282891070;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class failedToAddMember;

/**
 * Represents a list of users that has failed to be added to a chat.
 */
class failedToAddMembers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about users that weren't added to the chat.
  array<object_ptr<failedToAddMember>> failed_to_add_members_;

  /**
   * Represents a list of users that has failed to be added to a chat.
   */
  failedToAddMembers();

  /**
   * Represents a list of users that has failed to be added to a chat.
   *
   * \param[in] failed_to_add_members_ Information about users that weren't added to the chat.
   */
  explicit failedToAddMembers(array<object_ptr<failedToAddMember>> &&failed_to_add_members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -272587152;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class localFile;

class remoteFile;

/**
 * Represents a file.
 */
class file final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique file identifier.
  int32 id_;
  /// File size, in bytes; 0 if unknown.
  int53 size_;
  /// Approximate file size in bytes in case the exact file size is unknown. Can be used to show download/upload progress.
  int53 expected_size_;
  /// Information about the local copy of the file.
  object_ptr<localFile> local_;
  /// Information about the remote copy of the file.
  object_ptr<remoteFile> remote_;

  /**
   * Represents a file.
   */
  file();

  /**
   * Represents a file.
   *
   * \param[in] id_ Unique file identifier.
   * \param[in] size_ File size, in bytes; 0 if unknown.
   * \param[in] expected_size_ Approximate file size in bytes in case the exact file size is unknown. Can be used to show download/upload progress.
   * \param[in] local_ Information about the local copy of the file.
   * \param[in] remote_ Information about the remote copy of the file.
   */
  file(int32 id_, int53 size_, int53 expected_size_, object_ptr<localFile> &&local_, object_ptr<remoteFile> &&remote_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1263291956;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class message;

/**
 * Describes a file added to file download list.
 */
class fileDownload final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// File identifier.
  int32 file_id_;
  /// The message with the file.
  object_ptr<message> message_;
  /// Point in time (Unix timestamp) when the file was added to the download list.
  int32 add_date_;
  /// Point in time (Unix timestamp) when the file downloading was completed; 0 if the file downloading isn't completed.
  int32 complete_date_;
  /// True, if downloading of the file is paused.
  bool is_paused_;

  /**
   * Describes a file added to file download list.
   */
  fileDownload();

  /**
   * Describes a file added to file download list.
   *
   * \param[in] file_id_ File identifier.
   * \param[in] message_ The message with the file.
   * \param[in] add_date_ Point in time (Unix timestamp) when the file was added to the download list.
   * \param[in] complete_date_ Point in time (Unix timestamp) when the file downloading was completed; 0 if the file downloading isn't completed.
   * \param[in] is_paused_ True, if downloading of the file is paused.
   */
  fileDownload(int32 file_id_, object_ptr<message> &&message_, int32 add_date_, int32 complete_date_, bool is_paused_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2092100780;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains size of downloaded prefix of a file.
 */
class fileDownloadedPrefixSize final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The prefix size, in bytes.
  int53 size_;

  /**
   * Contains size of downloaded prefix of a file.
   */
  fileDownloadedPrefixSize();

  /**
   * Contains size of downloaded prefix of a file.
   *
   * \param[in] size_ The prefix size, in bytes.
   */
  explicit fileDownloadedPrefixSize(int53 size_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2015205381;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a part of a file.
 */
class filePart final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// File bytes.
  bytes data_;

  /**
   * Contains a part of a file.
   */
  filePart();

  /**
   * Contains a part of a file.
   *
   * \param[in] data_ File bytes.
   */
  explicit filePart(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 911821878;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents the type of file.
 */
class FileType: public Object {
 public:
};

/**
 * The data is not a file.
 */
class fileTypeNone final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The data is not a file.
   */
  fileTypeNone();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2003009189;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is an animation.
 */
class fileTypeAnimation final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is an animation.
   */
  fileTypeAnimation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -290816582;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is an audio file.
 */
class fileTypeAudio final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is an audio file.
   */
  fileTypeAudio();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -709112160;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a document.
 */
class fileTypeDocument final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a document.
   */
  fileTypeDocument();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -564722929;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a notification sound.
 */
class fileTypeNotificationSound final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a notification sound.
   */
  fileTypeNotificationSound();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1020289271;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a photo.
 */
class fileTypePhoto final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a photo.
   */
  fileTypePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1718914651;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a photo published as a story.
 */
class fileTypePhotoStory final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a photo published as a story.
   */
  fileTypePhotoStory();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2018995956;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a profile photo.
 */
class fileTypeProfilePhoto final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a profile photo.
   */
  fileTypeProfilePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1795089315;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file was sent to a secret chat (the file type is not known to the server).
 */
class fileTypeSecret final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file was sent to a secret chat (the file type is not known to the server).
   */
  fileTypeSecret();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1871899401;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a thumbnail of a file from a secret chat.
 */
class fileTypeSecretThumbnail final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a thumbnail of a file from a secret chat.
   */
  fileTypeSecretThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1401326026;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a file from Secure storage used for storing Telegram Passport files.
 */
class fileTypeSecure final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a file from Secure storage used for storing Telegram Passport files.
   */
  fileTypeSecure();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1419133146;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a self-destructing photo in a private chat.
 */
class fileTypeSelfDestructingPhoto final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a self-destructing photo in a private chat.
   */
  fileTypeSelfDestructingPhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2077176475;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a self-destructing video in a private chat.
 */
class fileTypeSelfDestructingVideo final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a self-destructing video in a private chat.
   */
  fileTypeSelfDestructingVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1223900123;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a self-destructing video note in a private chat.
 */
class fileTypeSelfDestructingVideoNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a self-destructing video note in a private chat.
   */
  fileTypeSelfDestructingVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1495274177;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a self-destructing voice note in a private chat.
 */
class fileTypeSelfDestructingVoiceNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a self-destructing voice note in a private chat.
   */
  fileTypeSelfDestructingVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1691409181;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a sticker.
 */
class fileTypeSticker final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a sticker.
   */
  fileTypeSticker();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 475233385;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a thumbnail of another file.
 */
class fileTypeThumbnail final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a thumbnail of another file.
   */
  fileTypeThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -12443298;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file type is not yet known.
 */
class fileTypeUnknown final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file type is not yet known.
   */
  fileTypeUnknown();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2011566768;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a video.
 */
class fileTypeVideo final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a video.
   */
  fileTypeVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1430816539;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a video note.
 */
class fileTypeVideoNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a video note.
   */
  fileTypeVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -518412385;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a video published as a story.
 */
class fileTypeVideoStory final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a video published as a story.
   */
  fileTypeVideoStory();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2146754143;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a voice note.
 */
class fileTypeVoiceNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a voice note.
   */
  fileTypeVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -588681661;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file is a wallpaper or a background pattern.
 */
class fileTypeWallpaper final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The file is a wallpaper or a background pattern.
   */
  fileTypeWallpaper();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1854930076;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains settings for Firebase Authentication in the official applications.
 */
class FirebaseAuthenticationSettings: public Object {
 public:
};

/**
 * Settings for Firebase Authentication in the official Android application.
 */
class firebaseAuthenticationSettingsAndroid final : public FirebaseAuthenticationSettings {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * Settings for Firebase Authentication in the official Android application.
   */
  firebaseAuthenticationSettingsAndroid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1771112932;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Settings for Firebase Authentication in the official iOS application.
 */
class firebaseAuthenticationSettingsIos final : public FirebaseAuthenticationSettings {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Device token from Apple Push Notification service.
  string device_token_;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;

  /**
   * Settings for Firebase Authentication in the official iOS application.
   */
  firebaseAuthenticationSettingsIos();

  /**
   * Settings for Firebase Authentication in the official iOS application.
   *
   * \param[in] device_token_ Device token from Apple Push Notification service.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   */
  firebaseAuthenticationSettingsIos(string const &device_token_, bool is_app_sandbox_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 222930116;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes parameters to be used for device verification.
 */
class FirebaseDeviceVerificationParameters: public Object {
 public:
};

/**
 * Device verification must be performed with the SafetyNet Attestation API.
 */
class firebaseDeviceVerificationParametersSafetyNet final : public FirebaseDeviceVerificationParameters {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Nonce to pass to the SafetyNet Attestation API.
  bytes nonce_;

  /**
   * Device verification must be performed with the SafetyNet Attestation API.
   */
  firebaseDeviceVerificationParametersSafetyNet();

  /**
   * Device verification must be performed with the SafetyNet Attestation API.
   *
   * \param[in] nonce_ Nonce to pass to the SafetyNet Attestation API.
   */
  explicit firebaseDeviceVerificationParametersSafetyNet(bytes const &nonce_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 731296497;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Device verification must be performed with the classic Play Integrity verification (https://developer.android.com/google/play/integrity/classic).
 */
class firebaseDeviceVerificationParametersPlayIntegrity final : public FirebaseDeviceVerificationParameters {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Base64url-encoded nonce to pass to the Play Integrity API.
  string nonce_;
  /// Cloud project number to pass to the Play Integrity API.
  int64 cloud_project_number_;

  /**
   * Device verification must be performed with the classic Play Integrity verification (https://developer.android.com/google/play/integrity/classic).
   */
  firebaseDeviceVerificationParametersPlayIntegrity();

  /**
   * Device verification must be performed with the classic Play Integrity verification (https://developer.android.com/google/play/integrity/classic).
   *
   * \param[in] nonce_ Base64url-encoded nonce to pass to the Play Integrity API.
   * \param[in] cloud_project_number_ Cloud project number to pass to the Play Integrity API.
   */
  firebaseDeviceVerificationParametersPlayIntegrity(string const &nonce_, int64 cloud_project_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -889936502;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class textEntity;

/**
 * A text with some entities.
 */
class formattedText final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The text.
  string text_;
  /// Entities contained in the text. Entities can be nested, but must not mutually intersect with each other. Pre, Code and PreCode entities can't contain other entities. BlockQuote entities can't contain other BlockQuote entities. Bold, Italic, Underline, Strikethrough, and Spoiler entities can contain and can be part of any other entities. All other entities can't contain each other.
  array<object_ptr<textEntity>> entities_;

  /**
   * A text with some entities.
   */
  formattedText();

  /**
   * A text with some entities.
   *
   * \param[in] text_ The text.
   * \param[in] entities_ Entities contained in the text. Entities can be nested, but must not mutually intersect with each other. Pre, Code and PreCode entities can't contain other entities. BlockQuote entities can't contain other BlockQuote entities. Bold, Italic, Underline, Strikethrough, and Spoiler entities can contain and can be part of any other entities. All other entities can't contain each other.
   */
  formattedText(string const &text_, array<object_ptr<textEntity>> &&entities_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -252624564;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatNotificationSettings;

class draftMessage;

class forumTopicInfo;

class message;

/**
 * Describes a forum topic.
 */
class forumTopic final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Basic information about the topic.
  object_ptr<forumTopicInfo> info_;
  /// Last message in the topic; may be null if unknown.
  object_ptr<message> last_message_;
  /// True, if the topic is pinned in the topic list.
  bool is_pinned_;
  /// Number of unread messages in the topic.
  int32 unread_count_;
  /// Identifier of the last read incoming message.
  int53 last_read_inbox_message_id_;
  /// Identifier of the last read outgoing message.
  int53 last_read_outbox_message_id_;
  /// Number of unread messages with a mention/reply in the topic.
  int32 unread_mention_count_;
  /// Number of messages with unread reactions in the topic.
  int32 unread_reaction_count_;
  /// Notification settings for the topic.
  object_ptr<chatNotificationSettings> notification_settings_;
  /// A draft of a message in the topic; may be null if none.
  object_ptr<draftMessage> draft_message_;

  /**
   * Describes a forum topic.
   */
  forumTopic();

  /**
   * Describes a forum topic.
   *
   * \param[in] info_ Basic information about the topic.
   * \param[in] last_message_ Last message in the topic; may be null if unknown.
   * \param[in] is_pinned_ True, if the topic is pinned in the topic list.
   * \param[in] unread_count_ Number of unread messages in the topic.
   * \param[in] last_read_inbox_message_id_ Identifier of the last read incoming message.
   * \param[in] last_read_outbox_message_id_ Identifier of the last read outgoing message.
   * \param[in] unread_mention_count_ Number of unread messages with a mention/reply in the topic.
   * \param[in] unread_reaction_count_ Number of messages with unread reactions in the topic.
   * \param[in] notification_settings_ Notification settings for the topic.
   * \param[in] draft_message_ A draft of a message in the topic; may be null if none.
   */
  forumTopic(object_ptr<forumTopicInfo> &&info_, object_ptr<message> &&last_message_, bool is_pinned_, int32 unread_count_, int53 last_read_inbox_message_id_, int53 last_read_outbox_message_id_, int32 unread_mention_count_, int32 unread_reaction_count_, object_ptr<chatNotificationSettings> &&notification_settings_, object_ptr<draftMessage> &&draft_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 303279334;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a forum topic icon.
 */
class forumTopicIcon final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Color of the topic icon in RGB format.
  int32 color_;
  /// Unique identifier of the custom emoji shown on the topic icon; 0 if none.
  int64 custom_emoji_id_;

  /**
   * Describes a forum topic icon.
   */
  forumTopicIcon();

  /**
   * Describes a forum topic icon.
   *
   * \param[in] color_ Color of the topic icon in RGB format.
   * \param[in] custom_emoji_id_ Unique identifier of the custom emoji shown on the topic icon; 0 if none.
   */
  forumTopicIcon(int32 color_, int64 custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -818765421;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class MessageSender;

class forumTopicIcon;

/**
 * Contains basic information about a forum topic.
 */
class forumTopicInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Message thread identifier of the topic.
  int53 message_thread_id_;
  /// Name of the topic.
  string name_;
  /// Icon of the topic.
  object_ptr<forumTopicIcon> icon_;
  /// Point in time (Unix timestamp) when the topic was created.
  int32 creation_date_;
  /// Identifier of the creator of the topic.
  object_ptr<MessageSender> creator_id_;
  /// True, if the topic is the General topic list.
  bool is_general_;
  /// True, if the topic was created by the current user.
  bool is_outgoing_;
  /// True, if the topic is closed.
  bool is_closed_;
  /// True, if the topic is hidden above the topic list and closed; for General topic only.
  bool is_hidden_;

  /**
   * Contains basic information about a forum topic.
   */
  forumTopicInfo();

  /**
   * Contains basic information about a forum topic.
   *
   * \param[in] message_thread_id_ Message thread identifier of the topic.
   * \param[in] name_ Name of the topic.
   * \param[in] icon_ Icon of the topic.
   * \param[in] creation_date_ Point in time (Unix timestamp) when the topic was created.
   * \param[in] creator_id_ Identifier of the creator of the topic.
   * \param[in] is_general_ True, if the topic is the General topic list.
   * \param[in] is_outgoing_ True, if the topic was created by the current user.
   * \param[in] is_closed_ True, if the topic is closed.
   * \param[in] is_hidden_ True, if the topic is hidden above the topic list and closed; for General topic only.
   */
  forumTopicInfo(int53 message_thread_id_, string const &name_, object_ptr<forumTopicIcon> &&icon_, int32 creation_date_, object_ptr<MessageSender> &&creator_id_, bool is_general_, bool is_outgoing_, bool is_closed_, bool is_hidden_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1879842914;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class forumTopic;

/**
 * Describes a list of forum topics.
 */
class forumTopics final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of forum topics found.
  int32 total_count_;
  /// List of forum topics.
  array<object_ptr<forumTopic>> topics_;
  /// Offset date for the next getForumTopics request.
  int32 next_offset_date_;
  /// Offset message identifier for the next getForumTopics request.
  int53 next_offset_message_id_;
  /// Offset message thread identifier for the next getForumTopics request.
  int53 next_offset_message_thread_id_;

  /**
   * Describes a list of forum topics.
   */
  forumTopics();

  /**
   * Describes a list of forum topics.
   *
   * \param[in] total_count_ Approximate total number of forum topics found.
   * \param[in] topics_ List of forum topics.
   * \param[in] next_offset_date_ Offset date for the next getForumTopics request.
   * \param[in] next_offset_message_id_ Offset message identifier for the next getForumTopics request.
   * \param[in] next_offset_message_thread_id_ Offset message thread identifier for the next getForumTopics request.
   */
  forumTopics(int32 total_count_, array<object_ptr<forumTopic>> &&topics_, int32 next_offset_date_, int53 next_offset_message_id_, int53 next_offset_message_thread_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 732819537;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class MessageSender;

/**
 * Contains information about the last message from which a new message was forwarded last time.
 */
class forwardSource final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the chat to which the message that was forwarded belonged; may be 0 if unknown.
  int53 chat_id_;
  /// Identifier of the message; may be 0 if unknown.
  int53 message_id_;
  /// Identifier of the sender of the message; may be null if unknown or the new message was forwarded not to Saved Messages.
  object_ptr<MessageSender> sender_id_;
  /// Name of the sender of the message if the sender is hidden by their privacy settings.
  string sender_name_;
  /// Point in time (Unix timestamp) when the message is sent; 0 if unknown.
  int32 date_;
  /// True, if the message that was forwarded is outgoing; always false if sender is unknown.
  bool is_outgoing_;

  /**
   * Contains information about the last message from which a new message was forwarded last time.
   */
  forwardSource();

  /**
   * Contains information about the last message from which a new message was forwarded last time.
   *
   * \param[in] chat_id_ Identifier of the chat to which the message that was forwarded belonged; may be 0 if unknown.
   * \param[in] message_id_ Identifier of the message; may be 0 if unknown.
   * \param[in] sender_id_ Identifier of the sender of the message; may be null if unknown or the new message was forwarded not to Saved Messages.
   * \param[in] sender_name_ Name of the sender of the message if the sender is hidden by their privacy settings.
   * \param[in] date_ Point in time (Unix timestamp) when the message is sent; 0 if unknown.
   * \param[in] is_outgoing_ True, if the message that was forwarded is outgoing; always false if sender is unknown.
   */
  forwardSource(int53 chat_id_, int53 message_id_, object_ptr<MessageSender> &&sender_id_, string const &sender_name_, int32 date_, bool is_outgoing_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1795337929;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class affiliateProgramInfo;

/**
 * Describes a found affiliate program.
 */
class foundAffiliateProgram final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the bot created the program.
  int53 bot_user_id_;
  /// Information about the affiliate program.
  object_ptr<affiliateProgramInfo> parameters_;

  /**
   * Describes a found affiliate program.
   */
  foundAffiliateProgram();

  /**
   * Describes a found affiliate program.
   *
   * \param[in] bot_user_id_ User identifier of the bot created the program.
   * \param[in] parameters_ Information about the affiliate program.
   */
  foundAffiliateProgram(int53 bot_user_id_, object_ptr<affiliateProgramInfo> &&parameters_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2134937582;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class foundAffiliateProgram;

/**
 * Represents a list of found affiliate programs.
 */
class foundAffiliatePrograms final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The total number of found affiliate programs.
  int32 total_count_;
  /// The list of affiliate programs.
  array<object_ptr<foundAffiliateProgram>> programs_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Represents a list of found affiliate programs.
   */
  foundAffiliatePrograms();

  /**
   * Represents a list of found affiliate programs.
   *
   * \param[in] total_count_ The total number of found affiliate programs.
   * \param[in] programs_ The list of affiliate programs.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundAffiliatePrograms(int32 total_count_, array<object_ptr<foundAffiliateProgram>> &&programs_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 186317057;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class chatBoost;

/**
 * Contains a list of boosts applied to a chat.
 */
class foundChatBoosts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Total number of boosts applied to the chat.
  int32 total_count_;
  /// List of boosts.
  array<object_ptr<chatBoost>> boosts_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Contains a list of boosts applied to a chat.
   */
  foundChatBoosts();

  /**
   * Contains a list of boosts applied to a chat.
   *
   * \param[in] total_count_ Total number of boosts applied to the chat.
   * \param[in] boosts_ List of boosts.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundChatBoosts(int32 total_count_, array<object_ptr<chatBoost>> &&boosts_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 51457680;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class message;

/**
 * Contains a list of messages found by a search in a given chat.
 */
class foundChatMessages final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of messages found; -1 if unknown.
  int32 total_count_;
  /// List of messages.
  array<object_ptr<message>> messages_;
  /// The offset for the next request. If 0, there are no more results.
  int53 next_from_message_id_;

  /**
   * Contains a list of messages found by a search in a given chat.
   */
  foundChatMessages();

  /**
   * Contains a list of messages found by a search in a given chat.
   *
   * \param[in] total_count_ Approximate total number of messages found; -1 if unknown.
   * \param[in] messages_ List of messages.
   * \param[in] next_from_message_id_ The offset for the next request. If 0, there are no more results.
   */
  foundChatMessages(int32 total_count_, array<object_ptr<message>> &&messages_, int53 next_from_message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 427484196;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class downloadedFileCounts;

class fileDownload;

/**
 * Contains a list of downloaded files, found by a search.
 */
class foundFileDownloads final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Total number of suitable files, ignoring offset.
  object_ptr<downloadedFileCounts> total_counts_;
  /// The list of files.
  array<object_ptr<fileDownload>> files_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Contains a list of downloaded files, found by a search.
   */
  foundFileDownloads();

  /**
   * Contains a list of downloaded files, found by a search.
   *
   * \param[in] total_counts_ Total number of suitable files, ignoring offset.
   * \param[in] files_ The list of files.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundFileDownloads(object_ptr<downloadedFileCounts> &&total_counts_, array<object_ptr<fileDownload>> &&files_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1395890392;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class message;

/**
 * Contains a list of messages found by a search.
 */
class foundMessages final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of messages found; -1 if unknown.
  int32 total_count_;
  /// List of messages.
  array<object_ptr<message>> messages_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Contains a list of messages found by a search.
   */
  foundMessages();

  /**
   * Contains a list of messages found by a search.
   *
   * \param[in] total_count_ Approximate total number of messages found; -1 if unknown.
   * \param[in] messages_ List of messages.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundMessages(int32 total_count_, array<object_ptr<message>> &&messages_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -529809608;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains 0-based match position.
 */
class foundPosition final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The position of the match.
  int32 position_;

  /**
   * Contains 0-based match position.
   */
  foundPosition();

  /**
   * Contains 0-based match position.
   *
   * \param[in] position_ The position of the match.
   */
  explicit foundPosition(int32 position_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1886724216;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains 0-based positions of matched objects.
 */
class foundPositions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Total number of matched objects.
  int32 total_count_;
  /// The positions of the matched objects.
  array<int32> positions_;

  /**
   * Contains 0-based positions of matched objects.
   */
  foundPositions();

  /**
   * Contains 0-based positions of matched objects.
   *
   * \param[in] total_count_ Total number of matched objects.
   * \param[in] positions_ The positions of the matched objects.
   */
  foundPositions(int32 total_count_, array<int32> &&positions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -80518368;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class story;

/**
 * Contains a list of stories found by a search.
 */
class foundStories final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Approximate total number of stories found.
  int32 total_count_;
  /// List of stories.
  array<object_ptr<story>> stories_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Contains a list of stories found by a search.
   */
  foundStories();

  /**
   * Contains a list of stories found by a search.
   *
   * \param[in] total_count_ Approximate total number of stories found.
   * \param[in] stories_ List of stories.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundStories(int32 total_count_, array<object_ptr<story>> &&stories_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1678513512;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a list of found users.
 */
class foundUsers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifiers of the found users.
  array<int53> user_ids_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Represents a list of found users.
   */
  foundUsers();

  /**
   * Represents a list of found users.
   *
   * \param[in] user_ids_ Identifiers of the found users.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundUsers(array<int53> &&user_ids_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1150570075;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class webApp;

/**
 * Contains information about a Web App found by its short name.
 */
class foundWebApp final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The Web App.
  object_ptr<webApp> web_app_;
  /// True, if the user must be asked for the permission to the bot to send them messages.
  bool request_write_access_;
  /// True, if there is no need to show an ordinary open URL confirmation before opening the Web App. The field must be ignored and confirmation must be shown anyway if the Web App link was hidden.
  bool skip_confirmation_;

  /**
   * Contains information about a Web App found by its short name.
   */
  foundWebApp();

  /**
   * Contains information about a Web App found by its short name.
   *
   * \param[in] web_app_ The Web App.
   * \param[in] request_write_access_ True, if the user must be asked for the permission to the bot to send them messages.
   * \param[in] skip_confirmation_ True, if there is no need to show an ordinary open URL confirmation before opening the Web App. The field must be ignored and confirmation must be shown anyway if the Web App link was hidden.
   */
  foundWebApp(object_ptr<webApp> &&web_app_, bool request_write_access_, bool skip_confirmation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -290926562;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class animation;

class formattedText;

class photo;

/**
 * Describes a game. Use getInternalLink with internalLinkTypeGame to share the game.
 */
class game final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique game identifier.
  int64 id_;
  /// Game short name.
  string short_name_;
  /// Game title.
  string title_;
  /// Game text, usually containing scoreboards for a game.
  object_ptr<formattedText> text_;
  /// Game description.
  string description_;
  /// Game photo.
  object_ptr<photo> photo_;
  /// Game animation; may be null.
  object_ptr<animation> animation_;

  /**
   * Describes a game. Use getInternalLink with internalLinkTypeGame to share the game.
   */
  game();

  /**
   * Describes a game. Use getInternalLink with internalLinkTypeGame to share the game.
   *
   * \param[in] id_ Unique game identifier.
   * \param[in] short_name_ Game short name.
   * \param[in] title_ Game title.
   * \param[in] text_ Game text, usually containing scoreboards for a game.
   * \param[in] description_ Game description.
   * \param[in] photo_ Game photo.
   * \param[in] animation_ Game animation; may be null.
   */
  game(int64 id_, string const &short_name_, string const &title_, object_ptr<formattedText> &&text_, string const &description_, object_ptr<photo> &&photo_, object_ptr<animation> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1565597752;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains one row of the game high score table.
 */
class gameHighScore final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Position in the high score table.
  int32 position_;
  /// User identifier.
  int53 user_id_;
  /// User score.
  int32 score_;

  /**
   * Contains one row of the game high score table.
   */
  gameHighScore();

  /**
   * Contains one row of the game high score table.
   *
   * \param[in] position_ Position in the high score table.
   * \param[in] user_id_ User identifier.
   * \param[in] score_ User score.
   */
  gameHighScore(int32 position_, int53 user_id_, int32 score_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 342871838;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class gameHighScore;

/**
 * Contains a list of game high scores.
 */
class gameHighScores final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A list of game high scores.
  array<object_ptr<gameHighScore>> scores_;

  /**
   * Contains a list of game high scores.
   */
  gameHighScores();

  /**
   * Contains a list of game high scores.
   *
   * \param[in] scores_ A list of game high scores.
   */
  explicit gameHighScores(array<object_ptr<gameHighScore>> &&scores_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -725770727;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class sticker;

/**
 * Describes a gift that can be sent to another user.
 */
class gift final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the gift.
  int64 id_;
  /// The sticker representing the gift.
  object_ptr<sticker> sticker_;
  /// Number of Telegram Stars that must be paid for the gift.
  int53 star_count_;
  /// Number of Telegram Stars that can be claimed by the receiver instead of the gift by default. If the gift was paid with just bought Telegram Stars, then full value can be claimed.
  int53 default_sell_star_count_;
  /// True, if the gift is a birthday gift.
  bool is_for_birthday_;
  /// Number of remaining times the gift can be purchased by all users; 0 if not limited or the gift was sold out.
  int32 remaining_count_;
  /// Number of total times the gift can be purchased by all users; 0 if not limited.
  int32 total_count_;
  /// Point in time (Unix timestamp) when the gift was send for the first time; for sold out gifts only.
  int32 first_send_date_;
  /// Point in time (Unix timestamp) when the gift was send for the last time; for sold out gifts only.
  int32 last_send_date_;

  /**
   * Describes a gift that can be sent to another user.
   */
  gift();

  /**
   * Describes a gift that can be sent to another user.
   *
   * \param[in] id_ Unique identifier of the gift.
   * \param[in] sticker_ The sticker representing the gift.
   * \param[in] star_count_ Number of Telegram Stars that must be paid for the gift.
   * \param[in] default_sell_star_count_ Number of Telegram Stars that can be claimed by the receiver instead of the gift by default. If the gift was paid with just bought Telegram Stars, then full value can be claimed.
   * \param[in] is_for_birthday_ True, if the gift is a birthday gift.
   * \param[in] remaining_count_ Number of remaining times the gift can be purchased by all users; 0 if not limited or the gift was sold out.
   * \param[in] total_count_ Number of total times the gift can be purchased by all users; 0 if not limited.
   * \param[in] first_send_date_ Point in time (Unix timestamp) when the gift was send for the first time; for sold out gifts only.
   * \param[in] last_send_date_ Point in time (Unix timestamp) when the gift was send for the last time; for sold out gifts only.
   */
  gift(int64 id_, object_ptr<sticker> &&sticker_, int53 star_count_, int53 default_sell_star_count_, bool is_for_birthday_, int32 remaining_count_, int32 total_count_, int32 first_send_date_, int32 last_send_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -752333618;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class gift;

/**
 * Contains a list of gifts that can be sent to another user.
 */
class gifts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The list of gifts.
  array<object_ptr<gift>> gifts_;

  /**
   * Contains a list of gifts that can be sent to another user.
   */
  gifts();

  /**
   * Contains a list of gifts that can be sent to another user.
   *
   * \param[in] gifts_ The list of gifts.
   */
  explicit gifts(array<object_ptr<gift>> &&gifts_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1652323382;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class GiveawayParticipantStatus;

/**
 * This class is an abstract base class.
 * Contains information about a giveaway.
 */
class GiveawayInfo: public Object {
 public:
};

/**
 * Describes an ongoing giveaway.
 */
class giveawayInfoOngoing final : public GiveawayInfo {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the giveaway was created.
  int32 creation_date_;
  /// Status of the current user in the giveaway.
  object_ptr<GiveawayParticipantStatus> status_;
  /// True, if the giveaway has ended and results are being prepared.
  bool is_ended_;

  /**
   * Describes an ongoing giveaway.
   */
  giveawayInfoOngoing();

  /**
   * Describes an ongoing giveaway.
   *
   * \param[in] creation_date_ Point in time (Unix timestamp) when the giveaway was created.
   * \param[in] status_ Status of the current user in the giveaway.
   * \param[in] is_ended_ True, if the giveaway has ended and results are being prepared.
   */
  giveawayInfoOngoing(int32 creation_date_, object_ptr<GiveawayParticipantStatus> &&status_, bool is_ended_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1649336400;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a completed giveaway.
 */
class giveawayInfoCompleted final : public GiveawayInfo {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the giveaway was created.
  int32 creation_date_;
  /// Point in time (Unix timestamp) when the winners were selected. May be bigger than winners selection date specified in parameters of the giveaway.
  int32 actual_winners_selection_date_;
  /// True, if the giveaway was canceled and was fully refunded.
  bool was_refunded_;
  /// True, if the current user is a winner of the giveaway.
  bool is_winner_;
  /// Number of winners in the giveaway.
  int32 winner_count_;
  /// Number of winners, which activated their gift codes; for Telegram Premium giveaways only.
  int32 activation_count_;
  /// Telegram Premium gift code that was received by the current user; empty if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Premium giveaway.
  string gift_code_;
  /// The amount of Telegram Stars won by the current user; 0 if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Star giveaway.
  int53 won_star_count_;

  /**
   * Describes a completed giveaway.
   */
  giveawayInfoCompleted();

  /**
   * Describes a completed giveaway.
   *
   * \param[in] creation_date_ Point in time (Unix timestamp) when the giveaway was created.
   * \param[in] actual_winners_selection_date_ Point in time (Unix timestamp) when the winners were selected. May be bigger than winners selection date specified in parameters of the giveaway.
   * \param[in] was_refunded_ True, if the giveaway was canceled and was fully refunded.
   * \param[in] is_winner_ True, if the current user is a winner of the giveaway.
   * \param[in] winner_count_ Number of winners in the giveaway.
   * \param[in] activation_count_ Number of winners, which activated their gift codes; for Telegram Premium giveaways only.
   * \param[in] gift_code_ Telegram Premium gift code that was received by the current user; empty if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Premium giveaway.
   * \param[in] won_star_count_ The amount of Telegram Stars won by the current user; 0 if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Star giveaway.
   */
  giveawayInfoCompleted(int32 creation_date_, int32 actual_winners_selection_date_, bool was_refunded_, bool is_winner_, int32 winner_count_, int32 activation_count_, string const &gift_code_, int53 won_star_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 848085852;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes parameters of a giveaway.
 */
class giveawayParameters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the supergroup or channel chat, which will be automatically boosted by the winners of the giveaway for duration of the Telegram Premium subscription, or for the specified time. If the chat is a channel, then can_post_messages right is required in the channel, otherwise, the user must be an administrator in the supergroup.
  int53 boosted_chat_id_;
  /// Identifiers of other supergroup or channel chats that must be subscribed by the users to be eligible for the giveaway. There can be up to getOption(&quot;giveaway_additional_chat_count_max&quot;) additional chats.
  array<int53> additional_chat_ids_;
  /// Point in time (Unix timestamp) when the giveaway is expected to be performed; must be 60-getOption(&quot;giveaway_duration_max&quot;) seconds in the future in scheduled giveaways.
  int32 winners_selection_date_;
  /// True, if only new members of the chats will be eligible for the giveaway.
  bool only_new_members_;
  /// True, if the list of winners of the giveaway will be available to everyone.
  bool has_public_winners_;
  /// The list of two-letter ISO 3166-1 alpha-2 codes of countries, users from which will be eligible for the giveaway. If empty, then all users can participate in the giveaway. There can be up to getOption(&quot;giveaway_country_count_max&quot;) chosen countries. Users with phone number that was bought at https://fragment.com can participate in any giveaway and the country code &quot;FT&quot; must not be specified in the list.
  array<string> country_codes_;
  /// Additional description of the giveaway prize; 0-128 characters.
  string prize_description_;

  /**
   * Describes parameters of a giveaway.
   */
  giveawayParameters();

  /**
   * Describes parameters of a giveaway.
   *
   * \param[in] boosted_chat_id_ Identifier of the supergroup or channel chat, which will be automatically boosted by the winners of the giveaway for duration of the Telegram Premium subscription, or for the specified time. If the chat is a channel, then can_post_messages right is required in the channel, otherwise, the user must be an administrator in the supergroup.
   * \param[in] additional_chat_ids_ Identifiers of other supergroup or channel chats that must be subscribed by the users to be eligible for the giveaway. There can be up to getOption(&quot;giveaway_additional_chat_count_max&quot;) additional chats.
   * \param[in] winners_selection_date_ Point in time (Unix timestamp) when the giveaway is expected to be performed; must be 60-getOption(&quot;giveaway_duration_max&quot;) seconds in the future in scheduled giveaways.
   * \param[in] only_new_members_ True, if only new members of the chats will be eligible for the giveaway.
   * \param[in] has_public_winners_ True, if the list of winners of the giveaway will be available to everyone.
   * \param[in] country_codes_ The list of two-letter ISO 3166-1 alpha-2 codes of countries, users from which will be eligible for the giveaway. If empty, then all users can participate in the giveaway. There can be up to getOption(&quot;giveaway_country_count_max&quot;) chosen countries. Users with phone number that was bought at https://fragment.com can participate in any giveaway and the country code &quot;FT&quot; must not be specified in the list.
   * \param[in] prize_description_ Additional description of the giveaway prize; 0-128 characters.
   */
  giveawayParameters(int53 boosted_chat_id_, array<int53> &&additional_chat_ids_, int32 winners_selection_date_, bool only_new_members_, bool has_public_winners_, array<string> &&country_codes_, string const &prize_description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1171549354;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about status of a user in a giveaway.
 */
class GiveawayParticipantStatus: public Object {
 public:
};

/**
 * The user is eligible for the giveaway.
 */
class giveawayParticipantStatusEligible final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user is eligible for the giveaway.
   */
  giveawayParticipantStatusEligible();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 304799383;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user participates in the giveaway.
 */
class giveawayParticipantStatusParticipating final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The user participates in the giveaway.
   */
  giveawayParticipantStatusParticipating();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 492036975;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user can't participate in the giveaway, because they have already been member of the chat.
 */
class giveawayParticipantStatusAlreadyWasMember final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Point in time (Unix timestamp) when the user joined the chat.
  int32 joined_chat_date_;

  /**
   * The user can't participate in the giveaway, because they have already been member of the chat.
   */
  giveawayParticipantStatusAlreadyWasMember();

  /**
   * The user can't participate in the giveaway, because they have already been member of the chat.
   *
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined the chat.
   */
  explicit giveawayParticipantStatusAlreadyWasMember(int32 joined_chat_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 301577632;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user can't participate in the giveaway, because they are an administrator in one of the chats that created the giveaway.
 */
class giveawayParticipantStatusAdministrator final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the chat administered by the user.
  int53 chat_id_;

  /**
   * The user can't participate in the giveaway, because they are an administrator in one of the chats that created the giveaway.
   */
  giveawayParticipantStatusAdministrator();

  /**
   * The user can't participate in the giveaway, because they are an administrator in one of the chats that created the giveaway.
   *
   * \param[in] chat_id_ Identifier of the chat administered by the user.
   */
  explicit giveawayParticipantStatusAdministrator(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -934593931;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The user can't participate in the giveaway, because they phone number is from a disallowed country.
 */
class giveawayParticipantStatusDisallowedCountry final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A two-letter ISO 3166-1 alpha-2 country code of the user's country.
  string user_country_code_;

  /**
   * The user can't participate in the giveaway, because they phone number is from a disallowed country.
   */
  giveawayParticipantStatusDisallowedCountry();

  /**
   * The user can't participate in the giveaway, because they phone number is from a disallowed country.
   *
   * \param[in] user_country_code_ A two-letter ISO 3166-1 alpha-2 country code of the user's country.
   */
  explicit giveawayParticipantStatusDisallowedCountry(string const &user_country_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1879794779;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about a giveaway prize.
 */
class GiveawayPrize: public Object {
 public:
};

/**
 * The giveaway sends Telegram Premium subscriptions to the winners.
 */
class giveawayPrizePremium final : public GiveawayPrize {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Number of months the Telegram Premium subscription will be active after code activation.
  int32 month_count_;

  /**
   * The giveaway sends Telegram Premium subscriptions to the winners.
   */
  giveawayPrizePremium();

  /**
   * The giveaway sends Telegram Premium subscriptions to the winners.
   *
   * \param[in] month_count_ Number of months the Telegram Premium subscription will be active after code activation.
   */
  explicit giveawayPrizePremium(int32 month_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 454224248;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The giveaway sends Telegram Stars to the winners.
 */
class giveawayPrizeStars final : public GiveawayPrize {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Number of Telegram Stars that will be shared by all winners.
  int53 star_count_;

  /**
   * The giveaway sends Telegram Stars to the winners.
   */
  giveawayPrizeStars();

  /**
   * The giveaway sends Telegram Stars to the winners.
   *
   * \param[in] star_count_ Number of Telegram Stars that will be shared by all winners.
   */
  explicit giveawayPrizeStars(int53 star_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1790173276;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class groupCallRecentSpeaker;

/**
 * Describes a group call.
 */
class groupCall final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Group call identifier.
  int32 id_;
  /// Group call title.
  string title_;
  /// Point in time (Unix timestamp) when the group call is expected to be started by an administrator; 0 if it is already active or was ended.
  int32 scheduled_start_date_;
  /// True, if the group call is scheduled and the current user will receive a notification when the group call starts.
  bool enabled_start_notification_;
  /// True, if the call is active.
  bool is_active_;
  /// True, if the chat is an RTMP stream instead of an ordinary video chat.
  bool is_rtmp_stream_;
  /// True, if the call is joined.
  bool is_joined_;
  /// True, if user was kicked from the call because of network loss and the call needs to be rejoined.
  bool need_rejoin_;
  /// True, if the current user can manage the group call.
  bool can_be_managed_;
  /// Number of participants in the group call.
  int32 participant_count_;
  /// True, if group call participants, which are muted, aren't returned in participant list.
  bool has_hidden_listeners_;
  /// True, if all group call participants are loaded.
  bool loaded_all_participants_;
  /// At most 3 recently speaking users in the group call.
  array<object_ptr<groupCallRecentSpeaker>> recent_speakers_;
  /// True, if the current user's video is enabled.
  bool is_my_video_enabled_;
  /// True, if the current user's video is paused.
  bool is_my_video_paused_;
  /// True, if the current user can broadcast video or share screen.
  bool can_enable_video_;
  /// True, if only group call administrators can unmute new participants.
  bool mute_new_participants_;
  /// True, if the current user can enable or disable mute_new_participants setting.
  bool can_toggle_mute_new_participants_;
  /// Duration of the ongoing group call recording, in seconds; 0 if none. An updateGroupCall update is not triggered when value of this field changes, but the same recording goes on.
  int32 record_duration_;
  /// True, if a video file is being recorded for the call.
  bool is_video_recorded_;
  /// Call duration, in seconds; for ended calls only.
  int32 duration_;

  /**
   * Describes a group call.
   */
  groupCall();

  /**
   * Describes a group call.
   *
   * \param[in] id_ Group call identifier.
   * \param[in] title_ Group call title.
   * \param[in] scheduled_start_date_ Point in time (Unix timestamp) when the group call is expected to be started by an administrator; 0 if it is already active or was ended.
   * \param[in] enabled_start_notification_ True, if the group call is scheduled and the current user will receive a notification when the group call starts.
   * \param[in] is_active_ True, if the call is active.
   * \param[in] is_rtmp_stream_ True, if the chat is an RTMP stream instead of an ordinary video chat.
   * \param[in] is_joined_ True, if the call is joined.
   * \param[in] need_rejoin_ True, if user was kicked from the call because of network loss and the call needs to be rejoined.
   * \param[in] can_be_managed_ True, if the current user can manage the group call.
   * \param[in] participant_count_ Number of participants in the group call.
   * \param[in] has_hidden_listeners_ True, if group call participants, which are muted, aren't returned in participant list.
   * \param[in] loaded_all_participants_ True, if all group call participants are loaded.
   * \param[in] recent_speakers_ At most 3 recently speaking users in the group call.
   * \param[in] is_my_video_enabled_ True, if the current user's video is enabled.
   * \param[in] is_my_video_paused_ True, if the current user's video is paused.
   * \param[in] can_enable_video_ True, if the current user can broadcast video or share screen.
   * \param[in] mute_new_participants_ True, if only group call administrators can unmute new participants.
   * \param[in] can_toggle_mute_new_participants_ True, if the current user can enable or disable mute_new_participants setting.
   * \param[in] record_duration_ Duration of the ongoing group call recording, in seconds; 0 if none. An updateGroupCall update is not triggered when value of this field changes, but the same recording goes on.
   * \param[in] is_video_recorded_ True, if a video file is being recorded for the call.
   * \param[in] duration_ Call duration, in seconds; for ended calls only.
   */
  groupCall(int32 id_, string const &title_, int32 scheduled_start_date_, bool enabled_start_notification_, bool is_active_, bool is_rtmp_stream_, bool is_joined_, bool need_rejoin_, bool can_be_managed_, int32 participant_count_, bool has_hidden_listeners_, bool loaded_all_participants_, array<object_ptr<groupCallRecentSpeaker>> &&recent_speakers_, bool is_my_video_enabled_, bool is_my_video_paused_, bool can_enable_video_, bool mute_new_participants_, bool can_toggle_mute_new_participants_, int32 record_duration_, bool is_video_recorded_, int32 duration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -123443355;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains the group call identifier.
 */
class groupCallId final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Group call identifier.
  int32 id_;

  /**
   * Contains the group call identifier.
   */
  groupCallId();

  /**
   * Contains the group call identifier.
   *
   * \param[in] id_ Group call identifier.
   */
  explicit groupCallId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 350534469;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class MessageSender;

class groupCallParticipantVideoInfo;

/**
 * Represents a group call participant.
 */
class groupCallParticipant final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the group call participant.
  object_ptr<MessageSender> participant_id_;
  /// User's audio channel synchronization source identifier.
  int32 audio_source_id_;
  /// User's screen sharing audio channel synchronization source identifier.
  int32 screen_sharing_audio_source_id_;
  /// Information about user's video channel; may be null if there is no active video.
  object_ptr<groupCallParticipantVideoInfo> video_info_;
  /// Information about user's screen sharing video channel; may be null if there is no active screen sharing video.
  object_ptr<groupCallParticipantVideoInfo> screen_sharing_video_info_;
  /// The participant user's bio or the participant chat's description.
  string bio_;
  /// True, if the participant is the current user.
  bool is_current_user_;
  /// True, if the participant is speaking as set by setGroupCallParticipantIsSpeaking.
  bool is_speaking_;
  /// True, if the participant hand is raised.
  bool is_hand_raised_;
  /// True, if the current user can mute the participant for all other group call participants.
  bool can_be_muted_for_all_users_;
  /// True, if the current user can allow the participant to unmute themselves or unmute the participant (if the participant is the current user).
  bool can_be_unmuted_for_all_users_;
  /// True, if the current user can mute the participant only for self.
  bool can_be_muted_for_current_user_;
  /// True, if the current user can unmute the participant for self.
  bool can_be_unmuted_for_current_user_;
  /// True, if the participant is muted for all users.
  bool is_muted_for_all_users_;
  /// True, if the participant is muted for the current user.
  bool is_muted_for_current_user_;
  /// True, if the participant is muted for all users, but can unmute themselves.
  bool can_unmute_self_;
  /// Participant's volume level; 1-20000 in hundreds of percents.
  int32 volume_level_;
  /// User's order in the group call participant list. Orders must be compared lexicographically. The bigger is order, the higher is user in the list. If order is empty, the user must be removed from the participant list.
  string order_;

  /**
   * Represents a group call participant.
   */
  groupCallParticipant();

  /**
   * Represents a group call participant.
   *
   * \param[in] participant_id_ Identifier of the group call participant.
   * \param[in] audio_source_id_ User's audio channel synchronization source identifier.
   * \param[in] screen_sharing_audio_source_id_ User's screen sharing audio channel synchronization source identifier.
   * \param[in] video_info_ Information about user's video channel; may be null if there is no active video.
   * \param[in] screen_sharing_video_info_ Information about user's screen sharing video channel; may be null if there is no active screen sharing video.
   * \param[in] bio_ The participant user's bio or the participant chat's description.
   * \param[in] is_current_user_ True, if the participant is the current user.
   * \param[in] is_speaking_ True, if the participant is speaking as set by setGroupCallParticipantIsSpeaking.
   * \param[in] is_hand_raised_ True, if the participant hand is raised.
   * \param[in] can_be_muted_for_all_users_ True, if the current user can mute the participant for all other group call participants.
   * \param[in] can_be_unmuted_for_all_users_ True, if the current user can allow the participant to unmute themselves or unmute the participant (if the participant is the current user).
   * \param[in] can_be_muted_for_current_user_ True, if the current user can mute the participant only for self.
   * \param[in] can_be_unmuted_for_current_user_ True, if the current user can unmute the participant for self.
   * \param[in] is_muted_for_all_users_ True, if the participant is muted for all users.
   * \param[in] is_muted_for_current_user_ True, if the participant is muted for the current user.
   * \param[in] can_unmute_self_ True, if the participant is muted for all users, but can unmute themselves.
   * \param[in] volume_level_ Participant's volume level; 1-20000 in hundreds of percents.
   * \param[in] order_ User's order in the group call participant list. Orders must be compared lexicographically. The bigger is order, the higher is user in the list. If order is empty, the user must be removed from the participant list.
   */
  groupCallParticipant(object_ptr<MessageSender> &&participant_id_, int32 audio_source_id_, int32 screen_sharing_audio_source_id_, object_ptr<groupCallParticipantVideoInfo> &&video_info_, object_ptr<groupCallParticipantVideoInfo> &&screen_sharing_video_info_, string const &bio_, bool is_current_user_, bool is_speaking_, bool is_hand_raised_, bool can_be_muted_for_all_users_, bool can_be_unmuted_for_all_users_, bool can_be_muted_for_current_user_, bool can_be_unmuted_for_current_user_, bool is_muted_for_all_users_, bool is_muted_for_current_user_, bool can_unmute_self_, int32 volume_level_, string const &order_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2059182571;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class groupCallVideoSourceGroup;

/**
 * Contains information about a group call participant's video channel.
 */
class groupCallParticipantVideoInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of synchronization source groups of the video.
  array<object_ptr<groupCallVideoSourceGroup>> source_groups_;
  /// Video channel endpoint identifier.
  string endpoint_id_;
  /// True, if the video is paused. This flag needs to be ignored, if new video frames are received.
  bool is_paused_;

  /**
   * Contains information about a group call participant's video channel.
   */
  groupCallParticipantVideoInfo();

  /**
   * Contains information about a group call participant's video channel.
   *
   * \param[in] source_groups_ List of synchronization source groups of the video.
   * \param[in] endpoint_id_ Video channel endpoint identifier.
   * \param[in] is_paused_ True, if the video is paused. This flag needs to be ignored, if new video frames are received.
   */
  groupCallParticipantVideoInfo(array<object_ptr<groupCallVideoSourceGroup>> &&source_groups_, string const &endpoint_id_, bool is_paused_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -14294645;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class MessageSender;

/**
 * Describes a recently speaking participant in a group call.
 */
class groupCallRecentSpeaker final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Group call participant identifier.
  object_ptr<MessageSender> participant_id_;
  /// True, is the user has spoken recently.
  bool is_speaking_;

  /**
   * Describes a recently speaking participant in a group call.
   */
  groupCallRecentSpeaker();

  /**
   * Describes a recently speaking participant in a group call.
   *
   * \param[in] participant_id_ Group call participant identifier.
   * \param[in] is_speaking_ True, is the user has spoken recently.
   */
  groupCallRecentSpeaker(object_ptr<MessageSender> &&participant_id_, bool is_speaking_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1819519436;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes an available stream in a group call.
 */
class groupCallStream final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of an audio/video channel.
  int32 channel_id_;
  /// Scale of segment durations in the stream. The duration is 1000/(2**scale) milliseconds.
  int32 scale_;
  /// Point in time when the stream currently ends; Unix timestamp in milliseconds.
  int53 time_offset_;

  /**
   * Describes an available stream in a group call.
   */
  groupCallStream();

  /**
   * Describes an available stream in a group call.
   *
   * \param[in] channel_id_ Identifier of an audio/video channel.
   * \param[in] scale_ Scale of segment durations in the stream. The duration is 1000/(2**scale) milliseconds.
   * \param[in] time_offset_ Point in time when the stream currently ends; Unix timestamp in milliseconds.
   */
  groupCallStream(int32 channel_id_, int32 scale_, int53 time_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -264564795;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class groupCallStream;

/**
 * Represents a list of group call streams.
 */
class groupCallStreams final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A list of group call streams.
  array<object_ptr<groupCallStream>> streams_;

  /**
   * Represents a list of group call streams.
   */
  groupCallStreams();

  /**
   * Represents a list of group call streams.
   *
   * \param[in] streams_ A list of group call streams.
   */
  explicit groupCallStreams(array<object_ptr<groupCallStream>> &&streams_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032959578;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes the quality of a group call video.
 */
class GroupCallVideoQuality: public Object {
 public:
};

/**
 * The worst available video quality.
 */
class groupCallVideoQualityThumbnail final : public GroupCallVideoQuality {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The worst available video quality.
   */
  groupCallVideoQualityThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -379186304;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The medium video quality.
 */
class groupCallVideoQualityMedium final : public GroupCallVideoQuality {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The medium video quality.
   */
  groupCallVideoQualityMedium();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 394968234;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The best available video quality.
 */
class groupCallVideoQualityFull final : public GroupCallVideoQuality {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The best available video quality.
   */
  groupCallVideoQualityFull();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2125916617;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a group of video synchronization source identifiers.
 */
class groupCallVideoSourceGroup final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The semantics of sources, one of &quot;SIM&quot; or &quot;FID&quot;.
  string semantics_;
  /// The list of synchronization source identifiers.
  array<int32> source_ids_;

  /**
   * Describes a group of video synchronization source identifiers.
   */
  groupCallVideoSourceGroup();

  /**
   * Describes a group of video synchronization source identifiers.
   *
   * \param[in] semantics_ The semantics of sources, one of &quot;SIM&quot; or &quot;FID&quot;.
   * \param[in] source_ids_ The list of synchronization source identifiers.
   */
  groupCallVideoSourceGroup(string const &semantics_, array<int32> &&source_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1190900785;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains a list of hashtags.
 */
class hashtags final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// A list of hashtags.
  array<string> hashtags_;

  /**
   * Contains a list of hashtags.
   */
  hashtags();

  /**
   * Contains a list of hashtags.
   *
   * \param[in] hashtags_ A list of hashtags.
   */
  explicit hashtags(array<string> &&hashtags_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 676798885;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Contains an HTTP URL.
 */
class httpUrl final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The URL.
  string url_;

  /**
   * Contains an HTTP URL.
   */
  httpUrl();

  /**
   * Contains an HTTP URL.
   *
   * \param[in] url_ The URL.
   */
  explicit httpUrl(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2018019930;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class date;

class datedFile;

/**
 * An identity document.
 */
class identityDocument final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Document number; 1-24 characters.
  string number_;
  /// Document expiration date; may be null if not applicable.
  object_ptr<date> expiration_date_;
  /// Front side of the document.
  object_ptr<datedFile> front_side_;
  /// Reverse side of the document; only for driver license and identity card; may be null.
  object_ptr<datedFile> reverse_side_;
  /// Selfie with the document; may be null.
  object_ptr<datedFile> selfie_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<datedFile>> translation_;

  /**
   * An identity document.
   */
  identityDocument();

  /**
   * An identity document.
   *
   * \param[in] number_ Document number; 1-24 characters.
   * \param[in] expiration_date_ Document expiration date; may be null if not applicable.
   * \param[in] front_side_ Front side of the document.
   * \param[in] reverse_side_ Reverse side of the document; only for driver license and identity card; may be null.
   * \param[in] selfie_ Selfie with the document; may be null.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  identityDocument(string const &number_, object_ptr<date> &&expiration_date_, object_ptr<datedFile> &&front_side_, object_ptr<datedFile> &&reverse_side_, object_ptr<datedFile> &&selfie_, array<object_ptr<datedFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1001703606;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents the result of an importContacts request.
 */
class importedContacts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifiers of the imported contacts in the same order as they were specified in the request; 0 if the contact is not yet a registered user.
  array<int53> user_ids_;
  /// The number of users that imported the corresponding contact; 0 for already registered users or if unavailable.
  array<int32> importer_count_;

  /**
   * Represents the result of an importContacts request.
   */
  importedContacts();

  /**
   * Represents the result of an importContacts request.
   *
   * \param[in] user_ids_ User identifiers of the imported contacts in the same order as they were specified in the request; 0 if the contact is not yet a registered user.
   * \param[in] importer_count_ The number of users that imported the corresponding contact; 0 for already registered users or if unavailable.
   */
  importedContacts(array<int53> &&user_ids_, array<int32> &&importer_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2068432290;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InlineKeyboardButtonType;

/**
 * Represents a single button in an inline keyboard.
 */
class inlineKeyboardButton final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text of the button.
  string text_;
  /// Type of the button.
  object_ptr<InlineKeyboardButtonType> type_;

  /**
   * Represents a single button in an inline keyboard.
   */
  inlineKeyboardButton();

  /**
   * Represents a single button in an inline keyboard.
   *
   * \param[in] text_ Text of the button.
   * \param[in] type_ Type of the button.
   */
  inlineKeyboardButton(string const &text_, object_ptr<InlineKeyboardButtonType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -372105704;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class TargetChat;

/**
 * This class is an abstract base class.
 * Describes the type of inline keyboard button.
 */
class InlineKeyboardButtonType: public Object {
 public:
};

/**
 * A button that opens a specified URL.
 */
class inlineKeyboardButtonTypeUrl final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// HTTP or tg:// URL to open. If the link is of the type internalLinkTypeWebApp, then the button must be marked as a Web App button.
  string url_;

  /**
   * A button that opens a specified URL.
   */
  inlineKeyboardButtonTypeUrl();

  /**
   * A button that opens a specified URL.
   *
   * \param[in] url_ HTTP or tg:// URL to open. If the link is of the type internalLinkTypeWebApp, then the button must be marked as a Web App button.
   */
  explicit inlineKeyboardButtonTypeUrl(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1130741420;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that opens a specified URL and automatically authorize the current user by calling getLoginUrlInfo.
 */
class inlineKeyboardButtonTypeLoginUrl final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// An HTTP URL to pass to getLoginUrlInfo.
  string url_;
  /// Unique button identifier.
  int53 id_;
  /// If non-empty, new text of the button in forwarded messages.
  string forward_text_;

  /**
   * A button that opens a specified URL and automatically authorize the current user by calling getLoginUrlInfo.
   */
  inlineKeyboardButtonTypeLoginUrl();

  /**
   * A button that opens a specified URL and automatically authorize the current user by calling getLoginUrlInfo.
   *
   * \param[in] url_ An HTTP URL to pass to getLoginUrlInfo.
   * \param[in] id_ Unique button identifier.
   * \param[in] forward_text_ If non-empty, new text of the button in forwarded messages.
   */
  inlineKeyboardButtonTypeLoginUrl(string const &url_, int53 id_, string const &forward_text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1203413081;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that opens a Web App by calling openWebApp.
 */
class inlineKeyboardButtonTypeWebApp final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// An HTTP URL to pass to openWebApp.
  string url_;

  /**
   * A button that opens a Web App by calling openWebApp.
   */
  inlineKeyboardButtonTypeWebApp();

  /**
   * A button that opens a Web App by calling openWebApp.
   *
   * \param[in] url_ An HTTP URL to pass to openWebApp.
   */
  explicit inlineKeyboardButtonTypeWebApp(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1767471672;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that sends a callback query to a bot.
 */
class inlineKeyboardButtonTypeCallback final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Data to be sent to the bot via a callback query.
  bytes data_;

  /**
   * A button that sends a callback query to a bot.
   */
  inlineKeyboardButtonTypeCallback();

  /**
   * A button that sends a callback query to a bot.
   *
   * \param[in] data_ Data to be sent to the bot via a callback query.
   */
  explicit inlineKeyboardButtonTypeCallback(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1127515139;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that asks for the 2-step verification password of the current user and then sends a callback query to a bot.
 */
class inlineKeyboardButtonTypeCallbackWithPassword final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Data to be sent to the bot via a callback query.
  bytes data_;

  /**
   * A button that asks for the 2-step verification password of the current user and then sends a callback query to a bot.
   */
  inlineKeyboardButtonTypeCallbackWithPassword();

  /**
   * A button that asks for the 2-step verification password of the current user and then sends a callback query to a bot.
   *
   * \param[in] data_ Data to be sent to the bot via a callback query.
   */
  explicit inlineKeyboardButtonTypeCallbackWithPassword(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 908018248;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button with a game that sends a callback query to a bot. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageGame.
 */
class inlineKeyboardButtonTypeCallbackGame final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A button with a game that sends a callback query to a bot. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageGame.
   */
  inlineKeyboardButtonTypeCallbackGame();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -383429528;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that forces an inline query to the bot to be inserted in the input field.
 */
class inlineKeyboardButtonTypeSwitchInline final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Inline query to be sent to the bot.
  string query_;
  /// Target chat from which to send the inline query.
  object_ptr<TargetChat> target_chat_;

  /**
   * A button that forces an inline query to the bot to be inserted in the input field.
   */
  inlineKeyboardButtonTypeSwitchInline();

  /**
   * A button that forces an inline query to the bot to be inserted in the input field.
   *
   * \param[in] query_ Inline query to be sent to the bot.
   * \param[in] target_chat_ Target chat from which to send the inline query.
   */
  inlineKeyboardButtonTypeSwitchInline(string const &query_, object_ptr<TargetChat> &&target_chat_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 544906485;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button to buy something. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageInvoice.
 */
class inlineKeyboardButtonTypeBuy final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * A button to buy something. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageInvoice.
   */
  inlineKeyboardButtonTypeBuy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1360739440;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
 */
class inlineKeyboardButtonTypeUser final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier.
  int53 user_id_;

  /**
   * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
   */
  inlineKeyboardButtonTypeUser();

  /**
   * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
   *
   * \param[in] user_id_ User identifier.
   */
  explicit inlineKeyboardButtonTypeUser(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1836574114;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A button that copies specified text to clipboard.
 */
class inlineKeyboardButtonTypeCopyText final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The text to copy to clipboard.
  string text_;

  /**
   * A button that copies specified text to clipboard.
   */
  inlineKeyboardButtonTypeCopyText();

  /**
   * A button that copies specified text to clipboard.
   *
   * \param[in] text_ The text to copy to clipboard.
   */
  explicit inlineKeyboardButtonTypeCopyText(string const &text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 68883206;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class animation;

class audio;

class contact;

class document;

class game;

class location;

class photo;

class sticker;

class thumbnail;

class venue;

class video;

class voiceNote;

/**
 * This class is an abstract base class.
 * Represents a single result of an inline query.
 */
class InlineQueryResult: public Object {
 public:
};

/**
 * Represents a link to an article or web page.
 */
class inlineQueryResultArticle final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// URL of the result, if it exists.
  string url_;
  /// True, if the URL must be not shown.
  bool hide_url_;
  /// Title of the result.
  string title_;
  /// A short description of the result.
  string description_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents a link to an article or web page.
   */
  inlineQueryResultArticle();

  /**
   * Represents a link to an article or web page.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] url_ URL of the result, if it exists.
   * \param[in] hide_url_ True, if the URL must be not shown.
   * \param[in] title_ Title of the result.
   * \param[in] description_ A short description of the result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultArticle(string const &id_, string const &url_, bool hide_url_, string const &title_, string const &description_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 206340825;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a user contact.
 */
class inlineQueryResultContact final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// A user contact.
  object_ptr<contact> contact_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents a user contact.
   */
  inlineQueryResultContact();

  /**
   * Represents a user contact.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] contact_ A user contact.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultContact(string const &id_, object_ptr<contact> &&contact_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -181960174;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a point on the map.
 */
class inlineQueryResultLocation final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Location result.
  object_ptr<location> location_;
  /// Title of the result.
  string title_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents a point on the map.
   */
  inlineQueryResultLocation();

  /**
   * Represents a point on the map.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] location_ Location result.
   * \param[in] title_ Title of the result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultLocation(string const &id_, object_ptr<location> &&location_, string const &title_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 466004752;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents information about a venue.
 */
class inlineQueryResultVenue final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Venue result.
  object_ptr<venue> venue_;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;

  /**
   * Represents information about a venue.
   */
  inlineQueryResultVenue();

  /**
   * Represents information about a venue.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] venue_ Venue result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultVenue(string const &id_, object_ptr<venue> &&venue_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1281036382;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents information about a game.
 */
class inlineQueryResultGame final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Game result.
  object_ptr<game> game_;

  /**
   * Represents information about a game.
   */
  inlineQueryResultGame();

  /**
   * Represents information about a game.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] game_ Game result.
   */
  inlineQueryResultGame(string const &id_, object_ptr<game> &&game_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1706916987;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an animation file.
 */
class inlineQueryResultAnimation final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Animation file.
  object_ptr<animation> animation_;
  /// Animation title.
  string title_;

  /**
   * Represents an animation file.
   */
  inlineQueryResultAnimation();

  /**
   * Represents an animation file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] animation_ Animation file.
   * \param[in] title_ Animation title.
   */
  inlineQueryResultAnimation(string const &id_, object_ptr<animation> &&animation_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2009984267;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents an audio file.
 */
class inlineQueryResultAudio final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Audio file.
  object_ptr<audio> audio_;

  /**
   * Represents an audio file.
   */
  inlineQueryResultAudio();

  /**
   * Represents an audio file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] audio_ Audio file.
   */
  inlineQueryResultAudio(string const &id_, object_ptr<audio> &&audio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 842650360;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a document.
 */
class inlineQueryResultDocument final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Document.
  object_ptr<document> document_;
  /// Document title.
  string title_;
  /// Document description.
  string description_;

  /**
   * Represents a document.
   */
  inlineQueryResultDocument();

  /**
   * Represents a document.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] document_ Document.
   * \param[in] title_ Document title.
   * \param[in] description_ Document description.
   */
  inlineQueryResultDocument(string const &id_, object_ptr<document> &&document_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1491268539;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a photo.
 */
class inlineQueryResultPhoto final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Photo.
  object_ptr<photo> photo_;
  /// Title of the result, if known.
  string title_;
  /// A short description of the result, if known.
  string description_;

  /**
   * Represents a photo.
   */
  inlineQueryResultPhoto();

  /**
   * Represents a photo.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] photo_ Photo.
   * \param[in] title_ Title of the result, if known.
   * \param[in] description_ A short description of the result, if known.
   */
  inlineQueryResultPhoto(string const &id_, object_ptr<photo> &&photo_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1848319440;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a sticker.
 */
class inlineQueryResultSticker final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Sticker.
  object_ptr<sticker> sticker_;

  /**
   * Represents a sticker.
   */
  inlineQueryResultSticker();

  /**
   * Represents a sticker.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] sticker_ Sticker.
   */
  inlineQueryResultSticker(string const &id_, object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1848224245;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a video.
 */
class inlineQueryResultVideo final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Video.
  object_ptr<video> video_;
  /// Title of the video.
  string title_;
  /// Description of the video.
  string description_;

  /**
   * Represents a video.
   */
  inlineQueryResultVideo();

  /**
   * Represents a video.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] video_ Video.
   * \param[in] title_ Title of the video.
   * \param[in] description_ Description of the video.
   */
  inlineQueryResultVideo(string const &id_, object_ptr<video> &&video_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1373158683;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a voice note.
 */
class inlineQueryResultVoiceNote final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Voice note.
  object_ptr<voiceNote> voice_note_;
  /// Title of the voice note.
  string title_;

  /**
   * Represents a voice note.
   */
  inlineQueryResultVoiceNote();

  /**
   * Represents a voice note.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] voice_note_ Voice note.
   * \param[in] title_ Title of the voice note.
   */
  inlineQueryResultVoiceNote(string const &id_, object_ptr<voiceNote> &&voice_note_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1897393105;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InlineQueryResult;

class inlineQueryResultsButton;

/**
 * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
 */
class inlineQueryResults final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the inline query.
  int64 inline_query_id_;
  /// Button to be shown above inline query results; may be null.
  object_ptr<inlineQueryResultsButton> button_;
  /// Results of the query.
  array<object_ptr<InlineQueryResult>> results_;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;

  /**
   * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
   */
  inlineQueryResults();

  /**
   * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
   *
   * \param[in] inline_query_id_ Unique identifier of the inline query.
   * \param[in] button_ Button to be shown above inline query results; may be null.
   * \param[in] results_ Results of the query.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  inlineQueryResults(int64 inline_query_id_, object_ptr<inlineQueryResultsButton> &&button_, array<object_ptr<InlineQueryResult>> &&results_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1830685615;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InlineQueryResultsButtonType;

/**
 * Represents a button to be shown above inline query results.
 */
class inlineQueryResultsButton final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The text of the button.
  string text_;
  /// Type of the button.
  object_ptr<InlineQueryResultsButtonType> type_;

  /**
   * Represents a button to be shown above inline query results.
   */
  inlineQueryResultsButton();

  /**
   * Represents a button to be shown above inline query results.
   *
   * \param[in] text_ The text of the button.
   * \param[in] type_ Type of the button.
   */
  inlineQueryResultsButton(string const &text_, object_ptr<InlineQueryResultsButtonType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -790689618;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Represents type of button in results of inline query.
 */
class InlineQueryResultsButtonType: public Object {
 public:
};

/**
 * Describes the button that opens a private chat with the bot and sends a start message to the bot with the given parameter.
 */
class inlineQueryResultsButtonTypeStartBot final : public InlineQueryResultsButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The parameter for the bot start message.
  string parameter_;

  /**
   * Describes the button that opens a private chat with the bot and sends a start message to the bot with the given parameter.
   */
  inlineQueryResultsButtonTypeStartBot();

  /**
   * Describes the button that opens a private chat with the bot and sends a start message to the bot with the given parameter.
   *
   * \param[in] parameter_ The parameter for the bot start message.
   */
  explicit inlineQueryResultsButtonTypeStartBot(string const &parameter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -23400235;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes the button that opens a Web App by calling getWebAppUrl.
 */
class inlineQueryResultsButtonTypeWebApp final : public InlineQueryResultsButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// An HTTP URL to pass to getWebAppUrl.
  string url_;

  /**
   * Describes the button that opens a Web App by calling getWebAppUrl.
   */
  inlineQueryResultsButtonTypeWebApp();

  /**
   * Describes the button that opens a Web App by calling getWebAppUrl.
   *
   * \param[in] url_ An HTTP URL to pass to getWebAppUrl.
   */
  explicit inlineQueryResultsButtonTypeWebApp(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1197382814;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

/**
 * This class is an abstract base class.
 * Contains information about background to set.
 */
class InputBackground: public Object {
 public:
};

/**
 * A background from a local file.
 */
class inputBackgroundLocal final : public InputBackground {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Background file to use. Only inputFileLocal and inputFileGenerated are supported. The file must be in JPEG format for wallpapers and in PNG format for patterns.
  object_ptr<InputFile> background_;

  /**
   * A background from a local file.
   */
  inputBackgroundLocal();

  /**
   * A background from a local file.
   *
   * \param[in] background_ Background file to use. Only inputFileLocal and inputFileGenerated are supported. The file must be in JPEG format for wallpapers and in PNG format for patterns.
   */
  explicit inputBackgroundLocal(object_ptr<InputFile> &&background_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1747094364;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A background from the server.
 */
class inputBackgroundRemote final : public InputBackground {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The background identifier.
  int64 background_id_;

  /**
   * A background from the server.
   */
  inputBackgroundRemote();

  /**
   * A background from the server.
   *
   * \param[in] background_id_ The background identifier.
   */
  explicit inputBackgroundRemote(int64 background_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -274976231;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A background previously set in the chat; for chat backgrounds only.
 */
class inputBackgroundPrevious final : public InputBackground {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the message with the background.
  int53 message_id_;

  /**
   * A background previously set in the chat; for chat backgrounds only.
   */
  inputBackgroundPrevious();

  /**
   * A background previously set in the chat; for chat backgrounds only.
   *
   * \param[in] message_id_ Identifier of the message with the background.
   */
  explicit inputBackgroundPrevious(int53 message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -351905954;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class formattedText;

/**
 * Describes a business chat link to create or edit.
 */
class inputBusinessChatLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Message draft text that will be added to the input field.
  object_ptr<formattedText> text_;
  /// Link title.
  string title_;

  /**
   * Describes a business chat link to create or edit.
   */
  inputBusinessChatLink();

  /**
   * Describes a business chat link to create or edit.
   *
   * \param[in] text_ Message draft text that will be added to the input field.
   * \param[in] title_ Link title.
   */
  inputBusinessChatLink(object_ptr<formattedText> &&text_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 237858296;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

/**
 * Describes settings for a business account start page to set.
 */
class inputBusinessStartPage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Title text of the start page; 0-getOption(&quot;business_start_page_title_length_max&quot;) characters.
  string title_;
  /// Message text of the start page; 0-getOption(&quot;business_start_page_message_length_max&quot;) characters.
  string message_;
  /// Greeting sticker of the start page; pass null if none. The sticker must belong to a sticker set and must not be a custom emoji.
  object_ptr<InputFile> sticker_;

  /**
   * Describes settings for a business account start page to set.
   */
  inputBusinessStartPage();

  /**
   * Describes settings for a business account start page to set.
   *
   * \param[in] title_ Title text of the start page; 0-getOption(&quot;business_start_page_title_length_max&quot;) characters.
   * \param[in] message_ Message text of the start page; 0-getOption(&quot;business_start_page_message_length_max&quot;) characters.
   * \param[in] sticker_ Greeting sticker of the start page; pass null if none. The sticker must belong to a sticker set and must not be a custom emoji.
   */
  inputBusinessStartPage(string const &title_, string const &message_, object_ptr<InputFile> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -327383072;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

class chatPhotoSticker;

/**
 * This class is an abstract base class.
 * Describes a photo to be set as a user profile or chat photo.
 */
class InputChatPhoto: public Object {
 public:
};

/**
 * A previously used profile photo of the current user.
 */
class inputChatPhotoPrevious final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the current user's profile photo to reuse.
  int64 chat_photo_id_;

  /**
   * A previously used profile photo of the current user.
   */
  inputChatPhotoPrevious();

  /**
   * A previously used profile photo of the current user.
   *
   * \param[in] chat_photo_id_ Identifier of the current user's profile photo to reuse.
   */
  explicit inputChatPhotoPrevious(int64 chat_photo_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 23128529;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A static photo in JPEG format.
 */
class inputChatPhotoStatic final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Photo to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
  object_ptr<InputFile> photo_;

  /**
   * A static photo in JPEG format.
   */
  inputChatPhotoStatic();

  /**
   * A static photo in JPEG format.
   *
   * \param[in] photo_ Photo to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
   */
  explicit inputChatPhotoStatic(object_ptr<InputFile> &&photo_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1979179699;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 1280 and be at most 2MB in size.
 */
class inputChatPhotoAnimation final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Animation to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
  object_ptr<InputFile> animation_;
  /// Timestamp of the frame, which will be used as static chat photo.
  double main_frame_timestamp_;

  /**
   * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 1280 and be at most 2MB in size.
   */
  inputChatPhotoAnimation();

  /**
   * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 1280 and be at most 2MB in size.
   *
   * \param[in] animation_ Animation to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
   * \param[in] main_frame_timestamp_ Timestamp of the frame, which will be used as static chat photo.
   */
  inputChatPhotoAnimation(object_ptr<InputFile> &&animation_, double main_frame_timestamp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 90846242;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A sticker on a custom background.
 */
class inputChatPhotoSticker final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Information about the sticker.
  object_ptr<chatPhotoSticker> sticker_;

  /**
   * A sticker on a custom background.
   */
  inputChatPhotoSticker();

  /**
   * A sticker on a custom background.
   *
   * \param[in] sticker_ Information about the sticker.
   */
  explicit inputChatPhotoSticker(object_ptr<chatPhotoSticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1315861341;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains information about the payment method chosen by the user.
 */
class InputCredentials: public Object {
 public:
};

/**
 * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
 */
class inputCredentialsSaved final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the saved credentials.
  string saved_credentials_id_;

  /**
   * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
   */
  inputCredentialsSaved();

  /**
   * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
   *
   * \param[in] saved_credentials_id_ Identifier of the saved credentials.
   */
  explicit inputCredentialsSaved(string const &saved_credentials_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2034385364;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Applies if a user enters new credentials on a payment provider website.
 */
class inputCredentialsNew final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// JSON-encoded data with the credential identifier from the payment provider.
  string data_;
  /// True, if the credential identifier can be saved on the server side.
  bool allow_save_;

  /**
   * Applies if a user enters new credentials on a payment provider website.
   */
  inputCredentialsNew();

  /**
   * Applies if a user enters new credentials on a payment provider website.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier from the payment provider.
   * \param[in] allow_save_ True, if the credential identifier can be saved on the server side.
   */
  inputCredentialsNew(string const &data_, bool allow_save_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -829689558;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Applies if a user enters new credentials using Apple Pay.
 */
class inputCredentialsApplePay final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// JSON-encoded data with the credential identifier.
  string data_;

  /**
   * Applies if a user enters new credentials using Apple Pay.
   */
  inputCredentialsApplePay();

  /**
   * Applies if a user enters new credentials using Apple Pay.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier.
   */
  explicit inputCredentialsApplePay(string const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1246570799;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Applies if a user enters new credentials using Google Pay.
 */
class inputCredentialsGooglePay final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// JSON-encoded data with the credential identifier.
  string data_;

  /**
   * Applies if a user enters new credentials using Google Pay.
   */
  inputCredentialsGooglePay();

  /**
   * Applies if a user enters new credentials using Google Pay.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier.
   */
  explicit inputCredentialsGooglePay(string const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 844384100;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Points to a file.
 */
class InputFile: public Object {
 public:
};

/**
 * A file defined by its unique identifier.
 */
class inputFileId final : public InputFile {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique file identifier.
  int32 id_;

  /**
   * A file defined by its unique identifier.
   */
  inputFileId();

  /**
   * A file defined by its unique identifier.
   *
   * \param[in] id_ Unique file identifier.
   */
  explicit inputFileId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1788906253;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A file defined by its remote identifier. The remote identifier is guaranteed to be usable only if the corresponding file is still accessible to the user and known to TDLib. For example, if the file is from a message, then the message must be not deleted and accessible to the user. If the file database is disabled, then the corresponding object with the file must be preloaded by the application.
 */
class inputFileRemote final : public InputFile {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Remote file identifier.
  string id_;

  /**
   * A file defined by its remote identifier. The remote identifier is guaranteed to be usable only if the corresponding file is still accessible to the user and known to TDLib. For example, if the file is from a message, then the message must be not deleted and accessible to the user. If the file database is disabled, then the corresponding object with the file must be preloaded by the application.
   */
  inputFileRemote();

  /**
   * A file defined by its remote identifier. The remote identifier is guaranteed to be usable only if the corresponding file is still accessible to the user and known to TDLib. For example, if the file is from a message, then the message must be not deleted and accessible to the user. If the file database is disabled, then the corresponding object with the file must be preloaded by the application.
   *
   * \param[in] id_ Remote file identifier.
   */
  explicit inputFileRemote(string const &id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -107574466;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A file defined by a local path.
 */
class inputFileLocal final : public InputFile {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Local path to the file.
  string path_;

  /**
   * A file defined by a local path.
   */
  inputFileLocal();

  /**
   * A file defined by a local path.
   *
   * \param[in] path_ Local path to the file.
   */
  explicit inputFileLocal(string const &path_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2056030919;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A file generated by the application. The application must handle updates updateFileGenerationStart and updateFileGenerationStop to generate the file when asked by TDLib.
 */
class inputFileGenerated final : public InputFile {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Local path to a file from which the file is generated. The path doesn't have to be a valid path and is used by TDLib only to detect name and MIME type of the generated file.
  string original_path_;
  /// String specifying the conversion applied to the original file; must be persistent across application restarts. Conversions beginning with '\#' are reserved for internal TDLib usage.
  string conversion_;
  /// Expected size of the generated file, in bytes; pass 0 if unknown.
  int53 expected_size_;

  /**
   * A file generated by the application. The application must handle updates updateFileGenerationStart and updateFileGenerationStop to generate the file when asked by TDLib.
   */
  inputFileGenerated();

  /**
   * A file generated by the application. The application must handle updates updateFileGenerationStart and updateFileGenerationStop to generate the file when asked by TDLib.
   *
   * \param[in] original_path_ Local path to a file from which the file is generated. The path doesn't have to be a valid path and is used by TDLib only to detect name and MIME type of the generated file.
   * \param[in] conversion_ String specifying the conversion applied to the original file; must be persistent across application restarts. Conversions beginning with '\#' are reserved for internal TDLib usage.
   * \param[in] expected_size_ Expected size of the generated file, in bytes; pass 0 if unknown.
   */
  inputFileGenerated(string const &original_path_, string const &conversion_, int53 expected_size_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1333385216;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

class date;

/**
 * An identity document to be saved to Telegram Passport.
 */
class inputIdentityDocument final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Document number; 1-24 characters.
  string number_;
  /// Document expiration date; pass null if not applicable.
  object_ptr<date> expiration_date_;
  /// Front side of the document.
  object_ptr<InputFile> front_side_;
  /// Reverse side of the document; only for driver license and identity card; pass null otherwise.
  object_ptr<InputFile> reverse_side_;
  /// Selfie with the document; pass null if unavailable.
  object_ptr<InputFile> selfie_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<InputFile>> translation_;

  /**
   * An identity document to be saved to Telegram Passport.
   */
  inputIdentityDocument();

  /**
   * An identity document to be saved to Telegram Passport.
   *
   * \param[in] number_ Document number; 1-24 characters.
   * \param[in] expiration_date_ Document expiration date; pass null if not applicable.
   * \param[in] front_side_ Front side of the document.
   * \param[in] reverse_side_ Reverse side of the document; only for driver license and identity card; pass null otherwise.
   * \param[in] selfie_ Selfie with the document; pass null if unavailable.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  inputIdentityDocument(string const &number_, object_ptr<date> &&expiration_date_, object_ptr<InputFile> &&front_side_, object_ptr<InputFile> &&reverse_side_, object_ptr<InputFile> &&selfie_, array<object_ptr<InputFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 767353688;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputMessageContent;

class ReplyMarkup;

class contact;

class location;

class venue;

/**
 * This class is an abstract base class.
 * Represents a single result of an inline query; for bots only.
 */
class InputInlineQueryResult: public Object {
 public:
};

/**
 * Represents a link to an animated GIF or an animated (i.e., without sound) H.264/MPEG-4 AVC video.
 */
class inputInlineQueryResultAnimation final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the query result.
  string title_;
  /// URL of the result thumbnail (JPEG, GIF, or MPEG4), if it exists.
  string thumbnail_url_;
  /// MIME type of the video thumbnail. If non-empty, must be one of &quot;image/jpeg&quot;, &quot;image/gif&quot; and &quot;video/mp4&quot;.
  string thumbnail_mime_type_;
  /// The URL of the video file (file size must not exceed 1MB).
  string video_url_;
  /// MIME type of the video file. Must be one of &quot;image/gif&quot; and &quot;video/mp4&quot;.
  string video_mime_type_;
  /// Duration of the video, in seconds.
  int32 video_duration_;
  /// Width of the video.
  int32 video_width_;
  /// Height of the video.
  int32 video_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAnimation, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an animated GIF or an animated (i.e., without sound) H.264/MPEG-4 AVC video.
   */
  inputInlineQueryResultAnimation();

  /**
   * Represents a link to an animated GIF or an animated (i.e., without sound) H.264/MPEG-4 AVC video.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the query result.
   * \param[in] thumbnail_url_ URL of the result thumbnail (JPEG, GIF, or MPEG4), if it exists.
   * \param[in] thumbnail_mime_type_ MIME type of the video thumbnail. If non-empty, must be one of &quot;image/jpeg&quot;, &quot;image/gif&quot; and &quot;video/mp4&quot;.
   * \param[in] video_url_ The URL of the video file (file size must not exceed 1MB).
   * \param[in] video_mime_type_ MIME type of the video file. Must be one of &quot;image/gif&quot; and &quot;video/mp4&quot;.
   * \param[in] video_duration_ Duration of the video, in seconds.
   * \param[in] video_width_ Width of the video.
   * \param[in] video_height_ Height of the video.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAnimation, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultAnimation(string const &id_, string const &title_, string const &thumbnail_url_, string const &thumbnail_mime_type_, string const &video_url_, string const &video_mime_type_, int32 video_duration_, int32 video_width_, int32 video_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1489808874;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to an article or web page.
 */
class inputInlineQueryResultArticle final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// URL of the result, if it exists.
  string url_;
  /// True, if the URL must be not shown.
  bool hide_url_;
  /// Title of the result.
  string title_;
  /// A short description of the result.
  string description_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an article or web page.
   */
  inputInlineQueryResultArticle();

  /**
   * Represents a link to an article or web page.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] url_ URL of the result, if it exists.
   * \param[in] hide_url_ True, if the URL must be not shown.
   * \param[in] title_ Title of the result.
   * \param[in] description_ A short description of the result.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultArticle(string const &id_, string const &url_, bool hide_url_, string const &title_, string const &description_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1973670156;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to an MP3 audio file.
 */
class inputInlineQueryResultAudio final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the audio file.
  string title_;
  /// Performer of the audio file.
  string performer_;
  /// The URL of the audio file.
  string audio_url_;
  /// Audio file duration, in seconds.
  int32 audio_duration_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAudio, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an MP3 audio file.
   */
  inputInlineQueryResultAudio();

  /**
   * Represents a link to an MP3 audio file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the audio file.
   * \param[in] performer_ Performer of the audio file.
   * \param[in] audio_url_ The URL of the audio file.
   * \param[in] audio_duration_ Audio file duration, in seconds.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageAudio, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultAudio(string const &id_, string const &title_, string const &performer_, string const &audio_url_, int32 audio_duration_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1260139988;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a user contact.
 */
class inputInlineQueryResultContact final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// User contact.
  object_ptr<contact> contact_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a user contact.
   */
  inputInlineQueryResultContact();

  /**
   * Represents a user contact.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] contact_ User contact.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultContact(string const &id_, object_ptr<contact> &&contact_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1846064594;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to a file.
 */
class inputInlineQueryResultDocument final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the resulting file.
  string title_;
  /// Short description of the result, if known.
  string description_;
  /// URL of the file.
  string document_url_;
  /// MIME type of the file content; only &quot;application/pdf&quot; and &quot;application/zip&quot; are currently allowed.
  string mime_type_;
  /// The URL of the file thumbnail, if it exists.
  string thumbnail_url_;
  /// Width of the thumbnail.
  int32 thumbnail_width_;
  /// Height of the thumbnail.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageDocument, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to a file.
   */
  inputInlineQueryResultDocument();

  /**
   * Represents a link to a file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the resulting file.
   * \param[in] description_ Short description of the result, if known.
   * \param[in] document_url_ URL of the file.
   * \param[in] mime_type_ MIME type of the file content; only &quot;application/pdf&quot; and &quot;application/zip&quot; are currently allowed.
   * \param[in] thumbnail_url_ The URL of the file thumbnail, if it exists.
   * \param[in] thumbnail_width_ Width of the thumbnail.
   * \param[in] thumbnail_height_ Height of the thumbnail.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageDocument, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultDocument(string const &id_, string const &title_, string const &description_, string const &document_url_, string const &mime_type_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 578801869;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a game.
 */
class inputInlineQueryResultGame final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Short name of the game.
  string game_short_name_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;

  /**
   * Represents a game.
   */
  inputInlineQueryResultGame();

  /**
   * Represents a game.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] game_short_name_ Short name of the game.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   */
  inputInlineQueryResultGame(string const &id_, string const &game_short_name_, object_ptr<ReplyMarkup> &&reply_markup_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 966074327;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a point on the map.
 */
class inputInlineQueryResultLocation final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Location result.
  object_ptr<location> location_;
  /// Amount of time relative to the message sent time until the location can be updated, in seconds.
  int32 live_period_;
  /// Title of the result.
  string title_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a point on the map.
   */
  inputInlineQueryResultLocation();

  /**
   * Represents a point on the map.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] location_ Location result.
   * \param[in] live_period_ Amount of time relative to the message sent time until the location can be updated, in seconds.
   * \param[in] title_ Title of the result.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultLocation(string const &id_, object_ptr<location> &&location_, int32 live_period_, string const &title_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1887650218;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents link to a JPEG image.
 */
class inputInlineQueryResultPhoto final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the result, if known.
  string title_;
  /// A short description of the result, if known.
  string description_;
  /// URL of the photo thumbnail, if it exists.
  string thumbnail_url_;
  /// The URL of the JPEG photo (photo size must not exceed 5MB).
  string photo_url_;
  /// Width of the photo.
  int32 photo_width_;
  /// Height of the photo.
  int32 photo_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessagePhoto, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents link to a JPEG image.
   */
  inputInlineQueryResultPhoto();

  /**
   * Represents link to a JPEG image.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the result, if known.
   * \param[in] description_ A short description of the result, if known.
   * \param[in] thumbnail_url_ URL of the photo thumbnail, if it exists.
   * \param[in] photo_url_ The URL of the JPEG photo (photo size must not exceed 5MB).
   * \param[in] photo_width_ Width of the photo.
   * \param[in] photo_height_ Height of the photo.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessagePhoto, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultPhoto(string const &id_, string const &title_, string const &description_, string const &thumbnail_url_, string const &photo_url_, int32 photo_width_, int32 photo_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1123338721;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to a WEBP, TGS, or WEBM sticker.
 */
class inputInlineQueryResultSticker final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// URL of the sticker thumbnail, if it exists.
  string thumbnail_url_;
  /// The URL of the WEBP, TGS, or WEBM sticker (sticker file size must not exceed 5MB).
  string sticker_url_;
  /// Width of the sticker.
  int32 sticker_width_;
  /// Height of the sticker.
  int32 sticker_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageSticker, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to a WEBP, TGS, or WEBM sticker.
   */
  inputInlineQueryResultSticker();

  /**
   * Represents a link to a WEBP, TGS, or WEBM sticker.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] thumbnail_url_ URL of the sticker thumbnail, if it exists.
   * \param[in] sticker_url_ The URL of the WEBP, TGS, or WEBM sticker (sticker file size must not exceed 5MB).
   * \param[in] sticker_width_ Width of the sticker.
   * \param[in] sticker_height_ Height of the sticker.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageSticker, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultSticker(string const &id_, string const &thumbnail_url_, string const &sticker_url_, int32 sticker_width_, int32 sticker_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 274007129;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents information about a venue.
 */
class inputInlineQueryResultVenue final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Venue result.
  object_ptr<venue> venue_;
  /// URL of the result thumbnail, if it exists.
  string thumbnail_url_;
  /// Thumbnail width, if known.
  int32 thumbnail_width_;
  /// Thumbnail height, if known.
  int32 thumbnail_height_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents information about a venue.
   */
  inputInlineQueryResultVenue();

  /**
   * Represents information about a venue.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] venue_ Venue result.
   * \param[in] thumbnail_url_ URL of the result thumbnail, if it exists.
   * \param[in] thumbnail_width_ Thumbnail width, if known.
   * \param[in] thumbnail_height_ Thumbnail height, if known.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultVenue(string const &id_, object_ptr<venue> &&venue_, string const &thumbnail_url_, int32 thumbnail_width_, int32 thumbnail_height_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 541704509;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to a page containing an embedded video player or a video file.
 */
class inputInlineQueryResultVideo final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the result.
  string title_;
  /// A short description of the result, if known.
  string description_;
  /// The URL of the video thumbnail (JPEG), if it exists.
  string thumbnail_url_;
  /// URL of the embedded video player or video file.
  string video_url_;
  /// MIME type of the content of the video URL, only &quot;text/html&quot; or &quot;video/mp4&quot; are currently supported.
  string mime_type_;
  /// Width of the video.
  int32 video_width_;
  /// Height of the video.
  int32 video_height_;
  /// Video duration, in seconds.
  int32 video_duration_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVideo, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to a page containing an embedded video player or a video file.
   */
  inputInlineQueryResultVideo();

  /**
   * Represents a link to a page containing an embedded video player or a video file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the result.
   * \param[in] description_ A short description of the result, if known.
   * \param[in] thumbnail_url_ The URL of the video thumbnail (JPEG), if it exists.
   * \param[in] video_url_ URL of the embedded video player or video file.
   * \param[in] mime_type_ MIME type of the content of the video URL, only &quot;text/html&quot; or &quot;video/mp4&quot; are currently supported.
   * \param[in] video_width_ Width of the video.
   * \param[in] video_height_ Height of the video.
   * \param[in] video_duration_ Video duration, in seconds.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVideo, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultVideo(string const &id_, string const &title_, string const &description_, string const &thumbnail_url_, string const &video_url_, string const &mime_type_, int32 video_width_, int32 video_height_, int32 video_duration_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1724073191;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Represents a link to an opus-encoded audio file within an OGG container, single channel audio.
 */
class inputInlineQueryResultVoiceNote final : public InputInlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Unique identifier of the query result.
  string id_;
  /// Title of the voice note.
  string title_;
  /// The URL of the voice note file.
  string voice_note_url_;
  /// Duration of the voice note, in seconds.
  int32 voice_note_duration_;
  /// The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
  object_ptr<ReplyMarkup> reply_markup_;
  /// The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVoiceNote, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
  object_ptr<InputMessageContent> input_message_content_;

  /**
   * Represents a link to an opus-encoded audio file within an OGG container, single channel audio.
   */
  inputInlineQueryResultVoiceNote();

  /**
   * Represents a link to an opus-encoded audio file within an OGG container, single channel audio.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] title_ Title of the voice note.
   * \param[in] voice_note_url_ The URL of the voice note file.
   * \param[in] voice_note_duration_ Duration of the voice note, in seconds.
   * \param[in] reply_markup_ The message reply markup; pass null if none. Must be of type replyMarkupInlineKeyboard or null.
   * \param[in] input_message_content_ The content of the message to be sent. Must be one of the following types: inputMessageText, inputMessageVoiceNote, inputMessageInvoice, inputMessageLocation, inputMessageVenue or inputMessageContact.
   */
  inputInlineQueryResultVoiceNote(string const &id_, string const &title_, string const &voice_note_url_, int32 voice_note_duration_, object_ptr<ReplyMarkup> &&reply_markup_, object_ptr<InputMessageContent> &&input_message_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1790072503;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class TelegramPaymentPurpose;

/**
 * This class is an abstract base class.
 * Describes an invoice to process.
 */
class InputInvoice: public Object {
 public:
};

/**
 * An invoice from a message of the type messageInvoice or paid media purchase from messagePaidMedia.
 */
class inputInvoiceMessage final : public InputInvoice {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Chat identifier of the message.
  int53 chat_id_;
  /// Message identifier. Use messageProperties.can_be_paid to check whether the message can be used in the method.
  int53 message_id_;

  /**
   * An invoice from a message of the type messageInvoice or paid media purchase from messagePaidMedia.
   */
  inputInvoiceMessage();

  /**
   * An invoice from a message of the type messageInvoice or paid media purchase from messagePaidMedia.
   *
   * \param[in] chat_id_ Chat identifier of the message.
   * \param[in] message_id_ Message identifier. Use messageProperties.can_be_paid to check whether the message can be used in the method.
   */
  inputInvoiceMessage(int53 chat_id_, int53 message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1490872848;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An invoice from a link of the type internalLinkTypeInvoice.
 */
class inputInvoiceName final : public InputInvoice {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the invoice.
  string name_;

  /**
   * An invoice from a link of the type internalLinkTypeInvoice.
   */
  inputInvoiceName();

  /**
   * An invoice from a link of the type internalLinkTypeInvoice.
   *
   * \param[in] name_ Name of the invoice.
   */
  explicit inputInvoiceName(string const &name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1312155917;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An invoice for a payment toward Telegram; must not be used in the in-store apps.
 */
class inputInvoiceTelegram final : public InputInvoice {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Transaction purpose.
  object_ptr<TelegramPaymentPurpose> purpose_;

  /**
   * An invoice for a payment toward Telegram; must not be used in the in-store apps.
   */
  inputInvoiceTelegram();

  /**
   * An invoice for a payment toward Telegram; must not be used in the in-store apps.
   *
   * \param[in] purpose_ Transaction purpose.
   */
  explicit inputInvoiceTelegram(object_ptr<TelegramPaymentPurpose> &&purpose_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1762853139;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

class MessageSelfDestructType;

class PollType;

class contact;

class formattedText;

class inputPaidMedia;

class inputThumbnail;

class invoice;

class linkPreviewOptions;

class location;

class messageCopyOptions;

class venue;

/**
 * This class is an abstract base class.
 * The content of a message to send.
 */
class InputMessageContent: public Object {
 public:
};

/**
 * A text message.
 */
class inputMessageText final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Formatted text to be sent; 0-getOption(&quot;message_text_length_max&quot;) characters. Only Bold, Italic, Underline, Strikethrough, Spoiler, CustomEmoji, BlockQuote, ExpandableBlockQuote, Code, Pre, PreCode, TextUrl and MentionName entities are allowed to be specified manually.
  object_ptr<formattedText> text_;
  /// Options to be used for generation of a link preview; may be null if none; pass null to use default link preview options.
  object_ptr<linkPreviewOptions> link_preview_options_;
  /// True, if a chat message draft must be deleted.
  bool clear_draft_;

  /**
   * A text message.
   */
  inputMessageText();

  /**
   * A text message.
   *
   * \param[in] text_ Formatted text to be sent; 0-getOption(&quot;message_text_length_max&quot;) characters. Only Bold, Italic, Underline, Strikethrough, Spoiler, CustomEmoji, BlockQuote, ExpandableBlockQuote, Code, Pre, PreCode, TextUrl and MentionName entities are allowed to be specified manually.
   * \param[in] link_preview_options_ Options to be used for generation of a link preview; may be null if none; pass null to use default link preview options.
   * \param[in] clear_draft_ True, if a chat message draft must be deleted.
   */
  inputMessageText(object_ptr<formattedText> &&text_, object_ptr<linkPreviewOptions> &&link_preview_options_, bool clear_draft_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -212805484;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An animation message (GIF-style).
 */
class inputMessageAnimation final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Animation file to be sent.
  object_ptr<InputFile> animation_;
  /// Animation thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// File identifiers of the stickers added to the animation, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Duration of the animation, in seconds.
  int32 duration_;
  /// Width of the animation; may be replaced by the server.
  int32 width_;
  /// Height of the animation; may be replaced by the server.
  int32 height_;
  /// Animation caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;
  /// True, if the caption must be shown above the animation; otherwise, the caption must be shown below the animation; not supported in secret chats.
  bool show_caption_above_media_;
  /// True, if the animation preview must be covered by a spoiler animation; not supported in secret chats.
  bool has_spoiler_;

  /**
   * An animation message (GIF-style).
   */
  inputMessageAnimation();

  /**
   * An animation message (GIF-style).
   *
   * \param[in] animation_ Animation file to be sent.
   * \param[in] thumbnail_ Animation thumbnail; pass null to skip thumbnail uploading.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the animation, if applicable.
   * \param[in] duration_ Duration of the animation, in seconds.
   * \param[in] width_ Width of the animation; may be replaced by the server.
   * \param[in] height_ Height of the animation; may be replaced by the server.
   * \param[in] caption_ Animation caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   * \param[in] show_caption_above_media_ True, if the caption must be shown above the animation; otherwise, the caption must be shown below the animation; not supported in secret chats.
   * \param[in] has_spoiler_ True, if the animation preview must be covered by a spoiler animation; not supported in secret chats.
   */
  inputMessageAnimation(object_ptr<InputFile> &&animation_, object_ptr<inputThumbnail> &&thumbnail_, array<int32> &&added_sticker_file_ids_, int32 duration_, int32 width_, int32 height_, object_ptr<formattedText> &&caption_, bool show_caption_above_media_, bool has_spoiler_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -210404059;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An audio message.
 */
class inputMessageAudio final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Audio file to be sent.
  object_ptr<InputFile> audio_;
  /// Thumbnail of the cover for the album; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> album_cover_thumbnail_;
  /// Duration of the audio, in seconds; may be replaced by the server.
  int32 duration_;
  /// Title of the audio; 0-64 characters; may be replaced by the server.
  string title_;
  /// Performer of the audio; 0-64 characters, may be replaced by the server.
  string performer_;
  /// Audio caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;

  /**
   * An audio message.
   */
  inputMessageAudio();

  /**
   * An audio message.
   *
   * \param[in] audio_ Audio file to be sent.
   * \param[in] album_cover_thumbnail_ Thumbnail of the cover for the album; pass null to skip thumbnail uploading.
   * \param[in] duration_ Duration of the audio, in seconds; may be replaced by the server.
   * \param[in] title_ Title of the audio; 0-64 characters; may be replaced by the server.
   * \param[in] performer_ Performer of the audio; 0-64 characters, may be replaced by the server.
   * \param[in] caption_ Audio caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   */
  inputMessageAudio(object_ptr<InputFile> &&audio_, object_ptr<inputThumbnail> &&album_cover_thumbnail_, int32 duration_, string const &title_, string const &performer_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -626786126;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A document message (general file).
 */
class inputMessageDocument final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Document to be sent.
  object_ptr<InputFile> document_;
  /// Document thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// Pass true to disable automatic file type detection and send the document as a file. Always true for files sent to secret chats.
  bool disable_content_type_detection_;
  /// Document caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;

  /**
   * A document message (general file).
   */
  inputMessageDocument();

  /**
   * A document message (general file).
   *
   * \param[in] document_ Document to be sent.
   * \param[in] thumbnail_ Document thumbnail; pass null to skip thumbnail uploading.
   * \param[in] disable_content_type_detection_ Pass true to disable automatic file type detection and send the document as a file. Always true for files sent to secret chats.
   * \param[in] caption_ Document caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   */
  inputMessageDocument(object_ptr<InputFile> &&document_, object_ptr<inputThumbnail> &&thumbnail_, bool disable_content_type_detection_, object_ptr<formattedText> &&caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1633383097;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with paid media; can be used only in channel chats with supergroupFullInfo.has_paid_media_allowed.
 */
class inputMessagePaidMedia final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The number of Telegram Stars that must be paid to see the media; 1-getOption(&quot;paid_media_message_star_count_max&quot;).
  int53 star_count_;
  /// The content of the paid media.
  array<object_ptr<inputPaidMedia>> paid_media_;
  /// Message caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;
  /// True, if the caption must be shown above the media; otherwise, the caption must be shown below the media; not supported in secret chats.
  bool show_caption_above_media_;
  /// Bot-provided data for the paid media; bots only.
  string payload_;

  /**
   * A message with paid media; can be used only in channel chats with supergroupFullInfo.has_paid_media_allowed.
   */
  inputMessagePaidMedia();

  /**
   * A message with paid media; can be used only in channel chats with supergroupFullInfo.has_paid_media_allowed.
   *
   * \param[in] star_count_ The number of Telegram Stars that must be paid to see the media; 1-getOption(&quot;paid_media_message_star_count_max&quot;).
   * \param[in] paid_media_ The content of the paid media.
   * \param[in] caption_ Message caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   * \param[in] show_caption_above_media_ True, if the caption must be shown above the media; otherwise, the caption must be shown below the media; not supported in secret chats.
   * \param[in] payload_ Bot-provided data for the paid media; bots only.
   */
  inputMessagePaidMedia(int53 star_count_, array<object_ptr<inputPaidMedia>> &&paid_media_, object_ptr<formattedText> &&caption_, bool show_caption_above_media_, string const &payload_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1274819374;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A photo message.
 */
class inputMessagePhoto final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Photo to send. The photo must be at most 10 MB in size. The photo's width and height must not exceed 10000 in total. Width and height ratio must be at most 20.
  object_ptr<InputFile> photo_;
  /// Photo thumbnail to be sent; pass null to skip thumbnail uploading. The thumbnail is sent to the other party only in secret chats.
  object_ptr<inputThumbnail> thumbnail_;
  /// File identifiers of the stickers added to the photo, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Photo width.
  int32 width_;
  /// Photo height.
  int32 height_;
  /// Photo caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;
  /// True, if the caption must be shown above the photo; otherwise, the caption must be shown below the photo; not supported in secret chats.
  bool show_caption_above_media_;
  /// Photo self-destruct type; pass null if none; private chats only.
  object_ptr<MessageSelfDestructType> self_destruct_type_;
  /// True, if the photo preview must be covered by a spoiler animation; not supported in secret chats.
  bool has_spoiler_;

  /**
   * A photo message.
   */
  inputMessagePhoto();

  /**
   * A photo message.
   *
   * \param[in] photo_ Photo to send. The photo must be at most 10 MB in size. The photo's width and height must not exceed 10000 in total. Width and height ratio must be at most 20.
   * \param[in] thumbnail_ Photo thumbnail to be sent; pass null to skip thumbnail uploading. The thumbnail is sent to the other party only in secret chats.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the photo, if applicable.
   * \param[in] width_ Photo width.
   * \param[in] height_ Photo height.
   * \param[in] caption_ Photo caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   * \param[in] show_caption_above_media_ True, if the caption must be shown above the photo; otherwise, the caption must be shown below the photo; not supported in secret chats.
   * \param[in] self_destruct_type_ Photo self-destruct type; pass null if none; private chats only.
   * \param[in] has_spoiler_ True, if the photo preview must be covered by a spoiler animation; not supported in secret chats.
   */
  inputMessagePhoto(object_ptr<InputFile> &&photo_, object_ptr<inputThumbnail> &&thumbnail_, array<int32> &&added_sticker_file_ids_, int32 width_, int32 height_, object_ptr<formattedText> &&caption_, bool show_caption_above_media_, object_ptr<MessageSelfDestructType> &&self_destruct_type_, bool has_spoiler_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -810129442;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A sticker message.
 */
class inputMessageSticker final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Sticker to be sent.
  object_ptr<InputFile> sticker_;
  /// Sticker thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// Sticker width.
  int32 width_;
  /// Sticker height.
  int32 height_;
  /// Emoji used to choose the sticker.
  string emoji_;

  /**
   * A sticker message.
   */
  inputMessageSticker();

  /**
   * A sticker message.
   *
   * \param[in] sticker_ Sticker to be sent.
   * \param[in] thumbnail_ Sticker thumbnail; pass null to skip thumbnail uploading.
   * \param[in] width_ Sticker width.
   * \param[in] height_ Sticker height.
   * \param[in] emoji_ Emoji used to choose the sticker.
   */
  inputMessageSticker(object_ptr<InputFile> &&sticker_, object_ptr<inputThumbnail> &&thumbnail_, int32 width_, int32 height_, string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1072805625;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video message.
 */
class inputMessageVideo final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Video to be sent. The video is expected to be re-encoded to MPEG4 format with H.264 codec by the sender.
  object_ptr<InputFile> video_;
  /// Video thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// File identifiers of the stickers added to the video, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Duration of the video, in seconds.
  int32 duration_;
  /// Video width.
  int32 width_;
  /// Video height.
  int32 height_;
  /// True, if the video is expected to be streamed.
  bool supports_streaming_;
  /// Video caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;
  /// True, if the caption must be shown above the video; otherwise, the caption must be shown below the video; not supported in secret chats.
  bool show_caption_above_media_;
  /// Video self-destruct type; pass null if none; private chats only.
  object_ptr<MessageSelfDestructType> self_destruct_type_;
  /// True, if the video preview must be covered by a spoiler animation; not supported in secret chats.
  bool has_spoiler_;

  /**
   * A video message.
   */
  inputMessageVideo();

  /**
   * A video message.
   *
   * \param[in] video_ Video to be sent. The video is expected to be re-encoded to MPEG4 format with H.264 codec by the sender.
   * \param[in] thumbnail_ Video thumbnail; pass null to skip thumbnail uploading.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the video, if applicable.
   * \param[in] duration_ Duration of the video, in seconds.
   * \param[in] width_ Video width.
   * \param[in] height_ Video height.
   * \param[in] supports_streaming_ True, if the video is expected to be streamed.
   * \param[in] caption_ Video caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   * \param[in] show_caption_above_media_ True, if the caption must be shown above the video; otherwise, the caption must be shown below the video; not supported in secret chats.
   * \param[in] self_destruct_type_ Video self-destruct type; pass null if none; private chats only.
   * \param[in] has_spoiler_ True, if the video preview must be covered by a spoiler animation; not supported in secret chats.
   */
  inputMessageVideo(object_ptr<InputFile> &&video_, object_ptr<inputThumbnail> &&thumbnail_, array<int32> &&added_sticker_file_ids_, int32 duration_, int32 width_, int32 height_, bool supports_streaming_, object_ptr<formattedText> &&caption_, bool show_caption_above_media_, object_ptr<MessageSelfDestructType> &&self_destruct_type_, bool has_spoiler_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 615537686;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video note message.
 */
class inputMessageVideoNote final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Video note to be sent. The video is expected to be encoded to MPEG4 format with H.264 codec and have no data outside of the visible circle.
  object_ptr<InputFile> video_note_;
  /// Video thumbnail; may be null if empty; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// Duration of the video, in seconds; 0-60.
  int32 duration_;
  /// Video width and height; must be positive and not greater than 640.
  int32 length_;
  /// Video note self-destruct type; may be null if none; pass null if none; private chats only.
  object_ptr<MessageSelfDestructType> self_destruct_type_;

  /**
   * A video note message.
   */
  inputMessageVideoNote();

  /**
   * A video note message.
   *
   * \param[in] video_note_ Video note to be sent. The video is expected to be encoded to MPEG4 format with H.264 codec and have no data outside of the visible circle.
   * \param[in] thumbnail_ Video thumbnail; may be null if empty; pass null to skip thumbnail uploading.
   * \param[in] duration_ Duration of the video, in seconds; 0-60.
   * \param[in] length_ Video width and height; must be positive and not greater than 640.
   * \param[in] self_destruct_type_ Video note self-destruct type; may be null if none; pass null if none; private chats only.
   */
  inputMessageVideoNote(object_ptr<InputFile> &&video_note_, object_ptr<inputThumbnail> &&thumbnail_, int32 duration_, int32 length_, object_ptr<MessageSelfDestructType> &&self_destruct_type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -714598691;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A voice note message.
 */
class inputMessageVoiceNote final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Voice note to be sent. The voice note must be encoded with the Opus codec and stored inside an OGG container with a single audio channel, or be in MP3 or M4A format as regular audio.
  object_ptr<InputFile> voice_note_;
  /// Duration of the voice note, in seconds.
  int32 duration_;
  /// Waveform representation of the voice note in 5-bit format.
  bytes waveform_;
  /// Voice note caption; may be null if empty; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> caption_;
  /// Voice note self-destruct type; may be null if none; pass null if none; private chats only.
  object_ptr<MessageSelfDestructType> self_destruct_type_;

  /**
   * A voice note message.
   */
  inputMessageVoiceNote();

  /**
   * A voice note message.
   *
   * \param[in] voice_note_ Voice note to be sent. The voice note must be encoded with the Opus codec and stored inside an OGG container with a single audio channel, or be in MP3 or M4A format as regular audio.
   * \param[in] duration_ Duration of the voice note, in seconds.
   * \param[in] waveform_ Waveform representation of the voice note in 5-bit format.
   * \param[in] caption_ Voice note caption; may be null if empty; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   * \param[in] self_destruct_type_ Voice note self-destruct type; may be null if none; pass null if none; private chats only.
   */
  inputMessageVoiceNote(object_ptr<InputFile> &&voice_note_, int32 duration_, bytes const &waveform_, object_ptr<formattedText> &&caption_, object_ptr<MessageSelfDestructType> &&self_destruct_type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1461977004;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a location.
 */
class inputMessageLocation final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Location to be sent.
  object_ptr<location> location_;
  /// Period for which the location can be updated, in seconds; must be between 60 and 86400 for a temporary live location, 0x7FFFFFFF for permanent live location, and 0 otherwise.
  int32 live_period_;
  /// For live locations, a direction in which the location moves, in degrees; 1-360. Pass 0 if unknown.
  int32 heading_;
  /// For live locations, a maximum distance to another chat member for proximity alerts, in meters (0-100000). Pass 0 if the notification is disabled. Can't be enabled in channels and Saved Messages.
  int32 proximity_alert_radius_;

  /**
   * A message with a location.
   */
  inputMessageLocation();

  /**
   * A message with a location.
   *
   * \param[in] location_ Location to be sent.
   * \param[in] live_period_ Period for which the location can be updated, in seconds; must be between 60 and 86400 for a temporary live location, 0x7FFFFFFF for permanent live location, and 0 otherwise.
   * \param[in] heading_ For live locations, a direction in which the location moves, in degrees; 1-360. Pass 0 if unknown.
   * \param[in] proximity_alert_radius_ For live locations, a maximum distance to another chat member for proximity alerts, in meters (0-100000). Pass 0 if the notification is disabled. Can't be enabled in channels and Saved Messages.
   */
  inputMessageLocation(object_ptr<location> &&location_, int32 live_period_, int32 heading_, int32 proximity_alert_radius_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 648735088;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with information about a venue.
 */
class inputMessageVenue final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Venue to send.
  object_ptr<venue> venue_;

  /**
   * A message with information about a venue.
   */
  inputMessageVenue();

  /**
   * A message with information about a venue.
   *
   * \param[in] venue_ Venue to send.
   */
  explicit inputMessageVenue(object_ptr<venue> &&venue_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1447926269;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message containing a user contact.
 */
class inputMessageContact final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Contact to send.
  object_ptr<contact> contact_;

  /**
   * A message containing a user contact.
   */
  inputMessageContact();

  /**
   * A message containing a user contact.
   *
   * \param[in] contact_ Contact to send.
   */
  explicit inputMessageContact(object_ptr<contact> &&contact_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -982446849;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A dice message.
 */
class inputMessageDice final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Emoji on which the dice throw animation is based.
  string emoji_;
  /// True, if the chat message draft must be deleted.
  bool clear_draft_;

  /**
   * A dice message.
   */
  inputMessageDice();

  /**
   * A dice message.
   *
   * \param[in] emoji_ Emoji on which the dice throw animation is based.
   * \param[in] clear_draft_ True, if the chat message draft must be deleted.
   */
  inputMessageDice(string const &emoji_, bool clear_draft_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 841574313;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a game; not supported for channels or secret chats.
 */
class inputMessageGame final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the bot that owns the game.
  int53 bot_user_id_;
  /// Short name of the game.
  string game_short_name_;

  /**
   * A message with a game; not supported for channels or secret chats.
   */
  inputMessageGame();

  /**
   * A message with a game; not supported for channels or secret chats.
   *
   * \param[in] bot_user_id_ User identifier of the bot that owns the game.
   * \param[in] game_short_name_ Short name of the game.
   */
  inputMessageGame(int53 bot_user_id_, string const &game_short_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1252944610;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with an invoice; can be used only by bots.
 */
class inputMessageInvoice final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Invoice.
  object_ptr<invoice> invoice_;
  /// Product title; 1-32 characters.
  string title_;
  /// Product description; 0-255 characters.
  string description_;
  /// Product photo URL; optional.
  string photo_url_;
  /// Product photo size.
  int32 photo_size_;
  /// Product photo width.
  int32 photo_width_;
  /// Product photo height.
  int32 photo_height_;
  /// The invoice payload.
  bytes payload_;
  /// Payment provider token; may be empty for payments in Telegram Stars.
  string provider_token_;
  /// JSON-encoded data about the invoice, which will be shared with the payment provider.
  string provider_data_;
  /// Unique invoice bot deep link parameter for the generation of this invoice. If empty, it would be possible to pay directly from forwards of the invoice message.
  string start_parameter_;
  /// The content of paid media attached to the invoice; pass null if none.
  object_ptr<inputPaidMedia> paid_media_;
  /// Paid media caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
  object_ptr<formattedText> paid_media_caption_;

  /**
   * A message with an invoice; can be used only by bots.
   */
  inputMessageInvoice();

  /**
   * A message with an invoice; can be used only by bots.
   *
   * \param[in] invoice_ Invoice.
   * \param[in] title_ Product title; 1-32 characters.
   * \param[in] description_ Product description; 0-255 characters.
   * \param[in] photo_url_ Product photo URL; optional.
   * \param[in] photo_size_ Product photo size.
   * \param[in] photo_width_ Product photo width.
   * \param[in] photo_height_ Product photo height.
   * \param[in] payload_ The invoice payload.
   * \param[in] provider_token_ Payment provider token; may be empty for payments in Telegram Stars.
   * \param[in] provider_data_ JSON-encoded data about the invoice, which will be shared with the payment provider.
   * \param[in] start_parameter_ Unique invoice bot deep link parameter for the generation of this invoice. If empty, it would be possible to pay directly from forwards of the invoice message.
   * \param[in] paid_media_ The content of paid media attached to the invoice; pass null if none.
   * \param[in] paid_media_caption_ Paid media caption; pass null to use an empty caption; 0-getOption(&quot;message_caption_length_max&quot;) characters.
   */
  inputMessageInvoice(object_ptr<invoice> &&invoice_, string const &title_, string const &description_, string const &photo_url_, int32 photo_size_, int32 photo_width_, int32 photo_height_, bytes const &payload_, string const &provider_token_, string const &provider_data_, string const &start_parameter_, object_ptr<inputPaidMedia> &&paid_media_, object_ptr<formattedText> &&paid_media_caption_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1162047631;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a poll. Polls can't be sent to secret chats. Polls can be sent only to a private chat with a bot.
 */
class inputMessagePoll final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Poll question; 1-255 characters (up to 300 characters for bots). Only custom emoji entities are allowed to be added and only by Premium users.
  object_ptr<formattedText> question_;
  /// List of poll answer options, 2-10 strings 1-100 characters each. Only custom emoji entities are allowed to be added and only by Premium users.
  array<object_ptr<formattedText>> options_;
  /// True, if the poll voters are anonymous. Non-anonymous polls can't be sent or forwarded to channels.
  bool is_anonymous_;
  /// Type of the poll.
  object_ptr<PollType> type_;
  /// Amount of time the poll will be active after creation, in seconds; for bots only.
  int32 open_period_;
  /// Point in time (Unix timestamp) when the poll will automatically be closed; for bots only.
  int32 close_date_;
  /// True, if the poll needs to be sent already closed; for bots only.
  bool is_closed_;

  /**
   * A message with a poll. Polls can't be sent to secret chats. Polls can be sent only to a private chat with a bot.
   */
  inputMessagePoll();

  /**
   * A message with a poll. Polls can't be sent to secret chats. Polls can be sent only to a private chat with a bot.
   *
   * \param[in] question_ Poll question; 1-255 characters (up to 300 characters for bots). Only custom emoji entities are allowed to be added and only by Premium users.
   * \param[in] options_ List of poll answer options, 2-10 strings 1-100 characters each. Only custom emoji entities are allowed to be added and only by Premium users.
   * \param[in] is_anonymous_ True, if the poll voters are anonymous. Non-anonymous polls can't be sent or forwarded to channels.
   * \param[in] type_ Type of the poll.
   * \param[in] open_period_ Amount of time the poll will be active after creation, in seconds; for bots only.
   * \param[in] close_date_ Point in time (Unix timestamp) when the poll will automatically be closed; for bots only.
   * \param[in] is_closed_ True, if the poll needs to be sent already closed; for bots only.
   */
  inputMessagePoll(object_ptr<formattedText> &&question_, array<object_ptr<formattedText>> &&options_, bool is_anonymous_, object_ptr<PollType> &&type_, int32 open_period_, int32 close_date_, bool is_closed_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -263337164;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A message with a forwarded story. Stories can't be sent to secret chats. A story can be forwarded only if story.can_be_forwarded.
 */
class inputMessageStory final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the chat that posted the story.
  int53 story_sender_chat_id_;
  /// Story identifier.
  int32 story_id_;

  /**
   * A message with a forwarded story. Stories can't be sent to secret chats. A story can be forwarded only if story.can_be_forwarded.
   */
  inputMessageStory();

  /**
   * A message with a forwarded story. Stories can't be sent to secret chats. A story can be forwarded only if story.can_be_forwarded.
   *
   * \param[in] story_sender_chat_id_ Identifier of the chat that posted the story.
   * \param[in] story_id_ Story identifier.
   */
  inputMessageStory(int53 story_sender_chat_id_, int32 story_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 554278243;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A forwarded message.
 */
class inputMessageForwarded final : public InputMessageContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier for the chat this forwarded message came from.
  int53 from_chat_id_;
  /// Identifier of the message to forward. A message can be forwarded only if messageProperties.can_be_forwarded.
  int53 message_id_;
  /// True, if a game message is being shared from a launched game; applies only to game messages.
  bool in_game_share_;
  /// Options to be used to copy content of the message without reference to the original sender; pass null to forward the message as usual.
  object_ptr<messageCopyOptions> copy_options_;

  /**
   * A forwarded message.
   */
  inputMessageForwarded();

  /**
   * A forwarded message.
   *
   * \param[in] from_chat_id_ Identifier for the chat this forwarded message came from.
   * \param[in] message_id_ Identifier of the message to forward. A message can be forwarded only if messageProperties.can_be_forwarded.
   * \param[in] in_game_share_ True, if a game message is being shared from a launched game; applies only to game messages.
   * \param[in] copy_options_ Options to be used to copy content of the message without reference to the original sender; pass null to forward the message as usual.
   */
  inputMessageForwarded(int53 from_chat_id_, int53 message_id_, bool in_game_share_, object_ptr<messageCopyOptions> &&copy_options_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1696232440;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class inputTextQuote;

/**
 * This class is an abstract base class.
 * Contains information about the message or the story to be replied.
 */
class InputMessageReplyTo: public Object {
 public:
};

/**
 * Describes a message to be replied in the same chat and forum topic.
 */
class inputMessageReplyToMessage final : public InputMessageReplyTo {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The identifier of the message to be replied in the same chat and forum topic. A message can be replied in the same chat and forum topic only if messageProperties.can_be_replied.
  int53 message_id_;
  /// Quote from the message to be replied; pass null if none. Must always be null for replies in secret chats.
  object_ptr<inputTextQuote> quote_;

  /**
   * Describes a message to be replied in the same chat and forum topic.
   */
  inputMessageReplyToMessage();

  /**
   * Describes a message to be replied in the same chat and forum topic.
   *
   * \param[in] message_id_ The identifier of the message to be replied in the same chat and forum topic. A message can be replied in the same chat and forum topic only if messageProperties.can_be_replied.
   * \param[in] quote_ Quote from the message to be replied; pass null if none. Must always be null for replies in secret chats.
   */
  inputMessageReplyToMessage(int53 message_id_, object_ptr<inputTextQuote> &&quote_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1033987837;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a message to be replied that is from a different chat or a forum topic; not supported in secret chats.
 */
class inputMessageReplyToExternalMessage final : public InputMessageReplyTo {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The identifier of the chat to which the message to be replied belongs.
  int53 chat_id_;
  /// The identifier of the message to be replied in the specified chat. A message can be replied in another chat or forum topic only if messageProperties.can_be_replied_in_another_chat.
  int53 message_id_;
  /// Quote from the message to be replied; pass null if none.
  object_ptr<inputTextQuote> quote_;

  /**
   * Describes a message to be replied that is from a different chat or a forum topic; not supported in secret chats.
   */
  inputMessageReplyToExternalMessage();

  /**
   * Describes a message to be replied that is from a different chat or a forum topic; not supported in secret chats.
   *
   * \param[in] chat_id_ The identifier of the chat to which the message to be replied belongs.
   * \param[in] message_id_ The identifier of the message to be replied in the specified chat. A message can be replied in another chat or forum topic only if messageProperties.can_be_replied_in_another_chat.
   * \param[in] quote_ Quote from the message to be replied; pass null if none.
   */
  inputMessageReplyToExternalMessage(int53 chat_id_, int53 message_id_, object_ptr<inputTextQuote> &&quote_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1993530582;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * Describes a story to be replied.
 */
class inputMessageReplyToStory final : public InputMessageReplyTo {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The identifier of the sender of the story. Currently, stories can be replied only in the sender's chat and channel stories can't be replied.
  int53 story_sender_chat_id_;
  /// The identifier of the story.
  int32 story_id_;

  /**
   * Describes a story to be replied.
   */
  inputMessageReplyToStory();

  /**
   * Describes a story to be replied.
   *
   * \param[in] story_sender_chat_id_ The identifier of the sender of the story. Currently, stories can be replied only in the sender's chat and channel stories can't be replied.
   * \param[in] story_id_ The identifier of the story.
   */
  inputMessageReplyToStory(int53 story_sender_chat_id_, int32 story_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1370410616;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

class InputPaidMediaType;

class inputThumbnail;

/**
 * Describes a paid media to be sent.
 */
class inputPaidMedia final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of the media.
  object_ptr<InputPaidMediaType> type_;
  /// Photo or video to be sent.
  object_ptr<InputFile> media_;
  /// Media thumbnail; pass null to skip thumbnail uploading.
  object_ptr<inputThumbnail> thumbnail_;
  /// File identifiers of the stickers added to the media, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Media width.
  int32 width_;
  /// Media height.
  int32 height_;

  /**
   * Describes a paid media to be sent.
   */
  inputPaidMedia();

  /**
   * Describes a paid media to be sent.
   *
   * \param[in] type_ Type of the media.
   * \param[in] media_ Photo or video to be sent.
   * \param[in] thumbnail_ Media thumbnail; pass null to skip thumbnail uploading.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the media, if applicable.
   * \param[in] width_ Media width.
   * \param[in] height_ Media height.
   */
  inputPaidMedia(object_ptr<InputPaidMediaType> &&type_, object_ptr<InputFile> &&media_, object_ptr<inputThumbnail> &&thumbnail_, array<int32> &&added_sticker_file_ids_, int32 width_, int32 height_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 475844035;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Describes type of paid media to sent.
 */
class InputPaidMediaType: public Object {
 public:
};

/**
 * The media is a photo. The photo must be at most 10 MB in size. The photo's width and height must not exceed 10000 in total. Width and height ratio must be at most 20.
 */
class inputPaidMediaTypePhoto final : public InputPaidMediaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The media is a photo. The photo must be at most 10 MB in size. The photo's width and height must not exceed 10000 in total. Width and height ratio must be at most 20.
   */
  inputPaidMediaTypePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -761660134;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The media is a video.
 */
class inputPaidMediaTypeVideo final : public InputPaidMediaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Duration of the video, in seconds.
  int32 duration_;
  /// True, if the video is expected to be streamed.
  bool supports_streaming_;

  /**
   * The media is a video.
   */
  inputPaidMediaTypeVideo();

  /**
   * The media is a video.
   *
   * \param[in] duration_ Duration of the video, in seconds.
   * \param[in] supports_streaming_ True, if the video is expected to be streamed.
   */
  inputPaidMediaTypeVideo(int32 duration_, bool supports_streaming_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1336673796;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class address;

class inputIdentityDocument;

class inputPersonalDocument;

class personalDetails;

/**
 * This class is an abstract base class.
 * Contains information about a Telegram Passport element to be saved.
 */
class InputPassportElement: public Object {
 public:
};

/**
 * A Telegram Passport element to be saved containing the user's personal details.
 */
class inputPassportElementPersonalDetails final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Personal details of the user.
  object_ptr<personalDetails> personal_details_;

  /**
   * A Telegram Passport element to be saved containing the user's personal details.
   */
  inputPassportElementPersonalDetails();

  /**
   * A Telegram Passport element to be saved containing the user's personal details.
   *
   * \param[in] personal_details_ Personal details of the user.
   */
  explicit inputPassportElementPersonalDetails(object_ptr<personalDetails> &&personal_details_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 164791359;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's passport.
 */
class inputPassportElementPassport final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The passport to be saved.
  object_ptr<inputIdentityDocument> passport_;

  /**
   * A Telegram Passport element to be saved containing the user's passport.
   */
  inputPassportElementPassport();

  /**
   * A Telegram Passport element to be saved containing the user's passport.
   *
   * \param[in] passport_ The passport to be saved.
   */
  explicit inputPassportElementPassport(object_ptr<inputIdentityDocument> &&passport_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -497011356;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's driver license.
 */
class inputPassportElementDriverLicense final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The driver license to be saved.
  object_ptr<inputIdentityDocument> driver_license_;

  /**
   * A Telegram Passport element to be saved containing the user's driver license.
   */
  inputPassportElementDriverLicense();

  /**
   * A Telegram Passport element to be saved containing the user's driver license.
   *
   * \param[in] driver_license_ The driver license to be saved.
   */
  explicit inputPassportElementDriverLicense(object_ptr<inputIdentityDocument> &&driver_license_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 304813264;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's identity card.
 */
class inputPassportElementIdentityCard final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The identity card to be saved.
  object_ptr<inputIdentityDocument> identity_card_;

  /**
   * A Telegram Passport element to be saved containing the user's identity card.
   */
  inputPassportElementIdentityCard();

  /**
   * A Telegram Passport element to be saved containing the user's identity card.
   *
   * \param[in] identity_card_ The identity card to be saved.
   */
  explicit inputPassportElementIdentityCard(object_ptr<inputIdentityDocument> &&identity_card_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -9963390;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's internal passport.
 */
class inputPassportElementInternalPassport final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The internal passport to be saved.
  object_ptr<inputIdentityDocument> internal_passport_;

  /**
   * A Telegram Passport element to be saved containing the user's internal passport.
   */
  inputPassportElementInternalPassport();

  /**
   * A Telegram Passport element to be saved containing the user's internal passport.
   *
   * \param[in] internal_passport_ The internal passport to be saved.
   */
  explicit inputPassportElementInternalPassport(object_ptr<inputIdentityDocument> &&internal_passport_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 715360043;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's address.
 */
class inputPassportElementAddress final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The address to be saved.
  object_ptr<address> address_;

  /**
   * A Telegram Passport element to be saved containing the user's address.
   */
  inputPassportElementAddress();

  /**
   * A Telegram Passport element to be saved containing the user's address.
   *
   * \param[in] address_ The address to be saved.
   */
  explicit inputPassportElementAddress(object_ptr<address> &&address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 461630480;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's utility bill.
 */
class inputPassportElementUtilityBill final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The utility bill to be saved.
  object_ptr<inputPersonalDocument> utility_bill_;

  /**
   * A Telegram Passport element to be saved containing the user's utility bill.
   */
  inputPassportElementUtilityBill();

  /**
   * A Telegram Passport element to be saved containing the user's utility bill.
   *
   * \param[in] utility_bill_ The utility bill to be saved.
   */
  explicit inputPassportElementUtilityBill(object_ptr<inputPersonalDocument> &&utility_bill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1389203841;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's bank statement.
 */
class inputPassportElementBankStatement final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The bank statement to be saved.
  object_ptr<inputPersonalDocument> bank_statement_;

  /**
   * A Telegram Passport element to be saved containing the user's bank statement.
   */
  inputPassportElementBankStatement();

  /**
   * A Telegram Passport element to be saved containing the user's bank statement.
   *
   * \param[in] bank_statement_ The bank statement to be saved.
   */
  explicit inputPassportElementBankStatement(object_ptr<inputPersonalDocument> &&bank_statement_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -26585208;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's rental agreement.
 */
class inputPassportElementRentalAgreement final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The rental agreement to be saved.
  object_ptr<inputPersonalDocument> rental_agreement_;

  /**
   * A Telegram Passport element to be saved containing the user's rental agreement.
   */
  inputPassportElementRentalAgreement();

  /**
   * A Telegram Passport element to be saved containing the user's rental agreement.
   *
   * \param[in] rental_agreement_ The rental agreement to be saved.
   */
  explicit inputPassportElementRentalAgreement(object_ptr<inputPersonalDocument> &&rental_agreement_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1736154155;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's passport registration.
 */
class inputPassportElementPassportRegistration final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The passport registration page to be saved.
  object_ptr<inputPersonalDocument> passport_registration_;

  /**
   * A Telegram Passport element to be saved containing the user's passport registration.
   */
  inputPassportElementPassportRegistration();

  /**
   * A Telegram Passport element to be saved containing the user's passport registration.
   *
   * \param[in] passport_registration_ The passport registration page to be saved.
   */
  explicit inputPassportElementPassportRegistration(object_ptr<inputPersonalDocument> &&passport_registration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1314562128;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's temporary registration.
 */
class inputPassportElementTemporaryRegistration final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The temporary registration document to be saved.
  object_ptr<inputPersonalDocument> temporary_registration_;

  /**
   * A Telegram Passport element to be saved containing the user's temporary registration.
   */
  inputPassportElementTemporaryRegistration();

  /**
   * A Telegram Passport element to be saved containing the user's temporary registration.
   *
   * \param[in] temporary_registration_ The temporary registration document to be saved.
   */
  explicit inputPassportElementTemporaryRegistration(object_ptr<inputPersonalDocument> &&temporary_registration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1913238047;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's phone number.
 */
class inputPassportElementPhoneNumber final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The phone number to be saved.
  string phone_number_;

  /**
   * A Telegram Passport element to be saved containing the user's phone number.
   */
  inputPassportElementPhoneNumber();

  /**
   * A Telegram Passport element to be saved containing the user's phone number.
   *
   * \param[in] phone_number_ The phone number to be saved.
   */
  explicit inputPassportElementPhoneNumber(string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1319357497;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A Telegram Passport element to be saved containing the user's email address.
 */
class inputPassportElementEmailAddress final : public InputPassportElement {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The email address to be saved.
  string email_address_;

  /**
   * A Telegram Passport element to be saved containing the user's email address.
   */
  inputPassportElementEmailAddress();

  /**
   * A Telegram Passport element to be saved containing the user's email address.
   *
   * \param[in] email_address_ The email address to be saved.
   */
  explicit inputPassportElementEmailAddress(string const &email_address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -248605659;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputPassportElementErrorSource;

class PassportElementType;

/**
 * Contains the description of an error in a Telegram Passport element; for bots only.
 */
class inputPassportElementError final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of Telegram Passport element that has the error.
  object_ptr<PassportElementType> type_;
  /// Error message.
  string message_;
  /// Error source.
  object_ptr<InputPassportElementErrorSource> source_;

  /**
   * Contains the description of an error in a Telegram Passport element; for bots only.
   */
  inputPassportElementError();

  /**
   * Contains the description of an error in a Telegram Passport element; for bots only.
   *
   * \param[in] type_ Type of Telegram Passport element that has the error.
   * \param[in] message_ Error message.
   * \param[in] source_ Error source.
   */
  inputPassportElementError(object_ptr<PassportElementType> &&type_, string const &message_, object_ptr<InputPassportElementErrorSource> &&source_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 285756898;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * This class is an abstract base class.
 * Contains the description of an error in a Telegram Passport element; for bots only.
 */
class InputPassportElementErrorSource: public Object {
 public:
};

/**
 * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
 */
class inputPassportElementErrorSourceUnspecified final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hash of the entire element.
  bytes element_hash_;

  /**
   * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
   */
  inputPassportElementErrorSourceUnspecified();

  /**
   * The element contains an error in an unspecified place. The error will be considered resolved when new data is added.
   *
   * \param[in] element_hash_ Current hash of the entire element.
   */
  explicit inputPassportElementErrorSourceUnspecified(bytes const &element_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 267230319;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A data field contains an error. The error is considered resolved when the field's value changes.
 */
class inputPassportElementErrorSourceDataField final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Field name.
  string field_name_;
  /// Current data hash.
  bytes data_hash_;

  /**
   * A data field contains an error. The error is considered resolved when the field's value changes.
   */
  inputPassportElementErrorSourceDataField();

  /**
   * A data field contains an error. The error is considered resolved when the field's value changes.
   *
   * \param[in] field_name_ Field name.
   * \param[in] data_hash_ Current data hash.
   */
  inputPassportElementErrorSourceDataField(string const &field_name_, bytes const &data_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -426795002;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The front side of the document contains an error. The error is considered resolved when the file with the front side of the document changes.
 */
class inputPassportElementErrorSourceFrontSide final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hash of the file containing the front side.
  bytes file_hash_;

  /**
   * The front side of the document contains an error. The error is considered resolved when the file with the front side of the document changes.
   */
  inputPassportElementErrorSourceFrontSide();

  /**
   * The front side of the document contains an error. The error is considered resolved when the file with the front side of the document changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the front side.
   */
  explicit inputPassportElementErrorSourceFrontSide(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 588023741;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The reverse side of the document contains an error. The error is considered resolved when the file with the reverse side of the document changes.
 */
class inputPassportElementErrorSourceReverseSide final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hash of the file containing the reverse side.
  bytes file_hash_;

  /**
   * The reverse side of the document contains an error. The error is considered resolved when the file with the reverse side of the document changes.
   */
  inputPassportElementErrorSourceReverseSide();

  /**
   * The reverse side of the document contains an error. The error is considered resolved when the file with the reverse side of the document changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the reverse side.
   */
  explicit inputPassportElementErrorSourceReverseSide(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 413072891;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The selfie contains an error. The error is considered resolved when the file with the selfie changes.
 */
class inputPassportElementErrorSourceSelfie final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hash of the file containing the selfie.
  bytes file_hash_;

  /**
   * The selfie contains an error. The error is considered resolved when the file with the selfie changes.
   */
  inputPassportElementErrorSourceSelfie();

  /**
   * The selfie contains an error. The error is considered resolved when the file with the selfie changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the selfie.
   */
  explicit inputPassportElementErrorSourceSelfie(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -773575528;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * One of the files containing the translation of the document contains an error. The error is considered resolved when the file with the translation changes.
 */
class inputPassportElementErrorSourceTranslationFile final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hash of the file containing the translation.
  bytes file_hash_;

  /**
   * One of the files containing the translation of the document contains an error. The error is considered resolved when the file with the translation changes.
   */
  inputPassportElementErrorSourceTranslationFile();

  /**
   * One of the files containing the translation of the document contains an error. The error is considered resolved when the file with the translation changes.
   *
   * \param[in] file_hash_ Current hash of the file containing the translation.
   */
  explicit inputPassportElementErrorSourceTranslationFile(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 505842299;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The translation of the document contains an error. The error is considered resolved when the list of files changes.
 */
class inputPassportElementErrorSourceTranslationFiles final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hashes of all files with the translation.
  array<bytes> file_hashes_;

  /**
   * The translation of the document contains an error. The error is considered resolved when the list of files changes.
   */
  inputPassportElementErrorSourceTranslationFiles();

  /**
   * The translation of the document contains an error. The error is considered resolved when the list of files changes.
   *
   * \param[in] file_hashes_ Current hashes of all files with the translation.
   */
  explicit inputPassportElementErrorSourceTranslationFiles(array<bytes> &&file_hashes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -527254048;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The file contains an error. The error is considered resolved when the file changes.
 */
class inputPassportElementErrorSourceFile final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hash of the file which has the error.
  bytes file_hash_;

  /**
   * The file contains an error. The error is considered resolved when the file changes.
   */
  inputPassportElementErrorSourceFile();

  /**
   * The file contains an error. The error is considered resolved when the file changes.
   *
   * \param[in] file_hash_ Current hash of the file which has the error.
   */
  explicit inputPassportElementErrorSourceFile(bytes const &file_hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -298492469;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The list of attached files contains an error. The error is considered resolved when the file list changes.
 */
class inputPassportElementErrorSourceFiles final : public InputPassportElementErrorSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Current hashes of all attached files.
  array<bytes> file_hashes_;

  /**
   * The list of attached files contains an error. The error is considered resolved when the file list changes.
   */
  inputPassportElementErrorSourceFiles();

  /**
   * The list of attached files contains an error. The error is considered resolved when the file list changes.
   *
   * \param[in] file_hashes_ Current hashes of all attached files.
   */
  explicit inputPassportElementErrorSourceFiles(array<bytes> &&file_hashes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2008541640;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

/**
 * A personal document to be saved to Telegram Passport.
 */
class inputPersonalDocument final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of files containing the pages of the document.
  array<object_ptr<InputFile>> files_;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<InputFile>> translation_;

  /**
   * A personal document to be saved to Telegram Passport.
   */
  inputPersonalDocument();

  /**
   * A personal document to be saved to Telegram Passport.
   *
   * \param[in] files_ List of files containing the pages of the document.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  inputPersonalDocument(array<object_ptr<InputFile>> &&files_, array<object_ptr<InputFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1676966826;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

class StickerFormat;

class maskPosition;

/**
 * A sticker to be added to a sticker set.
 */
class inputSticker final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// File with the sticker; must fit in a 512x512 square. For WEBP stickers the file must be in WEBP or PNG format, which will be converted to WEBP server-side. See https://core.telegram.org/animated_stickers\#technical-requirements for technical requirements.
  object_ptr<InputFile> sticker_;
  /// Format of the sticker.
  object_ptr<StickerFormat> format_;
  /// String with 1-20 emoji corresponding to the sticker.
  string emojis_;
  /// Position where the mask is placed; pass null if not specified.
  object_ptr<maskPosition> mask_position_;
  /// List of up to 20 keywords with total length up to 64 characters, which can be used to find the sticker.
  array<string> keywords_;

  /**
   * A sticker to be added to a sticker set.
   */
  inputSticker();

  /**
   * A sticker to be added to a sticker set.
   *
   * \param[in] sticker_ File with the sticker; must fit in a 512x512 square. For WEBP stickers the file must be in WEBP or PNG format, which will be converted to WEBP server-side. See https://core.telegram.org/animated_stickers\#technical-requirements for technical requirements.
   * \param[in] format_ Format of the sticker.
   * \param[in] emojis_ String with 1-20 emoji corresponding to the sticker.
   * \param[in] mask_position_ Position where the mask is placed; pass null if not specified.
   * \param[in] keywords_ List of up to 20 keywords with total length up to 64 characters, which can be used to find the sticker.
   */
  inputSticker(object_ptr<InputFile> &&sticker_, object_ptr<StickerFormat> &&format_, string const &emojis_, object_ptr<maskPosition> &&mask_position_, array<string> &&keywords_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1589392402;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputStoryAreaType;

class storyAreaPosition;

/**
 * Describes a clickable rectangle area on a story media to be added.
 */
class inputStoryArea final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Position of the area.
  object_ptr<storyAreaPosition> position_;
  /// Type of the area.
  object_ptr<InputStoryAreaType> type_;

  /**
   * Describes a clickable rectangle area on a story media to be added.
   */
  inputStoryArea();

  /**
   * Describes a clickable rectangle area on a story media to be added.
   *
   * \param[in] position_ Position of the area.
   * \param[in] type_ Type of the area.
   */
  inputStoryArea(object_ptr<storyAreaPosition> &&position_, object_ptr<InputStoryAreaType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 122859135;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ReactionType;

class location;

class locationAddress;

/**
 * This class is an abstract base class.
 * Describes type of clickable area on a story media to be added.
 */
class InputStoryAreaType: public Object {
 public:
};

/**
 * An area pointing to a location.
 */
class inputStoryAreaTypeLocation final : public InputStoryAreaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The location.
  object_ptr<location> location_;
  /// Address of the location; pass null if unknown.
  object_ptr<locationAddress> address_;

  /**
   * An area pointing to a location.
   */
  inputStoryAreaTypeLocation();

  /**
   * An area pointing to a location.
   *
   * \param[in] location_ The location.
   * \param[in] address_ Address of the location; pass null if unknown.
   */
  inputStoryAreaTypeLocation(object_ptr<location> &&location_, object_ptr<locationAddress> &&address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1433714887;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An area pointing to a venue found by the bot getOption(&quot;venue_search_bot_username&quot;).
 */
class inputStoryAreaTypeFoundVenue final : public InputStoryAreaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the inline query, used to found the venue.
  int64 query_id_;
  /// Identifier of the inline query result.
  string result_id_;

  /**
   * An area pointing to a venue found by the bot getOption(&quot;venue_search_bot_username&quot;).
   */
  inputStoryAreaTypeFoundVenue();

  /**
   * An area pointing to a venue found by the bot getOption(&quot;venue_search_bot_username&quot;).
   *
   * \param[in] query_id_ Identifier of the inline query, used to found the venue.
   * \param[in] result_id_ Identifier of the inline query result.
   */
  inputStoryAreaTypeFoundVenue(int64 query_id_, string const &result_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1395809130;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An area pointing to a venue already added to the story.
 */
class inputStoryAreaTypePreviousVenue final : public InputStoryAreaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Provider of the venue.
  string venue_provider_;
  /// Identifier of the venue in the provider database.
  string venue_id_;

  /**
   * An area pointing to a venue already added to the story.
   */
  inputStoryAreaTypePreviousVenue();

  /**
   * An area pointing to a venue already added to the story.
   *
   * \param[in] venue_provider_ Provider of the venue.
   * \param[in] venue_id_ Identifier of the venue in the provider database.
   */
  inputStoryAreaTypePreviousVenue(string const &venue_provider_, string const &venue_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1846693388;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An area pointing to a suggested reaction.
 */
class inputStoryAreaTypeSuggestedReaction final : public InputStoryAreaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Type of the reaction.
  object_ptr<ReactionType> reaction_type_;
  /// True, if reaction has a dark background.
  bool is_dark_;
  /// True, if reaction corner is flipped.
  bool is_flipped_;

  /**
   * An area pointing to a suggested reaction.
   */
  inputStoryAreaTypeSuggestedReaction();

  /**
   * An area pointing to a suggested reaction.
   *
   * \param[in] reaction_type_ Type of the reaction.
   * \param[in] is_dark_ True, if reaction has a dark background.
   * \param[in] is_flipped_ True, if reaction corner is flipped.
   */
  inputStoryAreaTypeSuggestedReaction(object_ptr<ReactionType> &&reaction_type_, bool is_dark_, bool is_flipped_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2101826003;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An area pointing to a message.
 */
class inputStoryAreaTypeMessage final : public InputStoryAreaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Identifier of the chat with the message. Currently, the chat must be a supergroup or a channel chat.
  int53 chat_id_;
  /// Identifier of the message. Use messageProperties.can_be_shared_in_story to check whether the message is suitable.
  int53 message_id_;

  /**
   * An area pointing to a message.
   */
  inputStoryAreaTypeMessage();

  /**
   * An area pointing to a message.
   *
   * \param[in] chat_id_ Identifier of the chat with the message. Currently, the chat must be a supergroup or a channel chat.
   * \param[in] message_id_ Identifier of the message. Use messageProperties.can_be_shared_in_story to check whether the message is suitable.
   */
  inputStoryAreaTypeMessage(int53 chat_id_, int53 message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -266607529;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An area pointing to a HTTP or tg:// link.
 */
class inputStoryAreaTypeLink final : public InputStoryAreaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// HTTP or tg:// URL to be opened when the area is clicked.
  string url_;

  /**
   * An area pointing to a HTTP or tg:// link.
   */
  inputStoryAreaTypeLink();

  /**
   * An area pointing to a HTTP or tg:// link.
   *
   * \param[in] url_ HTTP or tg:// URL to be opened when the area is clicked.
   */
  explicit inputStoryAreaTypeLink(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1408441160;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * An area with information about weather.
 */
class inputStoryAreaTypeWeather final : public InputStoryAreaType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Temperature, in degree Celsius.
  double temperature_;
  /// Emoji representing the weather.
  string emoji_;
  /// A color of the area background in the ARGB format.
  int32 background_color_;

  /**
   * An area with information about weather.
   */
  inputStoryAreaTypeWeather();

  /**
   * An area with information about weather.
   *
   * \param[in] temperature_ Temperature, in degree Celsius.
   * \param[in] emoji_ Emoji representing the weather.
   * \param[in] background_color_ A color of the area background in the ARGB format.
   */
  inputStoryAreaTypeWeather(double temperature_, string const &emoji_, int32 background_color_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1212686691;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class inputStoryArea;

/**
 * Contains a list of story areas to be added.
 */
class inputStoryAreas final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// List of input story areas. Currently, a story can have up to 10 inputStoryAreaTypeLocation, inputStoryAreaTypeFoundVenue, and inputStoryAreaTypePreviousVenue areas, up to getOption(&quot;story_suggested_reaction_area_count_max&quot;) inputStoryAreaTypeSuggestedReaction areas, up to 1 inputStoryAreaTypeMessage area, up to getOption(&quot;story_link_area_count_max&quot;) inputStoryAreaTypeLink areas if the current user is a Telegram Premium user, and up to 3 inputStoryAreaTypeWeather areas.
  array<object_ptr<inputStoryArea>> areas_;

  /**
   * Contains a list of story areas to be added.
   */
  inputStoryAreas();

  /**
   * Contains a list of story areas to be added.
   *
   * \param[in] areas_ List of input story areas. Currently, a story can have up to 10 inputStoryAreaTypeLocation, inputStoryAreaTypeFoundVenue, and inputStoryAreaTypePreviousVenue areas, up to getOption(&quot;story_suggested_reaction_area_count_max&quot;) inputStoryAreaTypeSuggestedReaction areas, up to 1 inputStoryAreaTypeMessage area, up to getOption(&quot;story_link_area_count_max&quot;) inputStoryAreaTypeLink areas if the current user is a Telegram Premium user, and up to 3 inputStoryAreaTypeWeather areas.
   */
  explicit inputStoryAreas(array<object_ptr<inputStoryArea>> &&areas_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -883247088;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

/**
 * This class is an abstract base class.
 * The content of a story to send.
 */
class InputStoryContent: public Object {
 public:
};

/**
 * A photo story.
 */
class inputStoryContentPhoto final : public InputStoryContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Photo to send. The photo must be at most 10 MB in size. The photo size must be 1080x1920.
  object_ptr<InputFile> photo_;
  /// File identifiers of the stickers added to the photo, if applicable.
  array<int32> added_sticker_file_ids_;

  /**
   * A photo story.
   */
  inputStoryContentPhoto();

  /**
   * A photo story.
   *
   * \param[in] photo_ Photo to send. The photo must be at most 10 MB in size. The photo size must be 1080x1920.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the photo, if applicable.
   */
  inputStoryContentPhoto(object_ptr<InputFile> &&photo_, array<int32> &&added_sticker_file_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -309196727;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * A video story.
 */
class inputStoryContentVideo final : public InputStoryContent {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Video to be sent. The video size must be 720x1280. The video must be streamable and stored in MPEG4 format, after encoding with H.265 codec and key frames added each second.
  object_ptr<InputFile> video_;
  /// File identifiers of the stickers added to the video, if applicable.
  array<int32> added_sticker_file_ids_;
  /// Precise duration of the video, in seconds; 0-60.
  double duration_;
  /// Timestamp of the frame, which will be used as video thumbnail.
  double cover_frame_timestamp_;
  /// True, if the video has no sound.
  bool is_animation_;

  /**
   * A video story.
   */
  inputStoryContentVideo();

  /**
   * A video story.
   *
   * \param[in] video_ Video to be sent. The video size must be 720x1280. The video must be streamable and stored in MPEG4 format, after encoding with H.265 codec and key frames added each second.
   * \param[in] added_sticker_file_ids_ File identifiers of the stickers added to the video, if applicable.
   * \param[in] duration_ Precise duration of the video, in seconds; 0-60.
   * \param[in] cover_frame_timestamp_ Timestamp of the frame, which will be used as video thumbnail.
   * \param[in] is_animation_ True, if the video has no sound.
   */
  inputStoryContentVideo(object_ptr<InputFile> &&video_, array<int32> &&added_sticker_file_ids_, double duration_, double cover_frame_timestamp_, bool is_animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 3809243;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class formattedText;

/**
 * Describes manually chosen quote from another message.
 */
class inputTextQuote final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Text of the quote; 0-getOption(&quot;message_reply_quote_length_max&quot;) characters. Only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities are allowed to be kept and must be kept in the quote.
  object_ptr<formattedText> text_;
  /// Quote position in the original message in UTF-16 code units.
  int32 position_;

  /**
   * Describes manually chosen quote from another message.
   */
  inputTextQuote();

  /**
   * Describes manually chosen quote from another message.
   *
   * \param[in] text_ Text of the quote; 0-getOption(&quot;message_reply_quote_length_max&quot;) characters. Only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities are allowed to be kept and must be kept in the quote.
   * \param[in] position_ Quote position in the original message in UTF-16 code units.
   */
  inputTextQuote(object_ptr<formattedText> &&text_, int32 position_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1219859172;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class InputFile;

/**
 * A thumbnail to be sent along with a file; must be in JPEG or WEBP format for stickers, and less than 200 KB in size.
 */
class inputThumbnail final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Thumbnail file to send. Sending thumbnails by file_id is currently not supported.
  object_ptr<InputFile> thumbnail_;
  /// Thumbnail width, usually shouldn't exceed 320. Use 0 if unknown.
  int32 width_;
  /// Thumbnail height, usually shouldn't exceed 320. Use 0 if unknown.
  int32 height_;

  /**
   * A thumbnail to be sent along with a file; must be in JPEG or WEBP format for stickers, and less than 200 KB in size.
   */
  inputThumbnail();

  /**
   * A thumbnail to be sent along with a file; must be in JPEG or WEBP format for stickers, and less than 200 KB in size.
   *
   * \param[in] thumbnail_ Thumbnail file to send. Sending thumbnails by file_id is currently not supported.
   * \param[in] width_ Thumbnail width, usually shouldn't exceed 320. Use 0 if unknown.
   * \param[in] height_ Thumbnail height, usually shouldn't exceed 320. Use 0 if unknown.
   */
  inputThumbnail(object_ptr<InputFile> &&thumbnail_, int32 width_, int32 height_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1582387236;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

class ProxyType;

class TargetChat;

class WebAppOpenMode;

class chatAdministratorRights;

class formattedText;

/**
 * This class is an abstract base class.
 * Describes an internal https://t.me or tg: link, which must be processed by the application in a special way.
 */
class InternalLinkType: public Object {
 public:
};

/**
 * The link is a link to the Devices section of the application. Use getActiveSessions to get the list of active sessions and show them to the user.
 */
class internalLinkTypeActiveSessions final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the Devices section of the application. Use getActiveSessions to get the list of active sessions and show them to the user.
   */
  internalLinkTypeActiveSessions();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1886108589;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to an attachment menu bot to be opened in the specified or a chosen chat. Process given target_chat to open the chat. Then, call searchPublicChat with the given bot username, check that the user is a bot and can be added to attachment menu. Then, use getAttachmentMenuBot to receive information about the bot. If the bot isn't added to attachment menu, then show a disclaimer about Mini Apps being third-party applications, ask the user to accept their Terms of service and confirm adding the bot to side and attachment menu. If the user accept the terms and confirms adding, then use toggleBotIsAddedToAttachmentMenu to add the bot. If the attachment menu bot can't be used in the opened chat, show an error to the user. If the bot is added to attachment menu and can be used in the chat, then use openWebApp with the given URL.
 */
class internalLinkTypeAttachmentMenuBot final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Target chat to be opened.
  object_ptr<TargetChat> target_chat_;
  /// Username of the bot.
  string bot_username_;
  /// URL to be passed to openWebApp.
  string url_;

  /**
   * The link is a link to an attachment menu bot to be opened in the specified or a chosen chat. Process given target_chat to open the chat. Then, call searchPublicChat with the given bot username, check that the user is a bot and can be added to attachment menu. Then, use getAttachmentMenuBot to receive information about the bot. If the bot isn't added to attachment menu, then show a disclaimer about Mini Apps being third-party applications, ask the user to accept their Terms of service and confirm adding the bot to side and attachment menu. If the user accept the terms and confirms adding, then use toggleBotIsAddedToAttachmentMenu to add the bot. If the attachment menu bot can't be used in the opened chat, show an error to the user. If the bot is added to attachment menu and can be used in the chat, then use openWebApp with the given URL.
   */
  internalLinkTypeAttachmentMenuBot();

  /**
   * The link is a link to an attachment menu bot to be opened in the specified or a chosen chat. Process given target_chat to open the chat. Then, call searchPublicChat with the given bot username, check that the user is a bot and can be added to attachment menu. Then, use getAttachmentMenuBot to receive information about the bot. If the bot isn't added to attachment menu, then show a disclaimer about Mini Apps being third-party applications, ask the user to accept their Terms of service and confirm adding the bot to side and attachment menu. If the user accept the terms and confirms adding, then use toggleBotIsAddedToAttachmentMenu to add the bot. If the attachment menu bot can't be used in the opened chat, show an error to the user. If the bot is added to attachment menu and can be used in the chat, then use openWebApp with the given URL.
   *
   * \param[in] target_chat_ Target chat to be opened.
   * \param[in] bot_username_ Username of the bot.
   * \param[in] url_ URL to be passed to openWebApp.
   */
  internalLinkTypeAttachmentMenuBot(object_ptr<TargetChat> &&target_chat_, string const &bot_username_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1682719269;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link contains an authentication code. Call checkAuthenticationCode with the code if the current authorization state is authorizationStateWaitCode.
 */
class internalLinkTypeAuthenticationCode final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The authentication code.
  string code_;

  /**
   * The link contains an authentication code. Call checkAuthenticationCode with the code if the current authorization state is authorizationStateWaitCode.
   */
  internalLinkTypeAuthenticationCode();

  /**
   * The link contains an authentication code. Call checkAuthenticationCode with the code if the current authorization state is authorizationStateWaitCode.
   *
   * \param[in] code_ The authentication code.
   */
  explicit internalLinkTypeAuthenticationCode(string const &code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -209235982;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a background. Call searchBackground with the given background name to process the link. If background is found and the user wants to apply it, then call setDefaultBackground.
 */
class internalLinkTypeBackground final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the background.
  string background_name_;

  /**
   * The link is a link to a background. Call searchBackground with the given background name to process the link. If background is found and the user wants to apply it, then call setDefaultBackground.
   */
  internalLinkTypeBackground();

  /**
   * The link is a link to a background. Call searchBackground with the given background name to process the link. If background is found and the user wants to apply it, then call setDefaultBackground.
   *
   * \param[in] background_name_ Name of the background.
   */
  explicit internalLinkTypeBackground(string const &background_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 185411848;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a Telegram bot, which is expected to be added to a channel chat as an administrator. Call searchPublicChat with the given bot username and check that the user is a bot, ask the current user to select a channel chat to add the bot to as an administrator. Then, call getChatMember to receive the current bot rights in the chat and if the bot already is an administrator, check that the current user can edit its administrator rights and combine received rights with the requested administrator rights. Then, show confirmation box to the user, and call setChatMemberStatus with the chosen chat and confirmed rights.
 */
class internalLinkTypeBotAddToChannel final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the bot.
  string bot_username_;
  /// Expected administrator rights for the bot.
  object_ptr<chatAdministratorRights> administrator_rights_;

  /**
   * The link is a link to a Telegram bot, which is expected to be added to a channel chat as an administrator. Call searchPublicChat with the given bot username and check that the user is a bot, ask the current user to select a channel chat to add the bot to as an administrator. Then, call getChatMember to receive the current bot rights in the chat and if the bot already is an administrator, check that the current user can edit its administrator rights and combine received rights with the requested administrator rights. Then, show confirmation box to the user, and call setChatMemberStatus with the chosen chat and confirmed rights.
   */
  internalLinkTypeBotAddToChannel();

  /**
   * The link is a link to a Telegram bot, which is expected to be added to a channel chat as an administrator. Call searchPublicChat with the given bot username and check that the user is a bot, ask the current user to select a channel chat to add the bot to as an administrator. Then, call getChatMember to receive the current bot rights in the chat and if the bot already is an administrator, check that the current user can edit its administrator rights and combine received rights with the requested administrator rights. Then, show confirmation box to the user, and call setChatMemberStatus with the chosen chat and confirmed rights.
   *
   * \param[in] bot_username_ Username of the bot.
   * \param[in] administrator_rights_ Expected administrator rights for the bot.
   */
  internalLinkTypeBotAddToChannel(string const &bot_username_, object_ptr<chatAdministratorRights> &&administrator_rights_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1401602752;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a chat with a Telegram bot. Call searchPublicChat with the given bot username, check that the user is a bot, show START button in the chat with the bot, and then call sendBotStartMessage with the given start parameter after the button is pressed.
 */
class internalLinkTypeBotStart final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the bot.
  string bot_username_;
  /// The parameter to be passed to sendBotStartMessage.
  string start_parameter_;
  /// True, if sendBotStartMessage must be called automatically without showing the START button.
  bool autostart_;

  /**
   * The link is a link to a chat with a Telegram bot. Call searchPublicChat with the given bot username, check that the user is a bot, show START button in the chat with the bot, and then call sendBotStartMessage with the given start parameter after the button is pressed.
   */
  internalLinkTypeBotStart();

  /**
   * The link is a link to a chat with a Telegram bot. Call searchPublicChat with the given bot username, check that the user is a bot, show START button in the chat with the bot, and then call sendBotStartMessage with the given start parameter after the button is pressed.
   *
   * \param[in] bot_username_ Username of the bot.
   * \param[in] start_parameter_ The parameter to be passed to sendBotStartMessage.
   * \param[in] autostart_ True, if sendBotStartMessage must be called automatically without showing the START button.
   */
  internalLinkTypeBotStart(string const &bot_username_, string const &start_parameter_, bool autostart_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1066950637;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a Telegram bot, which is expected to be added to a group chat. Call searchPublicChat with the given bot username, check that the user is a bot and can be added to groups, ask the current user to select a basic group or a supergroup chat to add the bot to, taking into account that bots can be added to a public supergroup only by administrators of the supergroup. If administrator rights are provided by the link, call getChatMember to receive the current bot rights in the chat and if the bot already is an administrator, check that the current user can edit its administrator rights, combine received rights with the requested administrator rights, show confirmation box to the user, and call setChatMemberStatus with the chosen chat and confirmed administrator rights. Before call to setChatMemberStatus it may be required to upgrade the chosen basic group chat to a supergroup chat. Then, if start_parameter isn't empty, call sendBotStartMessage with the given start parameter and the chosen chat; otherwise, just send /start message with bot's username added to the chat.
 */
class internalLinkTypeBotStartInGroup final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the bot.
  string bot_username_;
  /// The parameter to be passed to sendBotStartMessage.
  string start_parameter_;
  /// Expected administrator rights for the bot; may be null.
  object_ptr<chatAdministratorRights> administrator_rights_;

  /**
   * The link is a link to a Telegram bot, which is expected to be added to a group chat. Call searchPublicChat with the given bot username, check that the user is a bot and can be added to groups, ask the current user to select a basic group or a supergroup chat to add the bot to, taking into account that bots can be added to a public supergroup only by administrators of the supergroup. If administrator rights are provided by the link, call getChatMember to receive the current bot rights in the chat and if the bot already is an administrator, check that the current user can edit its administrator rights, combine received rights with the requested administrator rights, show confirmation box to the user, and call setChatMemberStatus with the chosen chat and confirmed administrator rights. Before call to setChatMemberStatus it may be required to upgrade the chosen basic group chat to a supergroup chat. Then, if start_parameter isn't empty, call sendBotStartMessage with the given start parameter and the chosen chat; otherwise, just send /start message with bot's username added to the chat.
   */
  internalLinkTypeBotStartInGroup();

  /**
   * The link is a link to a Telegram bot, which is expected to be added to a group chat. Call searchPublicChat with the given bot username, check that the user is a bot and can be added to groups, ask the current user to select a basic group or a supergroup chat to add the bot to, taking into account that bots can be added to a public supergroup only by administrators of the supergroup. If administrator rights are provided by the link, call getChatMember to receive the current bot rights in the chat and if the bot already is an administrator, check that the current user can edit its administrator rights, combine received rights with the requested administrator rights, show confirmation box to the user, and call setChatMemberStatus with the chosen chat and confirmed administrator rights. Before call to setChatMemberStatus it may be required to upgrade the chosen basic group chat to a supergroup chat. Then, if start_parameter isn't empty, call sendBotStartMessage with the given start parameter and the chosen chat; otherwise, just send /start message with bot's username added to the chat.
   *
   * \param[in] bot_username_ Username of the bot.
   * \param[in] start_parameter_ The parameter to be passed to sendBotStartMessage.
   * \param[in] administrator_rights_ Expected administrator rights for the bot; may be null.
   */
  internalLinkTypeBotStartInGroup(string const &bot_username_, string const &start_parameter_, object_ptr<chatAdministratorRights> &&administrator_rights_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -905081650;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a business chat. Use getBusinessChatLinkInfo with the provided link name to get information about the link, then open received private chat and replace chat draft with the provided text.
 */
class internalLinkTypeBusinessChat final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the link.
  string link_name_;

  /**
   * The link is a link to a business chat. Use getBusinessChatLinkInfo with the provided link name to get information about the link, then open received private chat and replace chat draft with the provided text.
   */
  internalLinkTypeBusinessChat();

  /**
   * The link is a link to a business chat. Use getBusinessChatLinkInfo with the provided link name to get information about the link, then open received private chat and replace chat draft with the provided text.
   *
   * \param[in] link_name_ Name of the link.
   */
  explicit internalLinkTypeBusinessChat(string const &link_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1606751785;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the Telegram Star purchase section of the application.
 */
class internalLinkTypeBuyStars final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The number of Telegram Stars that must be owned by the user.
  int53 star_count_;
  /// Purpose of Telegram Star purchase. Arbitrary string specified by the server, for example, &quot;subs&quot; if the Telegram Stars are required to extend channel subscriptions.
  string purpose_;

  /**
   * The link is a link to the Telegram Star purchase section of the application.
   */
  internalLinkTypeBuyStars();

  /**
   * The link is a link to the Telegram Star purchase section of the application.
   *
   * \param[in] star_count_ The number of Telegram Stars that must be owned by the user.
   * \param[in] purpose_ Purpose of Telegram Star purchase. Arbitrary string specified by the server, for example, &quot;subs&quot; if the Telegram Stars are required to extend channel subscriptions.
   */
  internalLinkTypeBuyStars(int53 star_count_, string const &purpose_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1454587065;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the change phone number section of the application.
 */
class internalLinkTypeChangePhoneNumber final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the change phone number section of the application.
   */
  internalLinkTypeChangePhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -265856255;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is an affiliate program link. Call searchChatAffiliateProgram with the given username and referrer to process the link.
 */
class internalLinkTypeChatAffiliateProgram final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username to be passed to searchChatAffiliateProgram.
  string username_;
  /// Referrer to be passed to searchChatAffiliateProgram.
  string referrer_;

  /**
   * The link is an affiliate program link. Call searchChatAffiliateProgram with the given username and referrer to process the link.
   */
  internalLinkTypeChatAffiliateProgram();

  /**
   * The link is an affiliate program link. Call searchChatAffiliateProgram with the given username and referrer to process the link.
   *
   * \param[in] username_ Username to be passed to searchChatAffiliateProgram.
   * \param[in] referrer_ Referrer to be passed to searchChatAffiliateProgram.
   */
  internalLinkTypeChatAffiliateProgram(string const &username_, string const &referrer_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 632049700;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to boost a Telegram chat. Call getChatBoostLinkInfo with the given URL to process the link. If the chat is found, then call getChatBoostStatus and getAvailableChatBoostSlots to get the current boost status and check whether the chat can be boosted. If the user wants to boost the chat and the chat can be boosted, then call boostChat.
 */
class internalLinkTypeChatBoost final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// URL to be passed to getChatBoostLinkInfo.
  string url_;

  /**
   * The link is a link to boost a Telegram chat. Call getChatBoostLinkInfo with the given URL to process the link. If the chat is found, then call getChatBoostStatus and getAvailableChatBoostSlots to get the current boost status and check whether the chat can be boosted. If the user wants to boost the chat and the chat can be boosted, then call boostChat.
   */
  internalLinkTypeChatBoost();

  /**
   * The link is a link to boost a Telegram chat. Call getChatBoostLinkInfo with the given URL to process the link. If the chat is found, then call getChatBoostStatus and getAvailableChatBoostSlots to get the current boost status and check whether the chat can be boosted. If the user wants to boost the chat and the chat can be boosted, then call boostChat.
   *
   * \param[in] url_ URL to be passed to getChatBoostLinkInfo.
   */
  explicit internalLinkTypeChatBoost(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -716571328;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is an invite link to a chat folder. Call checkChatFolderInviteLink with the given invite link to process the link. If the link is valid and the user wants to join the chat folder, then call addChatFolderByInviteLink.
 */
class internalLinkTypeChatFolderInvite final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Internal representation of the invite link.
  string invite_link_;

  /**
   * The link is an invite link to a chat folder. Call checkChatFolderInviteLink with the given invite link to process the link. If the link is valid and the user wants to join the chat folder, then call addChatFolderByInviteLink.
   */
  internalLinkTypeChatFolderInvite();

  /**
   * The link is an invite link to a chat folder. Call checkChatFolderInviteLink with the given invite link to process the link. If the link is valid and the user wants to join the chat folder, then call addChatFolderByInviteLink.
   *
   * \param[in] invite_link_ Internal representation of the invite link.
   */
  explicit internalLinkTypeChatFolderInvite(string const &invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1984804546;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the folder section of the application settings.
 */
class internalLinkTypeChatFolderSettings final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the folder section of the application settings.
   */
  internalLinkTypeChatFolderSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1073805988;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a chat invite link. Call checkChatInviteLink with the given invite link to process the link. If the link is valid and the user wants to join the chat, then call joinChatByInviteLink.
 */
class internalLinkTypeChatInvite final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Internal representation of the invite link.
  string invite_link_;

  /**
   * The link is a chat invite link. Call checkChatInviteLink with the given invite link to process the link. If the link is valid and the user wants to join the chat, then call joinChatByInviteLink.
   */
  internalLinkTypeChatInvite();

  /**
   * The link is a chat invite link. Call checkChatInviteLink with the given invite link to process the link. If the link is valid and the user wants to join the chat, then call joinChatByInviteLink.
   *
   * \param[in] invite_link_ Internal representation of the invite link.
   */
  explicit internalLinkTypeChatInvite(string const &invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 428621017;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the default message auto-delete timer settings section of the application settings.
 */
class internalLinkTypeDefaultMessageAutoDeleteTimerSettings final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the default message auto-delete timer settings section of the application settings.
   */
  internalLinkTypeDefaultMessageAutoDeleteTimerSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 732625201;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the edit profile section of the application settings.
 */
class internalLinkTypeEditProfileSettings final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the edit profile section of the application settings.
   */
  internalLinkTypeEditProfileSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1022472090;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a game. Call searchPublicChat with the given bot username, check that the user is a bot, ask the current user to select a chat to send the game, and then call sendMessage with inputMessageGame.
 */
class internalLinkTypeGame final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the bot that owns the game.
  string bot_username_;
  /// Short name of the game.
  string game_short_name_;

  /**
   * The link is a link to a game. Call searchPublicChat with the given bot username, check that the user is a bot, ask the current user to select a chat to send the game, and then call sendMessage with inputMessageGame.
   */
  internalLinkTypeGame();

  /**
   * The link is a link to a game. Call searchPublicChat with the given bot username, check that the user is a bot, ask the current user to select a chat to send the game, and then call sendMessage with inputMessageGame.
   *
   * \param[in] bot_username_ Username of the bot that owns the game.
   * \param[in] game_short_name_ Short name of the game.
   */
  internalLinkTypeGame(string const &bot_username_, string const &game_short_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -260788787;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link must be opened in an Instant View. Call getWebPageInstantView with the given URL to process the link. If Instant View is found, then show it, otherwise, open the fallback URL in an external browser.
 */
class internalLinkTypeInstantView final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// URL to be passed to getWebPageInstantView.
  string url_;
  /// An URL to open if getWebPageInstantView fails.
  string fallback_url_;

  /**
   * The link must be opened in an Instant View. Call getWebPageInstantView with the given URL to process the link. If Instant View is found, then show it, otherwise, open the fallback URL in an external browser.
   */
  internalLinkTypeInstantView();

  /**
   * The link must be opened in an Instant View. Call getWebPageInstantView with the given URL to process the link. If Instant View is found, then show it, otherwise, open the fallback URL in an external browser.
   *
   * \param[in] url_ URL to be passed to getWebPageInstantView.
   * \param[in] fallback_url_ An URL to open if getWebPageInstantView fails.
   */
  internalLinkTypeInstantView(string const &url_, string const &fallback_url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1776607039;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to an invoice. Call getPaymentForm with the given invoice name to process the link.
 */
class internalLinkTypeInvoice final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the invoice.
  string invoice_name_;

  /**
   * The link is a link to an invoice. Call getPaymentForm with the given invoice name to process the link.
   */
  internalLinkTypeInvoice();

  /**
   * The link is a link to an invoice. Call getPaymentForm with the given invoice name to process the link.
   *
   * \param[in] invoice_name_ Name of the invoice.
   */
  explicit internalLinkTypeInvoice(string const &invoice_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -213094996;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a language pack. Call getLanguagePackInfo with the given language pack identifier to process the link. If the language pack is found and the user wants to apply it, then call setOption for the option &quot;language_pack_id&quot;.
 */
class internalLinkTypeLanguagePack final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Language pack identifier.
  string language_pack_id_;

  /**
   * The link is a link to a language pack. Call getLanguagePackInfo with the given language pack identifier to process the link. If the language pack is found and the user wants to apply it, then call setOption for the option &quot;language_pack_id&quot;.
   */
  internalLinkTypeLanguagePack();

  /**
   * The link is a link to a language pack. Call getLanguagePackInfo with the given language pack identifier to process the link. If the language pack is found and the user wants to apply it, then call setOption for the option &quot;language_pack_id&quot;.
   *
   * \param[in] language_pack_id_ Language pack identifier.
   */
  explicit internalLinkTypeLanguagePack(string const &language_pack_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1450766996;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the language section of the application settings.
 */
class internalLinkTypeLanguageSettings final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the language section of the application settings.
   */
  internalLinkTypeLanguageSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1340479770;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the main Web App of a bot. Call searchPublicChat with the given bot username, check that the user is a bot and has the main Web App. If the bot can be added to attachment menu, then use getAttachmentMenuBot to receive information about the bot, then if the bot isn't added to side menu, show a disclaimer about Mini Apps being third-party applications, ask the user to accept their Terms of service and confirm adding the bot to side and attachment menu, then if the user accepts the terms and confirms adding, use toggleBotIsAddedToAttachmentMenu to add the bot. Then, use getMainWebApp with the given start parameter and mode and open the returned URL as a Web App.
 */
class internalLinkTypeMainWebApp final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the bot.
  string bot_username_;
  /// Start parameter to be passed to getMainWebApp.
  string start_parameter_;
  /// The mode to be passed to getMainWebApp.
  object_ptr<WebAppOpenMode> mode_;

  /**
   * The link is a link to the main Web App of a bot. Call searchPublicChat with the given bot username, check that the user is a bot and has the main Web App. If the bot can be added to attachment menu, then use getAttachmentMenuBot to receive information about the bot, then if the bot isn't added to side menu, show a disclaimer about Mini Apps being third-party applications, ask the user to accept their Terms of service and confirm adding the bot to side and attachment menu, then if the user accepts the terms and confirms adding, use toggleBotIsAddedToAttachmentMenu to add the bot. Then, use getMainWebApp with the given start parameter and mode and open the returned URL as a Web App.
   */
  internalLinkTypeMainWebApp();

  /**
   * The link is a link to the main Web App of a bot. Call searchPublicChat with the given bot username, check that the user is a bot and has the main Web App. If the bot can be added to attachment menu, then use getAttachmentMenuBot to receive information about the bot, then if the bot isn't added to side menu, show a disclaimer about Mini Apps being third-party applications, ask the user to accept their Terms of service and confirm adding the bot to side and attachment menu, then if the user accepts the terms and confirms adding, use toggleBotIsAddedToAttachmentMenu to add the bot. Then, use getMainWebApp with the given start parameter and mode and open the returned URL as a Web App.
   *
   * \param[in] bot_username_ Username of the bot.
   * \param[in] start_parameter_ Start parameter to be passed to getMainWebApp.
   * \param[in] mode_ The mode to be passed to getMainWebApp.
   */
  internalLinkTypeMainWebApp(string const &bot_username_, string const &start_parameter_, object_ptr<WebAppOpenMode> &&mode_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1574925033;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a Telegram message or a forum topic. Call getMessageLinkInfo with the given URL to process the link, and then open received forum topic or chat and show the message there.
 */
class internalLinkTypeMessage final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// URL to be passed to getMessageLinkInfo.
  string url_;

  /**
   * The link is a link to a Telegram message or a forum topic. Call getMessageLinkInfo with the given URL to process the link, and then open received forum topic or chat and show the message there.
   */
  internalLinkTypeMessage();

  /**
   * The link is a link to a Telegram message or a forum topic. Call getMessageLinkInfo with the given URL to process the link, and then open received forum topic or chat and show the message there.
   *
   * \param[in] url_ URL to be passed to getMessageLinkInfo.
   */
  explicit internalLinkTypeMessage(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 978541650;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link contains a message draft text. A share screen needs to be shown to the user, then the chosen chat must be opened and the text is added to the input field.
 */
class internalLinkTypeMessageDraft final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Message draft text.
  object_ptr<formattedText> text_;
  /// True, if the first line of the text contains a link. If true, the input field needs to be focused and the text after the link must be selected.
  bool contains_link_;

  /**
   * The link contains a message draft text. A share screen needs to be shown to the user, then the chosen chat must be opened and the text is added to the input field.
   */
  internalLinkTypeMessageDraft();

  /**
   * The link contains a message draft text. A share screen needs to be shown to the user, then the chosen chat must be opened and the text is added to the input field.
   *
   * \param[in] text_ Message draft text.
   * \param[in] contains_link_ True, if the first line of the text contains a link. If true, the input field needs to be focused and the text after the link must be selected.
   */
  internalLinkTypeMessageDraft(object_ptr<formattedText> &&text_, bool contains_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 661633749;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link contains a request of Telegram passport data. Call getPassportAuthorizationForm with the given parameters to process the link if the link was received from outside of the application; otherwise, ignore it.
 */
class internalLinkTypePassportDataRequest final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// User identifier of the service's bot; the corresponding user may be unknown yet.
  int53 bot_user_id_;
  /// Telegram Passport element types requested by the service.
  string scope_;
  /// Service's public key.
  string public_key_;
  /// Unique request identifier provided by the service.
  string nonce_;
  /// An HTTP URL to open once the request is finished, canceled, or failed with the parameters tg_passport=success, tg_passport=cancel, or tg_passport=error&amp;error=... respectively. If empty, then onActivityResult method must be used to return response on Android, or the link tgbot{bot_user_id}://passport/success or tgbot{bot_user_id}://passport/cancel must be opened otherwise.
  string callback_url_;

  /**
   * The link contains a request of Telegram passport data. Call getPassportAuthorizationForm with the given parameters to process the link if the link was received from outside of the application; otherwise, ignore it.
   */
  internalLinkTypePassportDataRequest();

  /**
   * The link contains a request of Telegram passport data. Call getPassportAuthorizationForm with the given parameters to process the link if the link was received from outside of the application; otherwise, ignore it.
   *
   * \param[in] bot_user_id_ User identifier of the service's bot; the corresponding user may be unknown yet.
   * \param[in] scope_ Telegram Passport element types requested by the service.
   * \param[in] public_key_ Service's public key.
   * \param[in] nonce_ Unique request identifier provided by the service.
   * \param[in] callback_url_ An HTTP URL to open once the request is finished, canceled, or failed with the parameters tg_passport=success, tg_passport=cancel, or tg_passport=error&amp;error=... respectively. If empty, then onActivityResult method must be used to return response on Android, or the link tgbot{bot_user_id}://passport/success or tgbot{bot_user_id}://passport/cancel must be opened otherwise.
   */
  internalLinkTypePassportDataRequest(int53 bot_user_id_, string const &scope_, string const &public_key_, string const &nonce_, string const &callback_url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -988819839;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link can be used to confirm ownership of a phone number to prevent account deletion. Call sendPhoneNumberCode with the given phone number and with phoneNumberCodeTypeConfirmOwnership with the given hash to process the link. If succeeded, call checkPhoneNumberCode to check entered by the user code, or resendPhoneNumberCode to resend it.
 */
class internalLinkTypePhoneNumberConfirmation final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Hash value from the link.
  string hash_;
  /// Phone number value from the link.
  string phone_number_;

  /**
   * The link can be used to confirm ownership of a phone number to prevent account deletion. Call sendPhoneNumberCode with the given phone number and with phoneNumberCodeTypeConfirmOwnership with the given hash to process the link. If succeeded, call checkPhoneNumberCode to check entered by the user code, or resendPhoneNumberCode to resend it.
   */
  internalLinkTypePhoneNumberConfirmation();

  /**
   * The link can be used to confirm ownership of a phone number to prevent account deletion. Call sendPhoneNumberCode with the given phone number and with phoneNumberCodeTypeConfirmOwnership with the given hash to process the link. If succeeded, call checkPhoneNumberCode to check entered by the user code, or resendPhoneNumberCode to resend it.
   *
   * \param[in] hash_ Hash value from the link.
   * \param[in] phone_number_ Phone number value from the link.
   */
  internalLinkTypePhoneNumberConfirmation(string const &hash_, string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1757375254;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the Premium features screen of the application from which the user can subscribe to Telegram Premium. Call getPremiumFeatures with the given referrer to process the link.
 */
class internalLinkTypePremiumFeatures final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Referrer specified in the link.
  string referrer_;

  /**
   * The link is a link to the Premium features screen of the application from which the user can subscribe to Telegram Premium. Call getPremiumFeatures with the given referrer to process the link.
   */
  internalLinkTypePremiumFeatures();

  /**
   * The link is a link to the Premium features screen of the application from which the user can subscribe to Telegram Premium. Call getPremiumFeatures with the given referrer to process the link.
   *
   * \param[in] referrer_ Referrer specified in the link.
   */
  explicit internalLinkTypePremiumFeatures(string const &referrer_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1216892745;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the screen for gifting Telegram Premium subscriptions to friends via inputInvoiceTelegram with telegramPaymentPurposePremiumGiftCodes payments or in-store purchases.
 */
class internalLinkTypePremiumGift final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Referrer specified in the link.
  string referrer_;

  /**
   * The link is a link to the screen for gifting Telegram Premium subscriptions to friends via inputInvoiceTelegram with telegramPaymentPurposePremiumGiftCodes payments or in-store purchases.
   */
  internalLinkTypePremiumGift();

  /**
   * The link is a link to the screen for gifting Telegram Premium subscriptions to friends via inputInvoiceTelegram with telegramPaymentPurposePremiumGiftCodes payments or in-store purchases.
   *
   * \param[in] referrer_ Referrer specified in the link.
   */
  explicit internalLinkTypePremiumGift(string const &referrer_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1523936577;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link with a Telegram Premium gift code. Call checkPremiumGiftCode with the given code to process the link. If the code is valid and the user wants to apply it, then call applyPremiumGiftCode.
 */
class internalLinkTypePremiumGiftCode final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The Telegram Premium gift code.
  string code_;

  /**
   * The link is a link with a Telegram Premium gift code. Call checkPremiumGiftCode with the given code to process the link. If the code is valid and the user wants to apply it, then call applyPremiumGiftCode.
   */
  internalLinkTypePremiumGiftCode();

  /**
   * The link is a link with a Telegram Premium gift code. Call checkPremiumGiftCode with the given code to process the link. If the code is valid and the user wants to apply it, then call applyPremiumGiftCode.
   *
   * \param[in] code_ The Telegram Premium gift code.
   */
  explicit internalLinkTypePremiumGiftCode(string const &code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -564356974;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the privacy and security section of the application settings.
 */
class internalLinkTypePrivacyAndSecuritySettings final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the privacy and security section of the application settings.
   */
  internalLinkTypePrivacyAndSecuritySettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1386255665;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a proxy. Call addProxy with the given parameters to process the link and add the proxy.
 */
class internalLinkTypeProxy final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Proxy server domain or IP address.
  string server_;
  /// Proxy server port.
  int32 port_;
  /// Type of the proxy.
  object_ptr<ProxyType> type_;

  /**
   * The link is a link to a proxy. Call addProxy with the given parameters to process the link and add the proxy.
   */
  internalLinkTypeProxy();

  /**
   * The link is a link to a proxy. Call addProxy with the given parameters to process the link and add the proxy.
   *
   * \param[in] server_ Proxy server domain or IP address.
   * \param[in] port_ Proxy server port.
   * \param[in] type_ Type of the proxy.
   */
  internalLinkTypeProxy(string const &server_, int32 port_, object_ptr<ProxyType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1313788694;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a chat by its username. Call searchPublicChat with the given chat username to process the link. If the chat is found, open its profile information screen or the chat itself. If draft text isn't empty and the chat is a private chat with a regular user, then put the draft text in the input field.
 */
class internalLinkTypePublicChat final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the chat.
  string chat_username_;
  /// Draft text for message to send in the chat.
  string draft_text_;
  /// True, if chat profile information screen must be opened; otherwise, the chat itself must be opened.
  bool open_profile_;

  /**
   * The link is a link to a chat by its username. Call searchPublicChat with the given chat username to process the link. If the chat is found, open its profile information screen or the chat itself. If draft text isn't empty and the chat is a private chat with a regular user, then put the draft text in the input field.
   */
  internalLinkTypePublicChat();

  /**
   * The link is a link to a chat by its username. Call searchPublicChat with the given chat username to process the link. If the chat is found, open its profile information screen or the chat itself. If draft text isn't empty and the chat is a private chat with a regular user, then put the draft text in the input field.
   *
   * \param[in] chat_username_ Username of the chat.
   * \param[in] draft_text_ Draft text for message to send in the chat.
   * \param[in] open_profile_ True, if chat profile information screen must be opened; otherwise, the chat itself must be opened.
   */
  internalLinkTypePublicChat(string const &chat_username_, string const &draft_text_, bool open_profile_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1769614592;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link can be used to login the current user on another device, but it must be scanned from QR-code using in-app camera. An alert similar to &quot;This code can be used to allow someone to log in to your Telegram account. To confirm Telegram login, please go to Settings &gt; Devices &gt; Scan QR and scan the code&quot; needs to be shown.
 */
class internalLinkTypeQrCodeAuthentication final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link can be used to login the current user on another device, but it must be scanned from QR-code using in-app camera. An alert similar to &quot;This code can be used to allow someone to log in to your Telegram account. To confirm Telegram login, please go to Settings &gt; Devices &gt; Scan QR and scan the code&quot; needs to be shown.
   */
  internalLinkTypeQrCodeAuthentication();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1089332956;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link forces restore of App Store purchases when opened. For official iOS application only.
 */
class internalLinkTypeRestorePurchases final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link forces restore of App Store purchases when opened. For official iOS application only.
   */
  internalLinkTypeRestorePurchases();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 606090371;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to application settings.
 */
class internalLinkTypeSettings final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to application settings.
   */
  internalLinkTypeSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 393561524;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a sticker set. Call searchStickerSet with the given sticker set name to process the link and show the sticker set. If the sticker set is found and the user wants to add it, then call changeStickerSet.
 */
class internalLinkTypeStickerSet final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the sticker set.
  string sticker_set_name_;
  /// True, if the sticker set is expected to contain custom emoji.
  bool expect_custom_emoji_;

  /**
   * The link is a link to a sticker set. Call searchStickerSet with the given sticker set name to process the link and show the sticker set. If the sticker set is found and the user wants to add it, then call changeStickerSet.
   */
  internalLinkTypeStickerSet();

  /**
   * The link is a link to a sticker set. Call searchStickerSet with the given sticker set name to process the link and show the sticker set. If the sticker set is found and the user wants to add it, then call changeStickerSet.
   *
   * \param[in] sticker_set_name_ Name of the sticker set.
   * \param[in] expect_custom_emoji_ True, if the sticker set is expected to contain custom emoji.
   */
  internalLinkTypeStickerSet(string const &sticker_set_name_, bool expect_custom_emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1589227614;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a story. Call searchPublicChat with the given sender username, then call getStory with the received chat identifier and the given story identifier, then show the story if received.
 */
class internalLinkTypeStory final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the sender of the story.
  string story_sender_username_;
  /// Story identifier.
  int32 story_id_;

  /**
   * The link is a link to a story. Call searchPublicChat with the given sender username, then call getStory with the received chat identifier and the given story identifier, then show the story if received.
   */
  internalLinkTypeStory();

  /**
   * The link is a link to a story. Call searchPublicChat with the given sender username, then call getStory with the received chat identifier and the given story identifier, then show the story if received.
   *
   * \param[in] story_sender_username_ Username of the sender of the story.
   * \param[in] story_id_ Story identifier.
   */
  internalLinkTypeStory(string const &story_sender_username_, int32 story_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1471997511;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a cloud theme. TDLib has no theme support yet.
 */
class internalLinkTypeTheme final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Name of the theme.
  string theme_name_;

  /**
   * The link is a link to a cloud theme. TDLib has no theme support yet.
   */
  internalLinkTypeTheme();

  /**
   * The link is a link to a cloud theme. TDLib has no theme support yet.
   *
   * \param[in] theme_name_ Name of the theme.
   */
  explicit internalLinkTypeTheme(string const &theme_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -200935417;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to the theme section of the application settings.
 */
class internalLinkTypeThemeSettings final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to the theme section of the application settings.
   */
  internalLinkTypeThemeSettings();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1051903722;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is an unknown tg: link. Call getDeepLinkInfo to process the link.
 */
class internalLinkTypeUnknownDeepLink final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Link to be passed to getDeepLinkInfo.
  string link_;

  /**
   * The link is an unknown tg: link. Call getDeepLinkInfo to process the link.
   */
  internalLinkTypeUnknownDeepLink();

  /**
   * The link is an unknown tg: link. Call getDeepLinkInfo to process the link.
   *
   * \param[in] link_ Link to be passed to getDeepLinkInfo.
   */
  explicit internalLinkTypeUnknownDeepLink(string const &link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 625596379;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to an unsupported proxy. An alert can be shown to the user.
 */
class internalLinkTypeUnsupportedProxy final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:

  /**
   * The link is a link to an unsupported proxy. An alert can be shown to the user.
   */
  internalLinkTypeUnsupportedProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -566649079;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a user by its phone number. Call searchUserByPhoneNumber with the given phone number to process the link. If the user is found, then call createPrivateChat and open user's profile information screen or the chat itself. If draft text isn't empty, then put the draft text in the input field.
 */
class internalLinkTypeUserPhoneNumber final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Phone number of the user.
  string phone_number_;
  /// Draft text for message to send in the chat.
  string draft_text_;
  /// True, if user's profile information screen must be opened; otherwise, the chat itself must be opened.
  bool open_profile_;

  /**
   * The link is a link to a user by its phone number. Call searchUserByPhoneNumber with the given phone number to process the link. If the user is found, then call createPrivateChat and open user's profile information screen or the chat itself. If draft text isn't empty, then put the draft text in the input field.
   */
  internalLinkTypeUserPhoneNumber();

  /**
   * The link is a link to a user by its phone number. Call searchUserByPhoneNumber with the given phone number to process the link. If the user is found, then call createPrivateChat and open user's profile information screen or the chat itself. If draft text isn't empty, then put the draft text in the input field.
   *
   * \param[in] phone_number_ Phone number of the user.
   * \param[in] draft_text_ Draft text for message to send in the chat.
   * \param[in] open_profile_ True, if user's profile information screen must be opened; otherwise, the chat itself must be opened.
   */
  internalLinkTypeUserPhoneNumber(string const &phone_number_, string const &draft_text_, bool open_profile_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 273398536;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a user by a temporary token. Call searchUserByToken with the given token to process the link. If the user is found, then call createPrivateChat and open the chat.
 */
class internalLinkTypeUserToken final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// The token.
  string token_;

  /**
   * The link is a link to a user by a temporary token. Call searchUserByToken with the given token to process the link. If the user is found, then call createPrivateChat and open the chat.
   */
  internalLinkTypeUserToken();

  /**
   * The link is a link to a user by a temporary token. Call searchUserByToken with the given token to process the link. If the user is found, then call createPrivateChat and open the chat.
   *
   * \param[in] token_ The token.
   */
  explicit internalLinkTypeUserToken(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1462248615;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a video chat. Call searchPublicChat with the given chat username, and then joinGroupCall with the given invite hash to process the link.
 */
class internalLinkTypeVideoChat final : public InternalLinkType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  /// Username of the chat with the video chat.
  string chat_username_;
  /// If non-empty, invite hash to be used to join the video chat without being muted by administrators.
  string invite_hash_;
  /// True, if the video chat is expected to be a live stream in a channel or a broadcast group.
  bool is_live_stream_;

  /**
   * The link is a link to a video chat. Call searchPublicChat with the given chat username, and then joinGroupCall with the given invite hash to process the link.
   */
  internalLinkTypeVideoChat();

  /**
   * The link is a link to a video chat. Call searchPublicChat with the given chat username, and then joinGroupCall with the given invite hash to process the link.
   *
   * \param[in] chat_username_ Username of the chat with the video chat.
   * \param[in] invite_hash_ If non-empty, invite hash to be used to join the video chat without being muted by administrators.
   * \param[in] is_live_stream_ True, if the video chat is expected to be a live stream in a channel or a broadcast group.
   */
  internalLinkTypeVideoChat(string const &chat_username_, string const &invite_hash_, bool is_live_stream_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2020149068;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;
};

/**
 * The link is a link to a Web App. Call searchPublicChat with the given bot username, check that the user is a bot, then call searchWebApp with the received bot and the given web_app_short_name. Process rec