/* $Id: gic_confmgr.h,v 1.5 2005/06/20 06:49:34 cegger Exp $
******************************************************************************

   Config Manager for LibGIC.

   Copyright (C) 1999 Andrew Apted	[andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#ifndef _GGI_GIC_CONFMGR_H
#define _GGI_GIC_CONFMGR_H

__BEGIN_DECLS

#include <ggi/gic.h>


/* Styles.  You can control the "look and feel" of the config manager by
 * the way your program renders boxes & text in these styles.
 */

typedef enum confmgr_style
{
	/* The background, which generally appears in the places that
	 * text doesn't :).  Note that text is never drawn using this
	 * style (or any other style below which contains `BACKGROUND').
	 */
	CONFMGR_STYLE_BACKGROUND,

	/* These are for pop-up menus that appear.  `HIGHLIGHT' is for
	 * the currently selected menu item.
	 */
	CONFMGR_STYLE_MENU_BACKGROUND,
	CONFMGR_STYLE_MENU_HEADING,
	CONFMGR_STYLE_MENU_ITEM,
	CONFMGR_STYLE_MENU_HIGHLIGHT,

	/* These are for pop-up message boxes that appear.
	 */
	CONFMGR_STYLE_MESSAGE_BACKGROUND,
	CONFMGR_STYLE_MESSAGE_NORMAL,
	CONFMGR_STYLE_MESSAGE_ERROR,
	
	/* These are for the section backgrounds.  Generally
	 * CONFMGR_STYLE_SECTION_BACKGROUND will be the same as
	 * CONFMGR_STYLE_BACKGROUND.  `HIGHLIGHT' is used for the
	 * background of the current section, and is only used when the
	 * flag CONFMGR_FLAG_HIGHLIGHT_SECTION is set.  Note that text
	 * is never drawn using these styles.
	 */
	CONFMGR_STYLE_SECTION_BACKGROUND,
	CONFMGR_STYLE_SECTION_HIGHLIGHT,

	/* These are for the various headings (mainly the section
	 * headings) that appear on-screen.  `HIGHLIGHT' is for the
	 * heading of the current section.
	 */
	CONFMGR_STYLE_HEADING_TEXT,
	CONFMGR_STYLE_HEADING_HIGHLIGHT,

	/* The names of `items', such as contexts, controls & features.
	 * Note that `CURRENT' is for the current item in each
	 * non-current section, and `HIGHLIGHT' is for the current item
	 * in the current section.
	 */
	CONFMGR_STYLE_ITEM_TEXT,
	CONFMGR_STYLE_ITEM_CURRENT,
	CONFMGR_STYLE_ITEM_HIGHLIGHT,

	/* The style for the bindings, analogous to the ones for items
	 * above.
	 */
	CONFMGR_STYLE_BINDING_TEXT,
	CONFMGR_STYLE_BINDING_CURRENT,
	CONFMGR_STYLE_BINDING_HIGHLIGHT,

	/* The background drawn behind the bars in TEST mode.
	 */
	CONFMGR_STYLE_TEST_BACKGROUND

} confmgr_style;


/* Sounds.  You can also enhance the "look and feel" of the config
 * manager by the way your program generates these sounds.
 */

typedef enum confmgr_sound
{
	/* Initial and final sounds, which occur at the very beginning
	 * and at the very end of running the config manager,
	 * respectively.
	 */
	CONFMGR_SOUND_START,
	CONFMGR_SOUND_STOP,

	/* Sounds for moving between two items in the same section (or
	 * menu) and between two sections.
	 */
	CONFMGR_SOUND_NEW_ITEM,
	CONFMGR_SOUND_NEW_SECTION,
	
	/* Sounds for pop-up menus.
	 */
	CONFMGR_SOUND_MENU_START,
	CONFMGR_SOUND_MENU_STOP,
	
	/* Sounds for pop-up messages.  The first two are for `good'
	 * messages, the second two are for `bad' messages.
	 */
	CONFMGR_SOUND_MESSAGE_START,
	CONFMGR_SOUND_MESSAGE_STOP,

	CONFMGR_SOUND_ERROR_START,
	CONFMGR_SOUND_ERROR_STOP,
	
	/* Sounds for feature training.
	 */
	CONFMGR_SOUND_TRAIN_START,
	CONFMGR_SOUND_TRAIN_STOP,
	
	/* Sounds for feature testing.
	 */
	CONFMGR_SOUND_TEST_START,
	CONFMGR_SOUND_TEST_STOP,
	
	/* The sound made after deleting a binding.
	 */
	CONFMGR_SOUND_DELETED,
	
	/* This sound is for when the user tried to do something that
	 * they cannot do (and which isn't serious enough to cause a
	 * pop-up message to appear).
	 */
	CONFMGR_SOUND_INVALID
	
} confmgr_sound;


/* Fonts.  There are two different font sizes, big and small, although a
 * config manager front-end is free to use different _styles_ depending
 * on the style given to draw_text().
 */

typedef enum confmgr_font
{
	CONFMGR_FONT_SMALL,
	CONFMGR_FONT_BIG

} confmgr_font;


/* Flags. */

/* This flag controls whether the background of sections is highlighted.
 * Only when this flag is set is draw_box() called with style ==
 * CONFMGR_STYLE_SECTION_HIGHLIGHT.
 */
#define CONFMGR_FLAG_HIGHLIGHT_SECTION    0x0001


/* Some helper structures */

typedef struct confmgr_coord
{
	int x, y;	

} confmgr_coord;

typedef struct confmgr_border
{
	int left, right, top, bottom;

} confmgr_border;


/* Info.  This structure defines the interface between the program and
 * the config manager.  It contains a callback API which the config
 * manager uses to read events and draw it's screen.
 */

typedef struct confmgr_info
{
	/* The LibGIC handle to which all the rest belongs.
	 */
	gic_handle_t handle;

	/* The LibGIC head, whose contexts, controls & features will be
	 * managed by the config manager.
	 */
	gic_head *head;

	/* Private fields, one for the application, and one for the
	 * config manager itself.
	 */
	void *appl_priv;
	void *manager_priv;

	/* Screen size.  Note that the config manager will never try to
	 * draw anything outside the bounds of the screen.
	 */
	confmgr_coord screen_size;

	/* Character sizes.  Note that text is assumed to be drawn
	 * monospaced.
	 */
	confmgr_coord small_size;
	confmgr_coord big_size;

	/* Max Lengths.  These define how many characters that the
	 * context, control, feature & binding items can contain.
	 */
	 
	size_t context_max;
	size_t control_max;
	size_t feature_max;
	size_t binding_max;

	/* Gaps.  These defined how much space to leave between
	 * sections, items (including menu items) and bindings.
	 */
	confmgr_coord section_gap;
	confmgr_coord item_gap;
	confmgr_coord binding_gap;

	/* Borders.  These define how far in from the background area to
	 * render the foreground stuff.
	 */
	confmgr_border section_border;
	confmgr_border menu_border;
	confmgr_border message_border;

	/* Boxes.  These define how much bigger to make boxes that
	 * contain text.
	 */
	confmgr_border heading_box;
	confmgr_border item_box;
	confmgr_border binding_box;

	/* Flags, used to tweak various things.  Set to 0 to get the
	 * default behaviour.
	 */
	int flags;


	/*---- Callback API ----*/
	

	/* Read an event.  The call should block if there aren't any
	 * events available, up to a maximum time given in `timeout', or
	 * indefinitely if timeout is NULL.  It should return 1 if an
	 * event was read, otherwise 0.
	 */
	int (* read_event)(struct confmgr_info *info, gii_event *event,
			   struct timeval *timeout);

	/* Draw a box, using the given style, at the given coordinates
	 * (x, y) and of the given size (w, h).
	 */
	void (* draw_box)(struct confmgr_info *info,
	                  confmgr_style style,
	                  int x, int y, int w, int h);
	
	/* Draw some text, using the given style & font, at the given
	 * coordinates (x, y).  The background of the text should be
	 * solid, matching the result of draw_box() over the same area
	 * with the same style.
	 */
	void (* draw_text)(struct confmgr_info *info,
	                   confmgr_style style,
			   confmgr_font font,
	                   int x, int y, const char *text);

	/* Draw a status-like bar, which shows the user visually the
	 * given state (from GIC_STATE_MIN to GIC_STATE_MAX).  The bar
	 * has the given coordinates (x, y) and size (w, h).  It can be
	 * assumed, that bar always moves horizontally (i.e. no tall
	 * skinny bars).  Note that `state' may be GIC_NOACTION.
	 */
	void (* draw_bar)(struct confmgr_info *info,
			  gic_state state, int x, int y, int w, int h);

	/* Flush: this call should ensure that anything drawn so far by
	 * the above routines is made visible.
	 */
	void (* flush)(struct confmgr_info *info);

	/* Make a sound, of the given type.
	 */
	void (* make_sound)(struct confmgr_info *info,
	                    confmgr_sound sound);

} confmgr_info;


/* The main function.
 *
 * This will do everything, using the information in the confmgr
 * info, such as draw the screen, read events from the user, allow the
 * user to browse the contexts, controls & features and train the
 * features (adding new bindings or removing old ones).
 *
 * It returns 0 if all went well, or a negative error code if some
 * serious problem occurred (like out of memory).  These error codes are
 * defined below.
 */

GGIGICAPIFUNC int gicConfigManager(confmgr_info *info);


/* Return codes
 */
#define CONFMGR_OK             0   /* Keep changes but don't save */
#define CONFMGR_SAVE           1   /* Keep and save the changes */
#define CONFMGR_CANCEL         2   /* Don't keep changes (i.e. reload) */

#define CONFMGR_ENOMEM		GGI_ENOMEM	/* Out of memory */
#define CONFMGR_ENOFILE		GGI_ENOFILE	/* Missing config file */
#define CONFMGR_EBADFILE	GGI_EBADFILE	/* Error in config file */
#define CONFMGR_EINVAL		GGI_ARGINVAL	/* Invalid argument(s) */
#define CONFMGR_ENOROOM		GGI_ENOSPACE	/* The screen is too small */

#define CONFMGR_EUNKNOWN	GGI_EUNKNOWN	/* Unspecified error */


__END_DECLS

#endif /* _GGI_GIC_CONFMGR_H */
