! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!==============================================================================
!
#define ESMF_FILENAME "ESMF_StateRemRep.cppF90"
!
! ESMF StateRemRep module
module ESMF_StateRemRepMod
!
!==============================================================================
!
! This file contains the State user API methods for Remove and Replace.
!
!------------------------------------------------------------------------------
! INCLUDES
!------------------------------------------------------------------------------
#include "ESMF.h"
!------------------------------------------------------------------------------
!BOPI
! !MODULE: ESMF_StateRemRepMod - Data exchange between components
!
! !DESCRIPTION:
!
! The code in this file implements the Fortran implementations
! of the {\tt StateRemove} and {\tt StateReplace} methods.
!
!
! !USES:
      use ESMF_InitMacrosMod
      use ESMF_LogErrMod
      use ESMF_BaseMod
      use ESMF_ContainerMod
      use ESMF_UtilStringMod
      use ESMF_UtilTypesMod
      use ESMF_ArrayMod
      use ESMF_ArrayBundleMod
      use ESMF_FieldMod
      use ESMF_FieldBundleMod
      use ESMF_RHandleMod
      use ESMF_StateAPIMod
      use ESMF_StateTypesMod
      use ESMF_StateVaMod
      use ESMF_StateInternalsMod
      implicit none
!------------------------------------------------------------------------------
! !PRIVATE TYPES:
      private
!------------------------------------------------------------------------------
! !PUBLIC MEMBER FUNCTIONS:
      public :: ESMF_StateRemove
      public :: ESMF_StateReplace
!EOPI
!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
      character(*), parameter, private :: version = &
      '$Id$'
!==============================================================================
!
! INTERFACE BLOCKS
!
!==============================================================================
!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_StateRemove -- Remove items in a State
! !INTERFACE:
  interface ESMF_StateRemove
! !PRIVATE MEMBER FUNCTIONS:
    module procedure ESMF_StateRemoveOneItem
    module procedure ESMF_StateRemoveList
! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_StateReplace} functions.
!
!EOPI
  end interface
!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_StateReplace -- Replace items in a State
! !INTERFACE:
  interface ESMF_StateReplace
! !PRIVATE MEMBER FUNCTIONS:
!
! module procedure ESMF_StateRepOneArray
! module procedure ESMF_StateRepOneArrayBundle
! module procedure ESMF_StateRepOneField
! module procedure ESMF_StateRepOneFieldBundle
! module procedure ESMF_StateRepOneState
    module procedure ESMF_StateRepArrayList
    module procedure ESMF_StateRepArrayBundleList
    module procedure ESMF_StateRepFieldList
    module procedure ESMF_StateRepFieldBundleList
    module procedure ESMF_StateRepRoutehandleList
    module procedure ESMF_StateRepStateList
! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_StateReplace} functions.
!
!EOPI
  end interface
!------------------------------------------------------------------------------
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
contains
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateRemove"
!BOP
! !IROUTINE: ESMF_StateRemove - Remove an item from a State - (DEPRECATED METHOD)
!
! !INTERFACE:
! ! Private name; call using ESMF_StateRemove ()
  subroutine ESMF_StateRemoveOneItem (state, itemName, keywordEnforcer, &
      relaxedFlag, rc)
!
! !ARGUMENTS:
    type(ESMF_State), intent(inout) :: state
    character(*), intent(in) :: itemName
    type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: relaxedFlag
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \item\apiDeprecatedMethodWithReplacement{5.3.1}{ESMF\_StateRemove}{esmfstateremovelist}
! Rationale: The list version is consistent with other ESMF container
! operations which use lists.
! \end{itemize}
!
! !DESCRIPTION:
! Remove an existing reference to an item from a {\tt State}.
!
! The arguments are:
! \begin{description}
! \item[state]
! The {\tt ESMF\_State} within which {\tt itemName} will be removed.
! \item[itemName]
! The name of the item to be removed. This is a reference only.
! The item itself is unchanged.
!
! If the {\tt state} contains nested {\tt ESMF\_State}s,
! the {\tt itemName} argument may specify a fully qualified name
! to remove the desired item with a single call. This is performed
! using the "/" character to separate the names of the intermediate
! State names leading to the desired item. (E.g.,
! {\tt itemName="state1/state12/item"}.
!
! Since an item could potentially be referenced by multiple containers,
! it remains the responsibility of the user to manage its
! destruction when it is no longer in use.
! \item[{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "remove",
! where it is {\em not} an error if {\tt itemName} is not present in the
! {\tt state}. For {\tt .false.} this is treated
! as an error condition. The default setting is {\tt .false.}.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: localrc
    ! Initialize return code; assume failure until success is certain
    if (present(rc)) rc = ESMF_RC_NOT_IMPL
    localrc = ESMF_RC_NOT_IMPL
    call ESMF_LogWrite ( &
        "Scalar itemName argument is deprecated.  Use itemNameList instead.", &
        logmsgFlag=ESMF_LOGMSG_WARNING, method=ESMF_METHOD, &
        rc=localrc)
    call ESMF_StateRemoveList (state, (/ itemName /), &
        relaxedflag=relaxedflag, rc=localrc)
    if (present(rc)) rc = localrc
  end subroutine ESMF_StateRemoveOneItem
!-------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_StateRemoveList"
!BOP
! !IROUTINE: ESMF_StateRemove - Remove a list of items from a State
! \label{esmfstateremovelist}
!
! !INTERFACE:
! ! Private name; call using ESMF_StateRemove ()
  subroutine ESMF_StateRemoveList (state, itemNameList, keywordEnforcer, relaxedFlag, rc)
!
! !ARGUMENTS:
    type(ESMF_State), intent(inout) :: state
    character(*), intent(in) :: itemNameList(:)
    type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    logical, intent(in), optional :: relaxedFlag
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.3.1}
! \end{itemize}
!
! !DESCRIPTION:
! Remove existing references to items from a {\tt State}.
!
! The arguments are:
! \begin{description}
! \item[state]
! The {\tt ESMF\_State} within which {\tt itemName} will be removed.
! \item[itemNameList]
! The name of the items to be removed. This is a reference only.
! The items themselves are unchanged.
!
! If the {\tt state} contains nested {\tt ESMF\_State}s,
! the {\tt itemName} arguments may specify fully qualified names
! to remove the desired items with a single call. This is performed
! using the "/" character to separate the names of the intermediate
! State names leading to the desired items. (E.g.,
! {\tt itemName="state1/state12/item"}.
!
! Since items could potentially be referenced by multiple containers,
! it remains the responsibility of the user to manage their
! destruction when they are no longer in use.
! \item[{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "remove",
! where it is {\em not} an error if an item in the {\tt itemNameList}
! is not present in the {\tt state}. For {\tt .false.} this is treated
! as an error condition. The default setting is {\tt .false.}.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    type(ESMF_StateItem), pointer :: dataitem
    type(ESMF_StateClass), pointer :: localstatep
    character(len=ESMF_MAXSTR) :: errmsg
    logical :: exists
    integer :: i
    integer :: ipos, iposnext
    type(ESMF_Logical) :: linkChange
    integer :: localrc
    logical :: localrelaxed
    integer :: memstat
    integer :: namelen
    ! Initialize return code; assume failure until success is certain
    if (present(rc)) rc = ESMF_RC_NOT_IMPL
    localrc = ESMF_RC_NOT_IMPL
    ! check input variables
    ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc)
    localrelaxed = .false.
    if (present (relaxedFlag)) then
      localrelaxed = relaxedFlag
    end if
    do, i=1, size (itemNameList)
      localstatep => null ()
      exists = ESMF_StateClassFindData (state%statep, &
                                         dataname=itemNameList(i), &
                                         expected=.true., &
                                         dataitem=dataitem, &
                                         dataState=localstatep, &
                                         rc=localrc)
      if (.not. exists) then
        if (localrelaxed) then
          localrc = ESMF_SUCCESS
          cycle
        else
          errmsg = ESMF_StringConcat ("could not find ", &
                   ESMF_StringConcat (trim (itemNameList(i)), " for removal"))
          if (ESMF_LogFoundError(ESMF_RC_NOT_FOUND, msg=errmsg, &
                ESMF_CONTEXT, rcToReturn=rc)) return
        end if
      end if
      ipos = 1
      namelen = len_trim (itemNameList(i))
      do
        iposnext = index (itemNameList(i)(ipos:namelen), '/')
        if (iposnext == 0) exit
        ipos = ipos + iposnext
      end do
      call ESMF_ContainerRemove (localstatep%stateContainer, &
          itemNameList=(/ itemNameList(i)(ipos:namelen) /), rc=localrc)
      if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    end do
    if (present(rc)) rc = localrc
  end subroutine ESMF_StateRemoveList
!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_StateReplace - Replace a single item to a State
!
! !INTERFACE:
! subroutine ESMF_StateReplace(state, <item>, rc)
!
! !ARGUMENTS:
! type(ESMF_State), intent(inout) :: state
! <item>, see below for supported values
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Replace an existing reference to a single <item> to an existing
! {\tt state}. The name of the <item> must be unique within the
! {\tt state}.
!
! Supported values for <item> are:
! \begin{description}
! \item type(ESMF\_Array), intent(in) :: array
! \item type(ESMF\_ArrayBundle), intent(in) :: arraybundle
! \item type(ESMF\_Field), intent(in) :: field
! \item type(ESMF\_FieldBundle), intent(in) :: fieldbundle
! \item type(ESMF\_State), intent(in) :: nestedState
! \end{description}
!
! The arguments are:
! \begin{description}
! \item[state]
! The {\tt ESMF\_State} to which <item>s will be replaced.
! \item[<item>]
! The replacement <item>. This is a reference only; when
! the {\tt state} is destroyed the <item>s contained in it will
! not be destroyed. Also, the <item> cannot be safely
! destroyed before the {\tt state} is destroyed.
! Since <item>s can be added to multiple containers, it remains
! the responsibility of the user to manage their
! destruction when they are no longer in use.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOPI
!------------------------------------------------------------------------------
! #define StateReplaceMacro(mtype, mentry, mname) ! !------------------------------------------------------------------------------ 
! ! undef ESMF_METHOD 
! ! define ESMF_METHOD "ESMF_StateReplace" 
! !BOPI 
! ! !IROUTINE: ESMF_StateReplace - Replace an Array in a State 
! ! 
! ! !INTERFACE: 
! ! Private name; call using ESMF_StateReplace() 
! subroutine ESMF_StateRep##mentry (state, mname, keywordEnforcer, rc) 
! ! 
! ! !ARGUMENTS: 
! type(ESMF_State), intent(inout) :: state 
! type(ESMF_##mtype),intent(in) :: mname 
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! integer, intent(out), optional :: rc 
! !EOPI 
! !------------------------------------------------------------------------------ 
! 
! integer :: localrc 
! character(ESMF_MAXSTR) :: itemname 
! 
! ! call ESMF_##mtype##Get (mname, name=itemname, rc=localrc) 
! ! if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
! ! ESMF_CONTEXT, rcToReturn=rc)) return 
! ! 
! ! call ESMF_StateRemove (state, itemName=itemname, rc=localrc) 
! ! if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
! ! ESMF_CONTEXT, rcToReturn=rc)) return 
! 
! call ESMF_StateAdd (state, mname, & 
! proxyflag=.false., & 
! replaceflag=.true., & 
! rc=rc) 
! 
! end subroutine ESMF_StateRep##mentry
!
! StateReplaceMacro(Array,OneArray,array)
! StateReplaceMacro(ArrayBundle,OneArrayBundle,arraybundle)
! StateReplaceMacro(Field,OneField,field)
! StateReplaceMacro(FieldBundle,OneFieldBundle,fieldbundle)
! StateReplaceMacro(State,OneState,nestedState)
!------------------------------------------------------------------------------
!BOP
! !IROUTINE: ESMF_StateReplace - Replace a list of items within a State
!
! !INTERFACE:
! subroutine ESMF_StateReplace(state, <itemList>, relaxedflag, rc)
!
! !ARGUMENTS:
! type(ESMF_State), intent(inout) :: state
! <itemList>, see below for supported values
! type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
! logical, intent(in), optional :: relaxedflag
! integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! Replace a list of items with a {\tt ESMF\_State}. If an item in
! <itemList> does not match any items already present in {\tt state}, an
! error is returned.
!
! Supported values for <itemList> are:
! \begin{description}
! \item type(ESMF\_Array), intent(in) :: arrayList(:)
! \item type(ESMF\_ArrayBundle), intent(in) :: arraybundleList(:)
! \item type(ESMF\_Field), intent(in) :: fieldList(:)
! \item type(ESMF\_FieldBundle), intent(in) :: fieldbundleList(:)
! \item type(ESMF\_RouteHandle), intent(in) :: routehandleList(:)
! \item type(ESMF\_State), intent(in) :: nestedStateList(:)
! \end{description}
!
! The arguments are:
! \begin{description}
! \item[state]
! An {\tt ESMF\_State} within which the <itemList> items will be replaced.
! \item[<itemList>]
! The list of items to be replaced.
! This is a reference only; when
! the {\tt ESMF\_State} is destroyed the <itemList> contained in it will
! not be destroyed. Also, the items in the <itemList> cannot be safely
! destroyed before the {\tt ESMF\_State} is destroyed.
! Since <itemList> items can be added to multiple containers, it remains
! the responsibility of the user to manage their
! destruction when they are no longer in use.
! \item[{[relaxedflag]}]
! A setting of {\tt .true.} indicates a relaxed definition of "replace",
! where it is {\em not} an error if {\tt <itemList>} contains items
! with names that are not found in {\tt state}. The {\tt State}
! is left unchanged for these items. For {\tt .false.} this is treated
! as an error condition. The default setting is {\tt .false.}.
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateReplaceList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateRepArrayList(state, arrayList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Array), intent(in) :: arrayList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lobject, lname, lvalue1, lvalue2 
#endif 
 logical :: localrelaxed 
 type(ESMF_Logical) :: linkChange 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (arrayList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayGetInit,arrayList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, arrayList, & 
 replaceflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateRepArrayList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateReplaceList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateRepArrayBundleList(state, arraybundleList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_ArrayBundle), intent(in) :: arraybundleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lobject, lname, lvalue1, lvalue2 
#endif 
 logical :: localrelaxed 
 type(ESMF_Logical) :: linkChange 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (arraybundleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_ArrayBundleGetInit,arraybundleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, arraybundleList, & 
 replaceflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateRepArrayBundleList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateReplaceList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateRepFieldList(state, fieldList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_Field), intent(in) :: fieldList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lobject, lname, lvalue1, lvalue2 
#endif 
 logical :: localrelaxed 
 type(ESMF_Logical) :: linkChange 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (fieldList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,fieldList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, fieldList, & 
 replaceflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateRepFieldList
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateReplaceList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateRepFieldBundleList(state, fieldbundleList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_FieldBundle), intent(in) :: fieldbundleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lobject, lname, lvalue1, lvalue2 
#endif 
 logical :: localrelaxed 
 type(ESMF_Logical) :: linkChange 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (fieldbundleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldBundleGetInit,fieldbundleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, fieldbundleList, & 
 replaceflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateRepFieldBundleList
#define noattributesversion
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateReplaceList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateRepRouteHandleList(state, routehandleList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_RouteHandle), intent(in) :: routehandleList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lobject, lname, lvalue1, lvalue2 
#endif 
 logical :: localrelaxed 
 type(ESMF_Logical) :: linkChange 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (routehandleList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_RouteHandleGetInit,routehandleList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, routehandleList, & 
 replaceflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateRepRouteHandleList
#undef noattributesversion
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_StateReplaceList" 
!BOPI 
! !IROUTINE: ESMF_StateAdd - Add a list of items to a State 
! 
! !INTERFACE: 
 ! Private name; call using ESMF_StateAdd() 
 subroutine ESMF_StateRepStateList(state, nestedStateList, & 
 keywordEnforcer, relaxedflag, rc) 
! 
! !ARGUMENTS: 
 type(ESMF_State), intent(inout) :: state 
 type(ESMF_State), intent(in) :: nestedStateList(:) 
 type(ESMF_KeywordEnforcer), optional :: keywordEnforcer ! must use keywords for the below 
 logical, intent(in), optional :: relaxedflag 
 integer, intent(out), optional :: rc 
!EOPI 
!------------------------------------------------------------------------------ 
 integer :: localrc, i 
 integer :: localcount 
#if !defined (noattributesversion) 
 character(ESMF_MAXSTR) :: lobject, lname, lvalue1, lvalue2 
#endif 
 logical :: localrelaxed 
 type(ESMF_Logical) :: linkChange 
 
 ! Initialize return code; assume routine not implemented 
 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 call ESMF_StateValidate(state, rc=localrc) 
 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 localrelaxed = .false. 
 if (present (relaxedflag)) then 
 localrelaxed = relaxedflag 
 end if 
 
 localcount = size (nestedStateList) 
 
 ! check input variables 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,state,rc) 
 do i=1, localcount 
 ESMF_INIT_CHECK_DEEP(ESMF_StateGetInit,nestedStateList(i),rc) 
 enddo 
 
 call ESMF_StateClsAdd (state%statep, nestedStateList, & 
 replaceflag=.true., relaxedflag=localrelaxed, rc=localrc) 
 if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 
 
 if (present(rc)) rc = ESMF_SUCCESS 
 end subroutine ESMF_StateRepStateList
end module ESMF_StateRemRepMod
