/* cLog - cross platform logging library
 * Copyright (c) 2003 WebGroup Media, LLC <support@webgroupmedia.com>
 *
 * The MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */
/** \mainpage CLogger

  \section introduction Introduction
  CLogger is a simple logger that logs to files. It is designed to be correct and portable.

  \section compiling Compiling
  \subsection configure Configure
    After unpacking the archive, you will need to run the ./configure
    script and perhaps add the --prefix tag to install the library to
    a different location than the default. The configure script will
    generate the makefile used in the next step.

  \subsection make Make
    After the configure script has finished generating the makefiles
    simply run 'gmake' in the directory and everything will compile.

  \subsection install Install
    When everything is installed just type 'gmake install' and the software
    installs itself.

  \section problems Problems
    Please report any problems with this library to Ben Halsted of WebGroup Media, LLC.
    ben@syght.com or http://www.webgroupmedia.com and find an email address there.

  \section license License
  COPYRIGHT AND PERMISSION NOTICE
 
  Copyright (c) 2002 WebGroup Media, LLC, All rights reserved.
 
  Permission is hereby granted, free of charge, to any person obtaining
  a copy of this software and associated documentation files (the
  "Software"), to deal in the Software without restriction, including
  without limitation the rights to use, copy, modify, merge, publish,
  distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to
  the following conditions:
 
  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.
 
  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
  CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
  TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
  SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  \author Ben Halsted (ben@webgroupmedia.com)
  \date Fri Aug 2 2002

*/

/*
 * COPYRIGHT AND PERMISSION NOTICE                                        
 *                                                                        
 * Copyright (c) 2002 WebGroup Media, LLC, All rights reserved.           
 *                                                                        
 * Permission is hereby granted, free of charge, to any person obtaining  
 * a copy of this software and associated documentation files (the        
 * "Software"), to deal in the Software without restriction, including    
 * without limitation the rights to use, copy, modify, merge, publish,    
 * distribute, sublicense, and/or sell copies of the Software, and to     
 * permit persons to whom the Software is furnished to do so, subject to  
 * the following conditions:                                              
 *                                                                        
 * The above copyright notice and this permission notice shall be         
 * included in all copies or substantial portions of the Software.        
 *                                                                        
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,        
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF     
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY   
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,   
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE      
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.                 
 */

#ifndef __CLOG_H__
#define __CLOG_H__ 1

#include <stdio.h>

/**
 * \defgroup clog CLogger API
 */
/*@{*/

/*!
  This is the type definition for the function pointer that can also get called when logging.

  Example:
  \code
  // our callback function
  void my_callback(int level, char *logtext)
  {
    // print out the message to stdout
    printf("%d - %s", level, logtext);
  }

  // our main function
  main(int argc, char **argv)
  {
    // define variables
    int loglevel1=0;
    int loglevel2=0;
    CLOG_INFO *log=NULL;

    // get log level (perhaps the string was from a config file)
    loglevel1=clog_getlevel("ERROR");
    loglevel2=clog_getlevel("TRACE");

    // open the log
    log = clog_open("/tmp/mylog.txt", loglevel1, my_callback, loglevel2);

    // close the log and free the memory
    clog_close(log);
  }
  \endcode
 */
typedef void (*C_CALLBACK)(int, char*);

/*! \brief The log levels in int format.

  Use these as the second parameter to clog

  \sa clog
*/
enum {
  CMARK  = 0, /*! MARK is used as a non-reporting level */
  /*! FATAL level only reports errors that cause the application to EXIT */
  CFATAL = 1,
  /*! ERROR level also reports errors that are serious but it can keep processing */
  CERROR = 2,
  /*! WARN level also reports errors that may be important */
  CWARN  = 3,
  /*! DEBUG level also reports information on what is happening so errors can be reported better */
  CDEBUG = 4,
  /*! TRACE level reports everything that is loggable */
  CTRACE = 5
};


/*!
  This is the struct that holds the destination info. Do not modify the values directly
*/
struct clog_info_struct {
  /** The FILE* we are going to write to. */
  FILE          * s_logfile;
  /** Log level for logging to file. */
  int    s_logfile_level;
  /** The callback we call if it's not NULL. */
  C_CALLBACK      s_callback;
  /** Log level for logging to the callback. */
  int    s_callback_level;
};

/**
  typedef so that it is easier to create clog_info_struct variables.
*/
typedef struct clog_info_struct CLOG_INFO;

/** \brief Convert char* log level to int value of log level

 Convert text to int log level. This function takes a char* and compares the
 first letter of the string and returns the associated log level.

 \param level Char pointer containing one of "FATAL","ERROR","WARN","DEBUG" ,"TRACE".

 \return returns an int of the associated ENUM (Should be greater than 0)

 \sa CFATAL CERROR CWARN CDEBUG CTRACE

 Example:
 \code
 // define variables
 int loglevel=0;
 CLOG_INFO *log=NULL;

 // get log level (perhaps the string was from a config file)
 loglevel=clog_getlevel("ERROR");

 // open the log
 log = clog_open("/tmp/mylog.txt", loglevel, NULL, 0);

 // close the log
 clog_close(log);
 \endcode
*/
extern int                clog_getlevel(const char *level);

/** \brief sets the logging to a new file

 Closes existing log file (if open) and then opens a new file to log to

 \param info The CLOG_INFO struct that was returned from the clog_open call.
 \param file The file name that you want to now log to.

 \return If successfull it returns 1, On failure it returns 0.

 \sa clog_open clog_setfilelevel clog_getlevel

 Example:
 \code
 // variable to hold the logging info
 CLOG_INFO* log=NULL;

 // open the log file
 log = clog_open("/var/tmp1.txt", clog_getlevel("TRACE"), NULL, 0);

 // change the log file and change the log level
 clog_setfile(log, "/var/tmp2.txt");

 // close the log and free the memory 'log' is pointing to
 clog_close(log);
 \endcode
*/
extern int                clog_setfile(CLOG_INFO* info, char * file);

/** \brief sets the logging to a new log level

 Sets the log level for logging to a file to the new log level.

 \param info The CLOG_INFO struct that was returned from the clog_open call.
 \param level The int that was returned from the clog_getlevel call.

 \return If successfull it returns 1, On failure it returns 0.

 \sa clog_open clog_setfile clog_getlevel

 Example:
 \code
 // variable to hold the logging info
 CLOG_INFO* log=NULL;

 // open the log file
 log = clog_open("/var/tmp1.txt", clog_getlevel("TRACE"), NULL, 0);

 // change the log file and change the log level
 clog_setfilelevel(log, clog_getlevel("FATAL"));

 // close the log and free the memory 'log' is pointing to
 clog_close(log);
 \endcode
*/
extern int                clog_setfilelevel(CLOG_INFO* info, unsigned int level);

/** \brief Sets the callback function

  Changes the callback of the logger after the clog_open function has already been called.
  You can pass NULL in the callback parameter to cause the logger to stop logging to the function.

  \param info The CLOG_INFO struct that was returned from the clog_open call.
  \param callback The callback function you would like called.

  \return If successfull it returns 1, On failure it returns 0.
*/
extern int                clog_setcallback(CLOG_INFO* info, C_CALLBACK callback);


/** \brief Sets the callback log level.

  Changes the callback log level of the logger after the clog_open function has already been called.

  \param info The CLOG_INFO struct that was returned from the clog_open call.
  \param level The int that was returned from the clog_getlevel call.

  \return If successfull it returns 1, On failure it returns 0.
*/
extern int                clog_setcallbacklevel(CLOG_INFO* info, int level);


/** \brief Opens the log file

  Opens the log file and returns a CLOG_INFO structure that has been allocated with malloc().

  \param file Filename of the log file you want to open. Directory must already exist, File will be appended to.
  \param f_level A message will only be saved to file if the message's log level is at least this level.
  \param callback A function that will also be called on a logging.
  \param c_level The callback function will only be called if the message's log level is at least this level.

  \sa clog_getlevel clog_close
*/
extern CLOG_INFO* clog_open(char * file, unsigned int f_level, C_CALLBACK callback, unsigned int c_level);

/** \brief logs to the file and function

  Logs to the file and function that were specified in the clog_open function. This function's last 2 parameters are the same as printf's 2 parameters.

  \param info The CLOG_INFO struct that was returned from the clog_open call.
  \param level The int that was returned from the clog_getlevel call.
  \param format see the manual on printf for the format parameter and the arguments

  \sa clog_open clog_getlevel printf

  Example:
  \code
  // define variables
  CLOG_INFO *info=NULL;
  int messagenumber = 0;

  // open the log
  info = clog_open("/tmp/mylog.txt", clog_getlevel("DEBUG"), NULL, 0);

  // will get logged
  clog(info, CFATAL, "fatal message %d", messagenumber++);

  // will get logged
  clog(info, CERROR, "error message %d", messagenumber++);

  // will get logged
  clog(info, CWARN , "warn  message %d", messagenumber++);

  // will NOT get logged
  clog(info, CDEBUG, "debug message %d", messagenumber++);

  // will NOT get logged
  clog(info, CTRACE, "trace message %d", messagenumber++);

  // close the log and free the memory
  clog_close(log);
  \endcode


*/
extern void               clog(CLOG_INFO* info, int level, const char *format, ...);

/** \brief Closes the file and free()s memory.

  Closes the open file pointed to in the CLOG_INFO* structure. Also free()s
  memory and that is in the CLOG_INFO* structure after setting it all to
  the defaults (NULLs and Zeros)

  \param info The CLOG_INFO struct that was returned from the clog_open call.

  \sa clog_open
*/
extern void               clog_close(CLOG_INFO* info);

/*@}*/

extern void clog_perror(CLOG_INFO* info, int level, const char *s);

// 'private' functions
void  clog_fileclose(CLOG_INFO * info);
FILE* clog_fileopen(char *file);

#endif // __CLOG_H__
